﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>

#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/tsc/tsc_Result.h>

#include <nn/tsc/tsc_Types.h>
#include <nn/tsc/tsc_ConfigContexts.h>

#include "tsc_TcpipStackConfigBase.h"
#include "tsc_Util.h"

namespace nn { namespace tsc {

// Pointer to the internal base config object defined in tsc_Library.h
extern class detail::TcpIpStackConfigBase* g_pObjStackConfig;
extern nn::os::Mutex g_objStackConfigMutex;

namespace
{
    Result VerifyIpv4Config(const Ipv4Config* pInIpv4Config)
    {
        NN_DETAIL_TSC_RETURN_ON_ERROR(detail::VerifyIpConfigMethod(pInIpv4Config->method));

        if(pInIpv4Config->method == Ipv4ConfigMethod_Static)
        {
            NN_DETAIL_TSC_RETURN_ON_ERROR(detail::VerifyInterfaceAddress(pInIpv4Config->interfaceAddress));
            NN_DETAIL_TSC_RETURN_ON_ERROR(detail::VerifySubnetMask(pInIpv4Config->subnetMask));
            NN_DETAIL_TSC_RETURN_ON_ERROR(detail::VerifyDefaultGateway(pInIpv4Config->defaultGateway));
        }

        if(pInIpv4Config->method == Ipv4ConfigMethod_Static || pInIpv4Config->method == Ipv4ConfigMethod_DhcpHybrid)
        {
            NN_DETAIL_TSC_RETURN_ON_ERROR(detail::VerifyPreferredDnsAddress(pInIpv4Config->preferredDns));
            NN_DETAIL_TSC_RETURN_ON_ERROR(detail::VerifyAlternativeDnsAddress(pInIpv4Config->alternativeDns));
        }

        return nn::ResultSuccess();
    }

    Result ApplyConfigInternal(Ipv4ConfigContext* pInConfig, int modeMask, int msecTimeout)
    {
        detail::TcpIpStackConfigBase* pConfigObj = nullptr;
        NN_DETAIL_TSC_GET_CONFIG_OR_RETURN_ON_UNINITIALIZED_STATE(pConfigObj);

        // Verify interface name
        char interfaceName[g_MaxInterfaceNameLength];
        NN_DETAIL_TSC_RETURN_ON_ERROR(pInConfig->GetInterfaceName(interfaceName, g_MaxInterfaceNameLength));
        NN_DETAIL_TSC_RETURN_ON_ERROR(detail::VerifyInterfaceName(interfaceName));

        // Verify MTU
        uint32_t mtuValue;
        NN_DETAIL_TSC_RETURN_ON_ERROR(pInConfig->GetMtu(&mtuValue));
        NN_DETAIL_TSC_RETURN_ON_ERROR(detail::VerifyMtu(mtuValue));

        // Verify IPv4 infomation
        Ipv4Config ipConfig;
        NN_DETAIL_TSC_RETURN_ON_ERROR(pInConfig->GetConfig(&ipConfig));
        NN_DETAIL_TSC_RETURN_ON_ERROR(VerifyIpv4Config(&ipConfig));

        // Set timeout if it's needed
        if (msecTimeout > 0)
        {
            pInConfig->SetMsecTimeout(msecTimeout);
        }

        return pConfigObj->ApplyIpConfig(pInConfig, modeMask);
    }
} // namespace

// ------------------------------------------------------------------------------------------------
// ConfigContext methods
// ------------------------------------------------------------------------------------------------
ConfigContext::ConfigContext(nn::os::EventClearMode eventClearMode)
    : m_Event(eventClearMode), m_TimerEvent(eventClearMode), m_IsUseSystemEvent(false)
{
    m_EventClerMode = eventClearMode;
    m_MsecTimeout   = 0;
    m_Mtu           = 0;
    m_ApplyResult   = nn::ResultSuccess();
    memset(m_InterfaceName, 0x00, g_MaxInterfaceNameLength);
}

ConfigContext::~ConfigContext()
{
    m_TimerEvent.Stop();
}

Result ConfigContext::SetInterfaceName(const char* pInInterfaceName) NN_NOEXCEPT
{
    NN_DETAIL_TSC_RETURN_ON_UNINITIALIZED_STATE();
    NN_SDK_REQUIRES_NOT_NULL(pInInterfaceName);

    uint32_t nameLength = strnlen(pInInterfaceName, g_MaxInterfaceNameLength);
    if (nameLength > g_MaxInterfaceNameLength - 1)
    {
        return tsc::ResultInterfaceNameInvalid();
    }

    strncpy(m_InterfaceName,
            pInInterfaceName,
            NN_DETAIL_TSC_GET_MIN(g_MaxInterfaceNameLength, nameLength));

    return nn::ResultSuccess();
}

Result ConfigContext::GetInterfaceName(char* pOutInterfaceName, int nameBufferLen) NN_NOEXCEPT
{
    NN_DETAIL_TSC_RETURN_ON_UNINITIALIZED_STATE();
    NN_SDK_REQUIRES_NOT_NULL(pOutInterfaceName);

    int nameLength = strnlen(m_InterfaceName, g_MaxInterfaceNameLength);
    if(nameLength > nameBufferLen - 1)
    {
        return tsc::ResultBufferTooShort();
    }

    strncpy(pOutInterfaceName,
            m_InterfaceName,
            NN_DETAIL_TSC_GET_MIN(g_MaxInterfaceNameLength, nameBufferLen));

    return nn::ResultSuccess();
}

Result ConfigContext::SetMtu(uint32_t mtuValue) NN_NOEXCEPT
{
    NN_DETAIL_TSC_RETURN_ON_UNINITIALIZED_STATE();

    NN_DETAIL_TSC_RETURN_ON_ERROR(detail::VerifyMtu(mtuValue));

    m_Mtu = mtuValue;
    return nn::ResultSuccess();
}

Result ConfigContext::GetMtu(uint32_t* pOutMtuValue) NN_NOEXCEPT
{
    NN_DETAIL_TSC_RETURN_ON_UNINITIALIZED_STATE();
    NN_SDK_REQUIRES_NOT_NULL(pOutMtuValue);

    *pOutMtuValue = m_Mtu;

    return nn::ResultSuccess();
}

void ConfigContext::SetApplyResult(nn::Result result) NN_NOEXCEPT
{
    m_ApplyResult = result;
}

nn::Result ConfigContext::GetApplyResult() NN_NOEXCEPT
{
    NN_DETAIL_TSC_RETURN_ON_UNINITIALIZED_STATE();
    return m_ApplyResult;
}

nn::os::TimerEvent* ConfigContext::GetTimerEventPointer() NN_NOEXCEPT
{
    return &m_TimerEvent;
}

nn::os::Event* ConfigContext::GetEventPointer() NN_NOEXCEPT
{
    return &m_Event;
}

nn::os::SystemEvent* ConfigContext::GetSystemEventPointer() NN_NOEXCEPT
{
    m_IsUseSystemEvent = true;
    return &m_SystemEvent;
}

nn::Result ConfigContext::SetMsecTimeout(uint32_t msecTimeout) NN_NOEXCEPT
{
    m_MsecTimeout = msecTimeout;
    return nn::ResultSuccess();
}

uint32_t ConfigContext::GetMsecTimeout() NN_NOEXCEPT
{
    return m_MsecTimeout;
}

nn::os::EventClearMode* ConfigContext::GetEventClearModePointer() NN_NOEXCEPT
{
    return &m_EventClerMode;
}

bool ConfigContext::IsUseSystemEvent() NN_NOEXCEPT
{
    return m_IsUseSystemEvent;
}


Result ConfigContext::NotifyInterfaceDown() NN_NOEXCEPT
{
    detail::TcpIpStackConfigBase* pConfigObj = nullptr;
    NN_DETAIL_TSC_GET_CONFIG_OR_RETURN_ON_UNINITIALIZED_STATE(pConfigObj);

    NN_DETAIL_TSC_RETURN_ON_ERROR(detail::VerifyInterfaceName(m_InterfaceName));

    return pConfigObj->NotifyInterfaceDown(m_InterfaceName);
}

// ------------------------------------------------------------------------------------------------
// Ipv4ConfigContext methods
// ------------------------------------------------------------------------------------------------
Ipv4ConfigContext::Ipv4ConfigContext(const char* pInInterfaceName, nn::os::EventClearMode eventClearMode)
    : ConfigContext(eventClearMode)
{
    ConfigContext::SetInterfaceName(pInInterfaceName);
    memset(&m_IpConfig, 0x00, sizeof(m_IpConfig));
}

Ipv4ConfigContext::~Ipv4ConfigContext()
{
    detail::TcpIpStackConfigBase* pConfigObj = g_pObjStackConfig;

    if( pConfigObj )
    {
        pConfigObj->CleanupConfigContext(this);
    }
}

Result Ipv4ConfigContext::SetConfig(const Ipv4Config* pInConfig) NN_NOEXCEPT
{
    NN_DETAIL_TSC_RETURN_ON_UNINITIALIZED_STATE();
    NN_SDK_REQUIRES_NOT_NULL(pInConfig);

    NN_DETAIL_TSC_RETURN_ON_ERROR(VerifyIpv4Config(pInConfig));

    m_IpConfig.method           = pInConfig->method;
    m_IpConfig.interfaceAddress = pInConfig->interfaceAddress;
    m_IpConfig.subnetMask       = pInConfig->subnetMask;
    m_IpConfig.defaultGateway   = pInConfig->defaultGateway;

    if(pInConfig->method == Ipv4ConfigMethod_Dhcp)
    {
        // Zero clear DNS addresses because underlying TCP/IP stack may use it if something is set
        m_IpConfig.preferredDns.storage   = 0;
        m_IpConfig.alternativeDns.storage = 0;
    }
    else
    {
        m_IpConfig.preferredDns     = pInConfig->preferredDns;
        m_IpConfig.alternativeDns   = pInConfig->alternativeDns;
    }

    return nn::ResultSuccess();
}

Result Ipv4ConfigContext::GetConfig(Ipv4Config* pOutConfig) NN_NOEXCEPT
{
    NN_DETAIL_TSC_RETURN_ON_UNINITIALIZED_STATE();
    NN_SDK_REQUIRES_NOT_NULL(pOutConfig);

     pOutConfig->method           = m_IpConfig.method;
     pOutConfig->interfaceAddress = m_IpConfig.interfaceAddress;
     pOutConfig->subnetMask       = m_IpConfig.subnetMask;
     pOutConfig->defaultGateway   = m_IpConfig.defaultGateway;
     pOutConfig->preferredDns     = m_IpConfig.preferredDns;
     pOutConfig->alternativeDns   = m_IpConfig.alternativeDns;

    return nn::ResultSuccess();
}

Result Ipv4ConfigContext::ApplyConfig(int modeMask) NN_NOEXCEPT
{
    return ApplyConfigInternal(this, modeMask, 0);
}

Result Ipv4ConfigContext::ApplyConfig(int modeMask, int msecTimeout) NN_NOEXCEPT
{
    return ApplyConfigInternal(this, modeMask, msecTimeout);
}

Result Ipv4ConfigContext::CancelApplyConfig() NN_NOEXCEPT
{
    detail::TcpIpStackConfigBase* pConfigObj = nullptr;
    NN_DETAIL_TSC_GET_CONFIG_OR_RETURN_ON_UNINITIALIZED_STATE(pConfigObj);

    return pConfigObj->CancelApplyIpConfig();
}

Result Ipv4ConfigContext::CheckDuplicateIp(int msecTimeout) NN_NOEXCEPT
{
    detail::TcpIpStackConfigBase* pConfigObj = nullptr;
    NN_DETAIL_TSC_GET_CONFIG_OR_RETURN_ON_UNINITIALIZED_STATE(pConfigObj);

    return pConfigObj->CheckDuplicateIp(msecTimeout);
}

}}
