﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_SdkLog.h>
#include <nn/nn_Abort.h>

#include <algorithm>

#include <nn/spl/smc/spl_SecureMonitorApi.h>

#include "spl_SecureMonitorCall.h"


namespace nn { namespace spl { namespace smc {

    namespace
    {
        const Bit32 SetConfigFunctionIndex                         = 0xC3000401;
        const Bit32 GetConfigFunctionIndex                         = 0xC3000002;
        const Bit32 GetResultFunctionIndex                         = 0xC3000003;
        const Bit32 GetResultBytesFunctionIndex                    = 0xC3000404;
        const Bit32 ModularExponentiateFunctionIndex               = 0xC3000E05;
        const Bit32 GenerateRandomBytesFunctionIndex               = 0xC3000006;
        const Bit32 GenerateAesKekFunctionIndex                    = 0xC3000007;
        const Bit32 LoadAesKeyFunctionIndex                        = 0xC3000008;
        const Bit32 ComputeAesFunctionIndex                        = 0xC3000009;
        const Bit32 GenerateSpecificAesKeyFunctionIndex            = 0xC300000A;
        const Bit32 ComputeCmacFunctionIndex                       = 0xC300040B;
        const Bit32 ReencryptDeviceUniqueDataFunctionIndex         = 0xC300D60C;
        const Bit32 DecryptDeviceUniqueDataFunctionIndex           = 0xC300100D;
        const Bit32 ModularExponentiateWithStorageKeyFunctionIndex = 0xC300060F;
        const Bit32 PrepareEsDeviceUniqueKeyFunctionIndex          = 0xC3000610;
        const Bit32 LoadPreparedAesKeyFunctionIndex                = 0xC3000011;
        const Bit32 PrepareCommonEsTitleKeyFunctionIndex           = 0xC3000012;
    }

    SmcResult ModularExponentiate(
        OperationKey*   pOutOperationKey,
        const void*     pBase,
        const void*     pExponent,
        size_t          exponentSize,
        const void*     pModulus ) NN_NOEXCEPT
    {
        Registers   r;
        r.r[0] = ModularExponentiateFunctionIndex;
        r.r[1] = reinterpret_cast<Bit64>(pBase);
        r.r[2] = reinterpret_cast<Bit64>(pExponent);
        r.r[3] = reinterpret_cast<Bit64>(pModulus);
        r.r[4] = exponentSize;

        CallSecureMonitor(&r);

        pOutOperationKey->value = r.r[1];
        return static_cast<SmcResult>(r.r[0]);
    }

    SmcResult GenerateAesKek(
        AccessKey*  pOutAccessKey,
        const Bit64 pSource[],
        int         generation,
        Bit32       option  ) NN_NOEXCEPT
    {
        Registers   r;
        r.r[0] = GenerateAesKekFunctionIndex;
        r.r[1] = pSource[0];
        r.r[2] = pSource[1];
        r.r[3] = generation;
        r.r[4] = option;

        CallSecureMonitor(&r);

        pOutAccessKey->value[0] = r.r[1];
        pOutAccessKey->value[1] = r.r[2];
        return static_cast<SmcResult>(r.r[0]);
    }

    SmcResult LoadAesKey(
        int slotIndex,
        const AccessKey& accessKey,
        const Bit64 pSource[] ) NN_NOEXCEPT
    {
        Registers   r;
        r.r[0] = LoadAesKeyFunctionIndex;
        r.r[1] = slotIndex;
        r.r[2] = accessKey.value[0];
        r.r[3] = accessKey.value[1];
        r.r[4] = pSource[0];
        r.r[5] = pSource[1];

        CallSecureMonitor(&r);

        return static_cast<SmcResult>(r.r[0]);
    }

    SmcResult ComputeAes(
        OperationKey*   pOutOperationKey,
        uint32_t        outputAddress,
        Bit32           mode,
        const Bit64     pIvic[],
        uint32_t        inputAddress,
        size_t          inoutSize ) NN_NOEXCEPT
    {
        Registers   r;
        r.r[0] = ComputeAesFunctionIndex;
        r.r[1] = mode;
        r.r[2] = pIvic[0];
        r.r[3] = pIvic[1];
        r.r[4] = inputAddress;
        r.r[5] = outputAddress;
        r.r[6] = inoutSize;

        CallSecureMonitor(&r);

        pOutOperationKey->value = r.r[1];
        return static_cast<SmcResult>(r.r[0]);
    }

    SmcResult ComputeCmac(
        Bit64           pOutMac[],
        int             slotIndex,
        const void*     pInBuffer,
        size_t          inBufferSize ) NN_NOEXCEPT
    {
        Registers   r;
        r.r[0] = ComputeCmacFunctionIndex;
        r.r[1] = slotIndex;
        r.r[2] = reinterpret_cast<Bit64>(pInBuffer);
        r.r[3] = inBufferSize;

        CallSecureMonitor(&r);

        pOutMac[0] = r.r[1];
        pOutMac[1] = r.r[2];
        return static_cast<SmcResult>(r.r[0]);
    }

    SmcResult GenerateSpecificAesKey(
        Bit64       pResult[],
        const Bit64 pSource[],
        int         generation,
        Bit32       purpose  ) NN_NOEXCEPT
    {
        Registers   r;
        r.r[0] = GenerateSpecificAesKeyFunctionIndex;
        r.r[1] = pSource[0];
        r.r[2] = pSource[1];
        r.r[3] = generation;
        r.r[4] = purpose;

        CallSecureMonitor(&r);

        pResult[0] = r.r[1];
        pResult[1] = r.r[2];
        return static_cast<SmcResult>(r.r[0]);
    }

    SmcResult  GetConfig(
        Bit64       pOutBuffer[],
        int         bufferCount,
        ConfigItem  key ) NN_NOEXCEPT
    {
        Registers   r;
        r.r[0] = GetConfigFunctionIndex;
        r.r[1] = key;

        CallSecureMonitor(&r);

        int copyCount = std::min(4, bufferCount);
        for( int i = 0; i < copyCount; ++i )
        {
            pOutBuffer[i] = r.r[1 + i];
        }

        return static_cast<SmcResult>(r.r[0]);
    }

    SmcResult  SetConfig(
        OperationKey*   pOutOperationKey,
        ConfigItem      key,
        const Bit64     pValue[],
        int             valueCount,
        const void*     pSign ) NN_NOEXCEPT
    {
        Registers   r;
        r.r[0] = SetConfigFunctionIndex;
        r.r[1] = key;
        r.r[2] = reinterpret_cast<Bit64>(pSign);

        int copyCount = std::min(4, valueCount);
        for( int i = 0; i < copyCount; ++i )
        {
            r.r[3 + i] = pValue[i];
        }

        CallSecureMonitor(&r);

        pOutOperationKey->value = r.r[1];
        return static_cast<SmcResult>(r.r[0]);
    }

    SmcResult  GenerateRandomBytes(
        void*   pOutBuffer,
        size_t  byteSize ) NN_NOEXCEPT
    {
        Registers   r;
        r.r[0] = GenerateRandomBytesFunctionIndex;
        r.r[1] = byteSize;

        CallSecureMonitor(&r);

        if( (r.r[0] == 0) && (byteSize <= sizeof(r) - sizeof(r.r[0])) )
        {
            std::memcpy(pOutBuffer, &r.r[1], byteSize);
        }

        return static_cast<SmcResult>(r.r[0]);
    }

    SmcResult  ModularExponentiateWithStorageKey(
        OperationKey*   pOutOperationKey,
        const void*     pCipher,
        const void*     pModulus,
        Bit32           option) NN_NOEXCEPT
    {
        Registers   r;
        r.r[0] = ModularExponentiateWithStorageKeyFunctionIndex;
        r.r[1] = reinterpret_cast<Bit64>(pCipher);
        r.r[2] = reinterpret_cast<Bit64>(pModulus);
        r.r[3] = option;

        CallSecureMonitor(&r);

        pOutOperationKey->value = r.r[1];
        return static_cast<SmcResult>(r.r[0]);
    }

    SmcResult  DecryptDeviceUniqueData(
        void*            pData,
        size_t           dataSize,
        const AccessKey& accessKey,
        const Bit64      pSource[],
        Bit32            option) NN_NOEXCEPT
    {
        Registers   r;
        r.r[0] = DecryptDeviceUniqueDataFunctionIndex;
        r.r[1] = accessKey.value[0];
        r.r[2] = accessKey.value[1];
        r.r[3] = option;
        r.r[4] = reinterpret_cast<Bit64>(pData);
        r.r[5] = dataSize;
        r.r[6] = pSource[0];
        r.r[7] = pSource[1];

        CallSecureMonitor(&r);

        return static_cast<SmcResult>(r.r[0]);
    }

    SmcResult  PrepareEsDeviceUniqueKey(
        OperationKey*   pOutOperationKey,
        const void*     pCipher,
        const void*     pModulus,
        const void*     pLabelDigest,
        size_t          labelDigestSize,
        Bit32           option) NN_NOEXCEPT
    {
        Registers   r;
        r.r[0] = PrepareEsDeviceUniqueKeyFunctionIndex;
        r.r[1] = reinterpret_cast<Bit64>(pCipher);
        r.r[2] = reinterpret_cast<Bit64>(pModulus);
        r.r[7] = option;

        std::memcpy(&r.r[3], pLabelDigest, std::min(static_cast<size_t>(32), labelDigestSize));

        CallSecureMonitor(&r);

        pOutOperationKey->value = r.r[1];
        return static_cast<SmcResult>(r.r[0]);
    }

    SmcResult  PrepareCommonEsTitleKey(
        AccessKey*  pOutAccessKey,
        const Bit64 pSource[],
        int         generation) NN_NOEXCEPT
    {
        Registers   r;
        r.r[0] = PrepareCommonEsTitleKeyFunctionIndex;
        r.r[1] = pSource[0];
        r.r[2] = pSource[1];
        r.r[3] = generation;

        CallSecureMonitor(&r);

        pOutAccessKey->value[0] = r.r[1];
        pOutAccessKey->value[1] = r.r[2];
        return static_cast<SmcResult>(r.r[0]);
    }

    SmcResult  LoadPreparedAesKey(
        int              slotIndex,
        const AccessKey& accessKey) NN_NOEXCEPT
    {
        Registers   r;
        r.r[0] = LoadPreparedAesKeyFunctionIndex;
        r.r[1] = slotIndex;
        r.r[2] = accessKey.value[0];
        r.r[3] = accessKey.value[1];

        CallSecureMonitor(&r);

        return static_cast<SmcResult>(r.r[0]);
    }

    SmcResult  ReencryptDeviceUniqueData(
        void*            pData,
        size_t           dataSize,
        const AccessKey& accessKeyForDecryption,
        const Bit64      pSourceForDecryption[],
        const AccessKey& accessKeyForEncryption,
        const Bit64      pSourceForEncryption[],
        Bit32            option) NN_NOEXCEPT
    {
        Registers   r;
        r.r[0] = ReencryptDeviceUniqueDataFunctionIndex;
        r.r[1] = reinterpret_cast<Bit64>(&accessKeyForDecryption);
        r.r[2] = reinterpret_cast<Bit64>(&accessKeyForEncryption);
        r.r[3] = option;
        r.r[4] = reinterpret_cast<Bit64>(pData);
        r.r[5] = dataSize;
        r.r[6] = reinterpret_cast<Bit64>(pSourceForDecryption);
        r.r[7] = reinterpret_cast<Bit64>(pSourceForEncryption);

        CallSecureMonitor(&r);

        return static_cast<SmcResult>(r.r[0]);
    }



    SmcResult  GetResult(SmcResult* pOut, OperationKey operationKey) NN_NOEXCEPT
    {
        Registers   r;
        r.r[0] = GetResultFunctionIndex;
        r.r[1] = operationKey.value;

        CallSecureMonitor(&r);

        *pOut = static_cast<SmcResult>(r.r[1]);
        return static_cast<SmcResult>(r.r[0]);
    }

    SmcResult  GetResultData(
        SmcResult*      pOut,
        void*           pOutBuffer,
        size_t          bufferSize,
        OperationKey    operationKey) NN_NOEXCEPT
    {
        Registers   r;
        r.r[0] = GetResultBytesFunctionIndex;
        r.r[1] = operationKey.value;
        r.r[2] = reinterpret_cast<Bit64>(pOutBuffer);
        r.r[3] = bufferSize;

        CallSecureMonitor(&r);

        *pOut = static_cast<SmcResult>(r.r[1]);
        return static_cast<SmcResult>(r.r[0]);
    }


}}}  // namespace nn::spl::smc

