﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/socket/socket_TypesPrivate.h>

namespace nn { namespace socket {

extern const unsigned g_ResolverOptionVersion;


/**
 * @brief This enum defines constants that are used with @ref
 * ValidateOptions to validate that certain keys are allowed within
 * a given context.
 */
enum OptionContext
{

    OptionContext_Get            =  1,

    OptionContext_Set            =  2,

    OptionContext_Request        =  4,

    OptionContext_PrivateGet     =  8,

    OptionContext_PrivateSet     = 16,

    OptionContext_PrivateRequest = 32,

    OptionContext_Maximum
};

/**
 * @brief This function checks that a tuple of specific key, sfdl
 * client request version, and the specified mask form a valid
 * request.
 *
 * @detail
 * This function is used for the ResolverOption "get" functionality.
 *
 * @param[in] key This is the @ref ResolverOptionKey key value that
 * you want to verify is valid on the current version and with the
 * provided mask.
 *
 * @param[in] version This parameter is the sfdl request version.
 *
 * @param[in] mask This parameter is a mask of valid contexts for the
 * given option.
 *
 * @return true if the combination is valid, false otherwise.
 */
bool ValidateKeyForVersionAndMask(ResolverOptionKey key,
                                  uint32_t version,
                                  const OptionContext mask);
/**
 * @brief This function validates that the given option is valid in
 * function and form for the set of contexts in the mask.
 *
 * @detail
 * This function checks that the option is well-formed by checking
 * that the type and size parameter line up. It also checks
 * that the option key value lies between the minimum and maximum
 * point on a number line every context value in the bitset mask.
 *
 * This function is used for the ResolverOption "set" functionality.
 *
 * @param[in] version This parameter is the sfdl request version.
 *
 * @param[in] mask This parameter is a mask of valid contexts for the
 * given option.
 *
 * @param[in] option This parameter is the @ref ResolverOption to
 * validate.
 */
bool ValidateOptionForVersionAndMask(const ResolverOption& option,
                                     uint32_t version,
                                     const OptionContext mask);

/**
 * @brief This function validates validates that the array passed
 * in from the service framework client contains @ref ResolverOption
 * values that are allowed in the current context.
 *
 * @detail
 * This function verifies the version number and array semantics such
 * as byte count prior to calling @ref ValidateOption for every member
 * in the array.
 *
 * This function is used for the ResolverOption resolver "request"
 * functionality.
 *
 * param[in] pOptions This parameter is the resolver options array.
 *
 * param[in] size This parameter is the resolver option array size in
 * bytes. Here, we are checking that a non-malicious buffer was passed
 * from the client shim to service framework.
 *
 * param[in] count This parameter is the number of elements present
 * in the ResolverOptions array.
 *
 * @param[in] version This parameter is the sfdl request version.
 *
 * param[in] mask This parameter is a bit set mask of @ref
 * OptionContext values that indicate the context of the caller.
 *
 * @return
 * This function returns true if all options are valid.
 */
bool ValidateOptionsArrayForVersionAndMask(const ResolverOption* pOptions,
                                           size_t count,
                                           const uint32_t version,
                                           OptionContext mask);

/**
 * @brief Find a boolean ResolverOption value in a ResolverOption array.
 *
 * @details
 * This and related functions simplify getting a request resolver option
 * out of the array based on type rather than name.
 *
 * @param[out] out This field is the boolean out value.
 *
 * @param[in] version This parameter is the ResolverOption feature
 * version used by the calling client process.
 *
 * @param[in] key This parameter is the key value corresponding to the
 * value you want to find in the array
 *
 * @param[in] count This parameter is the number of ResolverOptions
 * present in the array.
 */
int GetRequestOptionValue(bool& out,
                          const uint32_t version,
                          const ResolverOptionKey key,
                          const ResolverOption* pOptions,
                          size_t count);

/**
 * @brief Find an int ResolverOption value in a ResolverOption array.
 *
 * @details
 * This and related functions simplify getting a request resolver option
 * out of the array based on type rather than name.
 *
 * @param[out] out This field is the integer out value.
 *
 * @param[in] version This parameter is the ResolverOption feature
 * version used by the calling client process.
 *
 * @param[in] key This parameter is the key value corresponding to the
 * value you want to find in the array
 *
 * @param[in] count This parameter is the number of ResolverOptions
 * present in the array.
 */
int GetRequestOptionValue(int& out,
                          const uint32_t version,
                          const ResolverOptionKey key,
                          const ResolverOption* pOptions,
                          size_t count);

/**
 * @brief Find an uint32_t ResolverOption value in a ResolverOption array.
 *
 * @details
 * This and related functions simplify getting a request resolver option
 * out of the array based on type rather than name.
 *
 * @param[out] out This field is the unsigned 32-bit integer out value.
 *
 * @param[in] version This parameter is the ResolverOption feature
 * version used by the calling client process.
 *
 * @param[in] key This parameter is the key value corresponding to the
 * value you want to find in the array
 *
 * @param[in] count This parameter is the number of ResolverOptions
 * present in the array.
 */
int GetRequestOptionValue(uint32_t& out,
                          const uint32_t version,
                          const ResolverOptionKey key,
                          const ResolverOption* pOptions,
                          size_t count);

/**
 * @brief Find an uint64_t ResolverOption value in a ResolverOption array.
 *
 * @details
 * This and related functions simplify getting a request resolver option
 * out of the array based on type rather than name.
 *
 * @param[out] out This field is the unsigned 64-bit integer out value.
 *
 * @param[in] version This parameter is the ResolverOption feature
 * version used by the calling client process.
 *
 * @param[in] key This parameter is the key value corresponding to the
 * value you want to find in the array
 *
 * @param[in] count This parameter is the number of ResolverOptions
 * present in the array.
 */
int GetRequestOptionValue(uint64_t& out,
                          const uint32_t version,
                          const ResolverOptionKey key,
                          const ResolverOption* pOptions,
                          size_t count);

/**
 * @brief Find a double ResolverOption value in a ResolverOption array.
 *
 * @details
 * This and related functions simplify getting a request resolver option
 * out of the array based on type rather than name.
 *
 * @param[out] out This field is the double-precision floating point out value.
 *
 * @param[in] version This parameter is the ResolverOption feature
 * version used by the calling client process.
 *
 * @param[in] key This parameter is the key value corresponding to the
 * value you want to find in the array
 *
 * @param[in] count This parameter is the number of ResolverOptions
 * present in the array.
 */
int GetRequestOptionValue(double& out,
                          const uint32_t version,
                          const ResolverOptionKey key,
                          const ResolverOption* pOptions,
                          size_t count);

/**
 * @brief Find a pointer ResolverOption value in a ResolverOption array.
 *
 * @details
 * This and related functions simplify getting a request resolver option
 * out of the array based on type rather than name.
 *
 * @param[out] out This field is the pointer out value.
 *
 * @param[in] version This parameter is the ResolverOption feature
 * version used by the calling client process.
 *
 * @param[in] key This parameter is the key value corresponding to the
 * value you want to find in the array
 *
 * @param[in] count This parameter is the number of ResolverOptions
 * present in the array.
 */
int GetRequestOptionValue(const char *& out,
                          const uint32_t version,
                          const ResolverOptionKey key,
                          const ResolverOption* pOptions,
                          size_t count);
}}; // nn::socket
