﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <mutex>
#include "resolver_HipcServer.h"
#include "resolver_ClientServerShared.h"
#include "resolver_AutoContext.h"

extern "C"
{
#include <stdint.h>
};

namespace nn { namespace socket { namespace resolver {

/**
 * @brief This class is the class implements the table for the
 * Resolver Cancellation feature.
 *
 * @detail
 * The table class runs as a singleton. It contains an array of
 * @ref Slot "slots" below and each slot contains data necessary to
 * establish and maintain the feature.
 *
 * The data in the table is tied to a call to @ref GetHostByName,
 * @ref GetHostByAddr, @ref GetAddrInfo, or @ref GetNameInfo.
 *
 * The class is by and linked with the bionic resolver server library
 */
class ResolverCancelTable
{
protected:
    /**
     * @brief ctor
     */
    ResolverCancelTable();

    /**
     * @brief dtor
     */
    ~ResolverCancelTable();

    /**
     * @brief copy ctor; asserts
     */
    ResolverCancelTable(const ResolverCancelTable& o);

    /**
     * @brief operator==; asserts
     */
    bool operator==(const ResolverCancelTable& o);

private:
    /**
     * @brief maximum number of table entries
     */
    static const int TABLE_MAX = nn::socket::resolver::Server::MAX_THREADS - 1;

    /**
     * @brief The states of a slot
     */
    enum SlotState
    {
        /**
         * @brief the initial state of an empty slot
         */
        SlotState_Pending   = 0,

        /**
         * @brief the state of a slot once the resolver server is able to use
         */
        SlotState_Acquired  = 1,

        /**
         * @brief the state of a slot once the resolver starts to use the handle
         */
        SlotState_Running   = 2,

        /**
         * @brief indicates the client requested the resolver operation
         * associated with the handle discontinue further operation
         */
        SlotState_Cancelled = 3,

        /**
         * @brief indicates the server requested the resolver operation
         * associated with the handle discontinue further operation due to a timeout
         */
        SlotState_Expired = 4,
    };

    const char* GetSlotStateString(const SlotState& slotStateIn, bool includeEnumName) const;

    /**
     * @brief Container
     */
    struct Slot
    {
        /** @brief the state of the slot; all slots */
        SlotState m_SlotState;

        /** @brief the process ID */
        Bit64 m_ProcessID;

        /** @brief the cancel handle ID */
        int m_HandleID;

        /** @brief expiration time in seconds */
        int64_t m_ExpirationTimeInSeconds;
    };

    /**
     * @brief max allowed seconds until the call is expired
     */
    static const unsigned MaxAllowedSeconds = 8;

    /**
     * @brief table access lock
     */
    std::mutex m_AccessLock;

    /**
     * @brief Get or Create the static shared instance
     * @return the instance of the ResolverCancelTable
     */
    static ResolverCancelTable& GetCreateSharedInstance();

    /**
     * @brief table of cancel handle slots
     */
    Slot m_HandleSlots[TABLE_MAX];

    /**
     * @brief sort the slots by sequence number
     */
    void DumpTableUnlocked(const char* line) const;

    /**
     * @brief check all the slots in the table and expire any that are old
     */
    void EnforceExpirationTimesUnlocked();


    /**
     * @brief combined function to handle setting expired or cancelled states
     */
    static int SetSlotCancelledOrExpired(const int handleID, const Bit64 processID,
                                         SlotState slotStateIn,
                                         OperationState& operationStateOut);

public:
    /**
     * @brief acquire a pending slot for the given handle and process ID
     * @param handleID the cancel handle identifier
     * @note in the zero case the new handleID is returned through the handleID parameter
     * @param processID the identifier of the calling process
     * @param operationStateOut fine grained data pertaining to the state of the call
     * @return 0 on success, -1 and set errno otherwise:
     * - EINVAL: a problem was encountered generating the handleID
     * - EALREADY: there already exists a slot with the same handle & process ID
     * and that slot is in a non-pending state
     * @post on success a slot will be associated with handleID and processID and
     * the slot state will move from PENDING to ACQUIRED, otherwise nothing
     */
    static int AcquireSlot(int& handleID, const Bit64 processID,
                           OperationState& operationStateOut);

    /**
     * @brief set a slot to the running state
     * @note this means that the resolver call is actively processing
     * @param handleID the cancel handle identifier
     * @param processID the id of the calling process
     * @param operationStateOut fine grained data pertaining to the state of the call
     * @return 0 on success, -1 and errno set otherwise:
     * - EINVAL: returned when the slot is in a PENDING state or if you pass zero
     * - EALREADY: a slot already exists for the handleID and processID pair
     * and that slot is also in the not in the PENDING state.
     * @post on success the slot associated with handleID and processID will move
     * from ACQUIRED to RUNNING, otherwise nothing
     */
    static int SetSlotRunning(const int handleID, const Bit64 processID,
                              OperationState& operationStateOut);

    /**
     * @brief set a slot to the cancelled state
     * @param handleID the cancel handle identifier
     * @param processID the id of the calling process
     * @param operationStateOut fine grained data pertaining to the state of the call
     * @return 0 on success, -1 and errno set otherwise:
     * - EINVAL: returned when the slot is in a PENDING state or if you pass zero
     * - EALREADY: a slot already exists for the handleID and processID pair
     * and that slot is also in the not in the PENDING state.
     * @post on success the slot associated with handleID and processID will move
     * from ACQUIRED to RUNNING, otherwise nothing
     */
    static int SetSlotCancelled(const int handleID, const Bit64 processID,
                                OperationState& operationStateOut);

    /**
     * @brief check if the slot corresponding with handleID and processID is cancelled or expired
     * @note the bionic socket layer makes use of this function as it enters and exits the method
     * @note return true if cancelled or expired, if expired then operationStateOut
     *       will be OperationState_IsSlotCancelledOrExpiredExpired
     * @param handleID the cancel handle identifier
     * @param processID the id of the calling process
     * @param operationStateOut fine grained data pertaining to the state of the call
     * @return true if cancelled, false otherwise, errno set with error
     * - EINVAL: returned when the slot is either 0 or not found in the table
     * @post no effect on the table
     */
    static bool IsSlotCancelledOrExpired(const int handleID,
                                         const Bit64 processID,
                                         OperationState& operationStateOut);

    /**
     * @brief set a slot to the expired state
     * @param handleID the cancel handle identifier
     * @param processID the id of the calling process
     * @param operationStateOut fine grained data pertaining to the state of the call
     * @return 0 on success, -1 and errno set otherwise:
     * - EINVAL: returned when the slot is in a PENDING state or if you pass zero
     * - EALREADY: a slot already exists for the handleID and processID pair
     * and that slot is also in the not in the PENDING state.
     * @post on success the slot associated with handleID and processID will move
     * from ACQUIRED to RUNNING, otherwise nothing
     */
    static int SetSlotExpired(const int handleID, const Bit64 processID,
                              OperationState& operationStateOut);

    /**
     * @note no IsSlotExpired because we can avoid the lock and use the ResolverAutoContext instead
     */

    /**
     * @brief reset a slot to the PENDING state
     * @param handleID the cancel handle identifier
     * @param processID the id of the calling process
     * @param operationStateOut fine grained data pertaining to the state of the call
     * @return 0 on succeess, -1 and errno set otherwise:
     * - EINVAL: returned when the slot is either 0 or not found in the table
     * if state is running then state advances to PENDING otherwise state returns to pending
     * - EALREADY: the slot was already in the PENDING state
     */
    static int SetSlotPending(const int handleID, const Bit64 processID,
                         OperationState& operationStateOut);

    /**
     *@brief reset all non-pending slots for given id
     * @param processID the id of the calling process
     * @param operationStateOut fine grained data pertaining to the state of the call
     * return 0 on success, error otherwise
     */
    static void CancelAllSlotsForProcess(const Bit64 processID,
                                         OperationState& operationStateOut);
};

}}}; //nn::socket::resolver::server
