﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "resolver_AutoContext.h"

extern "C"
{
#include <siglo/BionicPrivateApi.h>
}

//#define LOG_LEVEL LOG_LEVEL_MAX
#define LOG_MODULE_NAME "AutoContext" // NOLINT(preprocessor/const)
#include <nn/socket/resolver/private/resolver_DebugLog.h>

namespace nn { namespace socket { namespace resolver {

// ResolverAutoContext

ResolverAutoContext::ResolverAutoContext() :
    m_ProcessID(0),
    m_HandleID(0),
    m_DidSucceed(false),
    m_CurrentOperationState(OperationState_Initial),
    m_Options(0),
    m_OptionsCount(0)
{
}

bool ResolverAutoContext::Initialize(const Bit64 & processID,
                                     int& cancelHandleInOut,
                                     const struct ResolverOption* options,
                                     uint32_t optionsCount)
{
    if ( true == m_DidSucceed ) // don't initialize twice
    {
        return m_DidSucceed;
    };

    m_ProcessID = processID;
    m_HandleID = cancelHandleInOut;
    m_DidSucceed = false;
    m_CurrentOperationState = OperationState_Initial;
    m_Options = options;
    m_OptionsCount = optionsCount;

    ResolverSetAutoContextIntoTLS(this);

    if (0 != ResolverCancelTable::AcquireSlot(m_HandleID,
                                              m_ProcessID,
                                              m_CurrentOperationState))
    {
        LogDebug("Unable to AcquireSlot; state: %s, error: (%d)\n",
                 GetOperationStateString(m_CurrentOperationState, false),
                 __LINE__);
        goto bail;
    }
    else if (0 != ResolverCancelTable::SetSlotRunning(m_HandleID,
                                                      m_ProcessID,
                                                      m_CurrentOperationState))
    {
        LogDebug("Unable to SetSlotRunning; state: %s, error: %d\n",
                 GetOperationStateString(m_CurrentOperationState, false),
                 __LINE__);
        goto bail;
    };

    LogDebug("SetSlotRunning success, status: %s\n",
             GetOperationStateString(m_CurrentOperationState,
                                     false));

    m_DidSucceed = true;
    cancelHandleInOut = m_HandleID;

bail:
    LogDebug("operation state for HandleID: %08x, ProcessID %08x: %s (%d)\n",
             m_HandleID, m_ProcessID,
             GetOperationStateString(m_CurrentOperationState, false),
             m_CurrentOperationState);
    return m_DidSucceed;
};

ResolverAutoContext::~ResolverAutoContext()
{
    // once the slot is in the pending state it can be reused by another request
    ResolverCancelTable::SetSlotPending(m_HandleID, m_ProcessID, m_CurrentOperationState);
    ResolverSetAutoContextIntoTLS(NULL);
    m_HandleID = 0;
    m_ProcessID = 0;
    m_DidSucceed = false;
};

const Bit64 & ResolverAutoContext::GetCallingProcessID() const
{
    return m_ProcessID;
};

const int & ResolverAutoContext::GetCancelHandleID() const
{
    return m_HandleID;
};

const bool & ResolverAutoContext::DidInitializationSucceed() const
{
    return m_DidSucceed;
}

ResolverAutoContext& ResolverAutoContext::CurrentContext()
{
    static ResolverAutoContext dontcrash;
    ResolverAutoContext* pointer = reinterpret_cast<ResolverAutoContext*>(ResolverGetAutoContextInTLS());

    if (NULL == pointer)
    {
        NN_SDK_ASSERT(false);
        LogDebug("Unable to ResolverGetAutoContextInTLS; error %d\n", __LINE__);
        pointer = &dontcrash;
    };

    return *static_cast<ResolverAutoContext*>(pointer);
};

bool ResolverAutoContext::IsSlotCancelledOrExpired(bool & isExpired)
{
    bool rc =
        nn::socket::resolver::
        ResolverCancelTable::IsSlotCancelledOrExpired(m_HandleID,
                                                      m_ProcessID,
                                                      m_CurrentOperationState);

    LogDebug("operation state for HandleID: %08x, ProcessID %08x: %s (%d)\n",
             m_HandleID, m_ProcessID,
             GetOperationStateString(m_CurrentOperationState, false),
             m_CurrentOperationState);

    isExpired = (OperationState_IsSlotCancelledOrExpiredExpired == m_CurrentOperationState );
    return rc;
};

const ResolverOption* ResolverAutoContext::GetOptionsArray() const
{
    return m_Options;
};

size_t ResolverAutoContext::GetOptionsCount() const
{
    return m_OptionsCount;
};

}}}; // nn::socket::resolver
