﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "serializer.h"
#include "serializer_Specializations.h"

/*
// IPv4 AF_INET sockets:

struct sockaddr_in {
short            sin_family;
unsigned short   sin_port;
struct in_addr   sin_addr;
char             sin_zero[8];
};

struct in_addr {
unsigned long s_addr;          // load with inet_pton()
};
*/

//#define LOG_LEVEL LOG_LEVEL_MAJOR
#define LOG_MODULE_NAME "sockaddr_in serializer" // NOLINT(preprocessor/const)
#include <nn/socket/resolver/private/resolver_DebugLog.h>

namespace nn { namespace socket { namespace resolver { namespace serializer {

template <>
size_t DNSSerializer::SizeOf(const nn::socket::SockAddrIn & sockaddrIn4)
{
    size_t rc = 0;
    rc += sizeof(uint16_t); // sin_family
    rc += sizeof(uint16_t); // sin_port
    rc += SizeOf(sockaddrIn4.sin_addr); // sin_addr
    rc += sizeof(sockaddrIn4.sin_zero); // sin_zero
    return rc;
}

template <>
size_t DNSSerializer::SizeOf(const struct sockaddr_in & sockaddrIn4)
{
    size_t rc = 0;
    rc += sizeof(uint16_t); // sin_family
    rc += sizeof(uint16_t); // sin_port
    rc += SizeOf(sockaddrIn4.sin_addr); // sin_addr
    rc += sizeof(sockaddrIn4.sin_zero); // sin_zero
    return rc;
}

// serialize a nn::socket::SockAddrIn (ipv4) to the buffer
template <>
ssize_t DNSSerializer::ToBuffer(uint8_t * const pBufferOut,
                                size_t     bufferSize,
                                const nn::socket::SockAddrIn & sockaddrIn4)
{
    ssize_t rc = -1;
    uint8_t* pBufferCurrent = pBufferOut;

    uint16_t shortValue;  // use for sin_family, sin_port
    uint32_t longValue;   // use for in_addr.s_addr

    if ( -1 == CheckToBufferArguments(pBufferCurrent,
                                      bufferSize,
                                      sizeof(sockaddrIn4),
                                      __LINE__))
    {
        rc = -1;
        goto bail;
    }

    shortValue = static_cast<uint16_t>(sockaddrIn4.sin_family);
    if (-1 == ( rc = ToBuffer(pBufferCurrent,
                              bufferSize - (pBufferCurrent - pBufferOut),
                              shortValue)))
    {
        LogDebug("error: sin_family (%d)", __LINE__);
        goto bail;
    };
    pBufferCurrent += rc;

    shortValue = sockaddrIn4.sin_port;
    if (-1 == ( rc = ToBuffer(pBufferCurrent,
                              bufferSize - (pBufferCurrent - pBufferOut),
                              shortValue)))
    {
        LogDebug("error: sin_port (%d)", __LINE__);
        goto bail;
    };
    pBufferCurrent += rc;

    longValue = sockaddrIn4.sin_addr.S_addr;
    if (-1 == ( rc = ToBuffer(pBufferCurrent,
                              bufferSize - (pBufferCurrent - pBufferOut),
                              longValue)))
    {
        LogDebug("error: sin_addr.s_addr (%d)", __LINE__);
        goto bail;
    };
    pBufferCurrent += rc;

    if ( bufferSize - (pBufferCurrent - pBufferOut ) < sizeof(sockaddrIn4.sin_zero) )
    {
        LogDebug("error: sin_zero: (%d)", __LINE__);
        rc = -1;
        goto bail;
    };

    memcpy(pBufferCurrent, sockaddrIn4.sin_zero, sizeof(sockaddrIn4.sin_zero));
    pBufferCurrent += sizeof(sockaddrIn4.sin_zero);
    rc = pBufferCurrent - pBufferOut;

bail:
    return rc;
}

// serialize a struct sockaddr_in (ipv4) to the buffer
template <>
ssize_t DNSSerializer::ToBuffer(uint8_t * const pBufferOut,
                                size_t     bufferSize,
                                const struct sockaddr_in & sockaddrIn4)
{
    ssize_t rc = -1;
    uint8_t* pBufferCurrent = pBufferOut;

    uint16_t shortValue;  // use for sin_family, sin_port
    uint32_t longValue;   // use for in_addr.s_addr

    if ( -1 == CheckToBufferArguments(pBufferCurrent,
                                      bufferSize,
                                      sizeof(struct sockaddr_in),
                                      __LINE__))
    {
        rc = -1;
        goto bail;
    }

    shortValue = sockaddrIn4.sin_family;
    if (-1 == ( rc = ToBuffer(pBufferCurrent,
                              bufferSize - (pBufferCurrent - pBufferOut),
                              shortValue)))
    {
        LogDebug("error: sin_family\n");
        goto bail;
    };
    pBufferCurrent += rc;

    shortValue = sockaddrIn4.sin_port;
    if (-1 == ( rc = ToBuffer(pBufferCurrent,
                              bufferSize - (pBufferCurrent - pBufferOut),
                              shortValue)))
    {
        LogDebug("error: sin_port\n");
        goto bail;
    };
    pBufferCurrent += rc;

    longValue = sockaddrIn4.sin_addr.s_addr;
    if (-1 == ( rc = ToBuffer(pBufferCurrent,
                              bufferSize - (pBufferCurrent - pBufferOut),
                              longValue)))
    {
        LogDebug("error: sin_addr.s_addr\n");
        goto bail;
    };
    pBufferCurrent += rc;

    if ( bufferSize - (pBufferCurrent - pBufferOut ) < sizeof(sockaddrIn4.sin_zero) )
    {
        LogDebug("error: sin_zero:\n");
        rc = -1;
        goto bail;
    };

    memcpy(pBufferCurrent, sockaddrIn4.sin_zero, sizeof(sockaddrIn4.sin_zero));
    pBufferCurrent += sizeof(sockaddrIn4.sin_zero);
    rc = pBufferCurrent - pBufferOut;

bail:
    return rc;
}

template <>
ssize_t DNSSerializer::FromBuffer(nn::socket::SockAddrIn & sockaddrOut4,
                                  const uint8_t      * pBufferIn,
                                  size_t               bufferSize)
{
    ssize_t rc = -1;
    const uint8_t* pBufferCurrent = pBufferIn;

    uint16_t shortValue;  // use for sin_family, sin_port
    uint32_t longValue;   // use for in_addr.s_addr

    if (-1 == ( rc = FromBuffer(shortValue,
                                pBufferCurrent,
                                bufferSize - (pBufferCurrent - pBufferIn))))
    {
        LogDebug("error: sin_family (%d)", __LINE__);
        goto bail;
    };
    sockaddrOut4.sin_family = static_cast<nn::socket::Family>(shortValue);
    pBufferCurrent += rc;

    if (-1 == ( rc = FromBuffer(shortValue,
                                pBufferCurrent,
                                bufferSize - (pBufferCurrent - pBufferIn))))
    {
        LogDebug("error: sin_port (%d)", __LINE__);
        goto bail;
    };
    sockaddrOut4.sin_port = static_cast<nn::socket::InPortT>(shortValue);
    pBufferCurrent += rc;

    if (-1 == ( rc = FromBuffer(longValue,
                                pBufferCurrent,
                                bufferSize - (pBufferCurrent - pBufferIn))))
    {
        LogDebug("error: sin_addr.s_addr (%d)", __LINE__);
        goto bail;
    };
    sockaddrOut4.sin_addr.S_addr = static_cast<nn::socket::InAddrT>(longValue);
    pBufferCurrent += rc;

    if ( bufferSize - (pBufferCurrent - pBufferIn ) < sizeof(sockaddrOut4.sin_zero) )
    {
        LogDebug("error: sin_zero: (%d)", __LINE__);
        rc = -1;
        goto bail;
    };

    memcpy(sockaddrOut4.sin_zero, pBufferCurrent, sizeof(sockaddrOut4.sin_zero));
    pBufferCurrent += sizeof(sockaddrOut4.sin_zero);
    rc = pBufferCurrent - pBufferIn;

bail:
    return rc;
}

template <>
ssize_t DNSSerializer::FromBuffer(struct sockaddr_in & sockaddrOut4,
                                  const uint8_t      * pBufferIn,
                                  size_t               bufferSize)
{
    ssize_t rc = -1;
    const uint8_t* pBufferCurrent = pBufferIn;

    uint16_t shortValue;  // use for sin_family, sin_port
    uint32_t longValue;   // use for in_addr.s_addr

    if (-1 == ( rc = FromBuffer(shortValue,
                                pBufferCurrent,
                                bufferSize - (pBufferCurrent - pBufferIn))))
    {
        LogDebug("error: sin_family\n");
        goto bail;
    };
    sockaddrOut4.sin_family = shortValue;
    pBufferCurrent += rc;

    if (-1 == ( rc = FromBuffer(shortValue,
                                pBufferCurrent,
                                bufferSize - (pBufferCurrent - pBufferIn))))
    {
        LogDebug("error: sin_port\n");
        goto bail;
    };
    sockaddrOut4.sin_port = shortValue;
    pBufferCurrent += rc;

    if (-1 == ( rc = FromBuffer(longValue,
                                pBufferCurrent,
                                bufferSize - (pBufferCurrent - pBufferIn))))
    {
        LogDebug("error: sin_addr.s_addr\n");
        goto bail;
    };
    sockaddrOut4.sin_addr.s_addr = longValue;
    pBufferCurrent += rc;

    if ( bufferSize - (pBufferCurrent - pBufferIn ) < sizeof(sockaddrOut4.sin_zero) )
    {
        LogDebug("error: sin_zero:\n");
        rc = -1;
        goto bail;
    };

    memcpy(sockaddrOut4.sin_zero, pBufferCurrent, sizeof(sockaddrOut4.sin_zero));
    pBufferCurrent += sizeof(sockaddrOut4.sin_zero);
    rc = pBufferCurrent - pBufferIn;

bail:
    return rc;
}

}}}}// namespace nn::socket::resolver::serializer
