﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

extern "C"
{
#include <stdint.h>
};

#include <cstdio>
#include <cstring>
#include <cstdlib>
#include <nn/socket/socket_Api.h>
#include <nn/nn_SdkLog.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>

namespace nn { namespace socket { namespace resolver { namespace serializer {

#ifdef DEBUG_SERIALIZER
#define SERIALIZER_HEXDUMP(p, q, format, ...) DNSSerializer::hexdump
#else
#define SERIALIZER_HEXDUMP(fmt, ...) do {} while (0)
#endif

class DNSSerializer
{
public:

    /**
     * @brief get the size necessary to serialize a T to a buffer
     * @param in the T used to calculate the size
     * @note if your data type includes a list then this function
     * should only be called at the head
     * @return size needed to serialize into a buffer
     */
    template <typename T>
    size_t SizeOf(const T& in);

    /**
     * @brief Depending on the type provided this function returns the
     * serialized size of a single pointer to T or multiple T's.
     * @details The design of this function suggests that the type
     * is a collection of T with a definite end condition such as a
     * NULL-terminator.
     * @param in the T used to calculate the size
     * @note if your data type includes a list then this function
     * should only be called at the head
     * @return size needed to serialize into a buffer
     */
    template <typename T>
    size_t SizeOf(const T* in);

    /**
     * @brief Depending on the type provided this function returns the
     * serialized size of a single pointer to T or multiple T's.
     * @details The design of this function implies that the 'in'
     * parameter is a collection of T that contains 'countIn' objects.
     * @param in the T used to calculate the size
     * @param countIn the number of T in the 'in' parameter
     * @note if your data type includes a list then this function
     * should only be called at the head
     * @return size needed to serialize into a buffer
     */
    template <typename T>
    size_t SizeOf(const T* in, size_t countIn);

    /**
     * @brief get the size necessary to serialize an array of pointers
     * of T to a buffer
     * @param inArray the head of the array of T used to calculate the
     * size
     * @param countOut the number of T found
     * @return size needed to serialize into a buffer, includes the
     * length parameter
     * @note this includes a leading length parameter
     */
    template <typename T>
    size_t SizeOf(const T** inArray, uint32_t & countOut);

    /**
     * @brief serialize a T to a buffer
     * @param pBufferOut a pointer to the buffer
     * @param bufferSize size of the buffer
     * @return size of buffer wrote to or -1 on error
     */
    template <typename T>
    ssize_t ToBuffer(uint8_t* const pBufferOut,
                     size_t   bufferSize,
                     const T& In);

    /**
     * @brief create a T from a buffer
     * @param out the T to write to
     * @param pBufferIn buffer to read from
     * @param bufferSize size of the buffer
     * @return size of buffer read from or -1 on error
     */
    template <typename T>
    ssize_t FromBuffer(T&             out,
                       const uint8_t* pBufferIn,
                       size_t         bufferSize);

    /**
     * @brief serialize a T pointer value to a buffer
     * @param pBufferOut a pointer to the buffer
     * @param bufferSize size of the buffer
     * @return size of buffer wrote to or -1 on error
     * \note the first parameter in the buffer will often be a length
     *       parameter, indicating the number of T* serialized. This is
     *       not the case when T* = char*
     */
    template <typename T>
    ssize_t ToBuffer(uint8_t* const pBufferOut,
                     size_t    bufferSize,
                     T*        pIn); // const me

    /**
     * @brief serialize an array of T pointers to a buffer
     * @param pBufferOut a pointer to the buffer
     * @param bufferSize size of the buffer
     * @param ppIn an array of pointers to write to a buffer
     * @return size of buffer wrote to or -1 on error
     * \note the first parameter will, almost always, be a length
     *       parameter, indicating the number of T* serialized
     */
    template <typename T>
    ssize_t ToBuffer(uint8_t* const pBufferOut,
                     size_t         bufferSize,
                     T**            ppArrayIn);

    /**
     * @brief create a new T from a buffer
     * @param allocate a new T at the pointer address of pOut
     * @param pBufferIn buffer to read from
     * @param bufferSize size of the buffer
     * @return size of buffer read from or -1 on error
     * \note the caller is responsible for cleaning up pOut later
     */
    template <typename T>
    ssize_t FromBuffer(T             *& pOut,
                       const uint8_t *  pBufferIn,
                       size_t           bufferSize);

    /**
     * @brief create a new array of pointers to T from a buffer
     * @param allocates a new array of pointers to T of pArrayOut
     * @param pBufferIn buffer to read from
     * @param bufferSize size of the buffer
     * @return size of buffer read from or -1 on error
     * \note the caller is responsible for cleaning up pArrayOut later
     */
    template <typename T>
    ssize_t FromBuffer(T**&           ppArrayOut,
                       const uint8_t* pBufferIn,
                       size_t         bufferSize);

    /**
     * @brief serialize an array of T to a buffer
     * @param pBufferOut a pointer to the buffer
     * @param bufferSize size of the buffer
     * @return size of buffer wrote to or -1 on error
     */
    template <typename T>
    ssize_t ToBuffer(uint8_t * const pBufferOut,
                     size_t          bufferSize,
                     const T * const pIn,
                     size_t          count);

    /**
     * @brief fill a preallocated array of T from a serialized buffer
     * with count objects
     * @details bufferIn contains a serialized array of T and
     * bufferSize is the size of the serialized buffer. pOut is an
     * array of T to fill and pOutSize is the native (sizeof) size.
     * While in and out bytes remain or until an overflow might occur
     * T will will  to pOut at the iteration index (i.e. pOut[0],
     * pOut[1], etc.)
     * @param pOutSize the size of the preallocated buffer
     * @param pBufferIn buffer to read from.
     * @param bufferSize size of the buffer
     * @return size of buffer read from or -1 on error; null counts
     * as zero
     */
    template <typename T>
    ssize_t FromBuffer(T * const             pArray,
                       size_t                arraySize,
                       const uint8_t * const pBufferIn,
                       size_t                bufferSize,
                       size_t                count);
protected:

    /**
     * @brief serialize a T to a buffer
     * @note for use with iterative structures (i.e. struct addrinfo)
     * @param pBufferOut a pointer to the buffer
     * @param bufferSize size of the buffer
     * @return size of buffer wrote to or -1 on error
     */
    template <typename T>
    ssize_t ToBufferInternal(uint8_t* const pBufferOut,
                             size_t   bufferSize,
                             const T& In);

    /**
     * @brief create a T from a buffer
     * @note for use with iterative structures (i.e. struct addrinfo)
     * @param out the T to write to
     * @param pBufferIn buffer to read from
     * @param bufferSize size of the buffer
     * @return size of buffer read from or -1 on error
     */
    template <typename T>
    ssize_t FromBufferInternal(T&             out,
                               const uint8_t* pBufferIn,
                               size_t         bufferSize);

    /**
     * @brief check parameters passed to FromBuffer
     * @param bufferSize the size of the buffer
     * @param sizeRequired the required size of the buffer
     * @param errorNumber a unique error number
     * @return -1 on error
     */
    ssize_t CheckToBufferArguments(const uint8_t* pBuffer,
                                   size_t         bufferSize,
                                   size_t         sizeRequired,
                                   int            errorNumber);

    /**
     * @brief host to network for 'long' values
     */
    uint32_t InternalHton(const uint32_t& in);

    /**
     * @brief host to network for 'short' values
     */
    uint16_t InternalHton(const uint16_t& in);

    /**
     * @brief network to host for 'long' values
     */
    uint32_t InternalNtoh(const uint32_t& in);

    /**
     * @brief network to host for 'short' values
     */
    uint16_t InternalNtoh(const uint16_t& in);

    /**
     * @brief debug/hexdump something to DEBUG_SERIALIZER
     * @param tag a tag for the hex dump
     * @param p pointer
     * @param q size
     */
    static void hexdump(const char* tag, const uint8_t* p, ssize_t q);
};

}}}}// namespace nn::socket::resolver::serializer
