﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "resolver_ClientServerShared.h"

#include <nn/socket/socket_Api.h>
#include <nn/nn_Assert.h>
#include <nn/nn_SdkLog.h>
#include <algorithm>
#include <functional>
#include <random>

#include <nn/socket/socket_ResolverOptionsPrivate.h>

/**
 * Types to string names
 */

#define StringifyCase(cast, type)                                       \
    case static_cast<cast>(type):                                       \
    {                                                                   \
        const char* name = #type;                                       \
        return name;                                                    \
    };

/**
 * Functions in this file are in the client resolver library because
 * the client library is also included with the server. The header improves
 * visibility to these functions
 */

namespace nn { namespace socket { namespace resolver {

int CreateRandomId(int &randomIdOut) NN_NOEXCEPT
{
    // version 1.0 of the cancellation feature reserved from 0-4
    // we also reserve -1 for errors; so the range for allowable
    // handles is [5-2147483647]

    static const uint32_t CancelMinimum = 4;
    uint64_t tickValue = nn::os::GetSystemTick().GetInt64Value();
    std::mt19937 mt(static_cast<uint64_t>(tickValue));
    randomIdOut = std::uniform_int_distribution<int>(CancelMinimum, INT_MAX)(mt);
    return 0;
};

// OperationState
const char* GetOperationStateString(const OperationState& stateIn)
{
    switch (stateIn)
    {
        StringifyCase(OperationState, OperationState_Initial);
        StringifyCase(OperationState, OperationState_AcquireSlotGenericError);
        StringifyCase(OperationState, OperationState_AcquireSlotUnableToCreateHandleID);
        StringifyCase(OperationState, OperationState_AcquireSlotAlreadyExists);
        StringifyCase(OperationState, OperationState_AcquireSlotSuccess);
        StringifyCase(OperationState, OperationState_AcquireSlotTableFull);

        StringifyCase(OperationState, OperationState_SetSlotRunningGenericError);
        StringifyCase(OperationState, OperationState_SetSlotRunningZeroHandle);
        StringifyCase(OperationState, OperationState_SetSlotRunningPending);
        StringifyCase(OperationState, OperationState_SetSlotRunningSuccess);
        StringifyCase(OperationState, OperationState_SetSlotRunningDefault);
        StringifyCase(OperationState, OperationState_SetSlotRunningNotFound);

        StringifyCase(OperationState, OperationState_IsSlotCancelledOrExpiredGenericError);
        StringifyCase(OperationState, OperationState_IsSlotCancelledOrExpiredZeroHandle);
        StringifyCase(OperationState, OperationState_IsSlotCancelledOrExpiredPending);
        StringifyCase(OperationState, OperationState_IsSlotCancelledOrExpiredAcquired);
        StringifyCase(OperationState, OperationState_IsSlotCancelledOrExpiredRunning);
        StringifyCase(OperationState, OperationState_IsSlotCancelledOrExpiredCancelled);
        StringifyCase(OperationState, OperationState_IsSlotCancelledOrExpiredExpired);
        StringifyCase(OperationState, OperationState_IsSlotCancelledOrExpiredNotFound);

        StringifyCase(OperationState, OperationState_SetSlotCancelledOrExpiredGenericError);
        StringifyCase(OperationState, OperationState_SetSlotCancelledOrExpiredZeroHandle);
        StringifyCase(OperationState, OperationState_SetSlotCancelledOrExpiredPending);
        StringifyCase(OperationState, OperationState_SetSlotCancelledOrExpiredAcquired);
        StringifyCase(OperationState, OperationState_SetSlotCancelledOrExpiredRunning);
        StringifyCase(OperationState, OperationState_SetSlotCancelledOrExpiredCancelled);
        StringifyCase(OperationState, OperationState_SetSlotCancelledOrExpiredExpired);
        StringifyCase(OperationState, OperationState_SetSlotCancelledOrExpiredNotFound);

        StringifyCase(OperationState, OperationState_SetSlotPendingGenericError);
        StringifyCase(OperationState, OperationState_SetSlotPendingZeroHandle);
        StringifyCase(OperationState, OperationState_SetSlotPendingPending);
        StringifyCase(OperationState, OperationState_SetSlotPendingAcquired);
        StringifyCase(OperationState, OperationState_SetSlotPendingRunning);
        StringifyCase(OperationState, OperationState_SetSlotPendingCancelled);
        StringifyCase(OperationState, OperationState_SetSlotPendingExpired);
        StringifyCase(OperationState, OperationState_SetSlotPendingNotFound);

        StringifyCase(OperationState, OperationState_CancelAllSlotsForProcessIDGenericError);
        StringifyCase(OperationState, OperationState_CancelAllSlotsForProcessIDNoSlotsCancelled);
        StringifyCase(OperationState, OperationState_CancelAllSlotsForProcessIDSlotsCancelled);
    default:
        ;
    };

    return "OperationState_Undefined";
}; //NOLINT(impl/function_size)



/**
 * @brief Get the type string for the provided option.
 *
 * @param in This parameter is the ResolverOptionKey to stringify.
 */
const char* GetResolverOptionKeyString(const nn::socket::ResolverOptionKey & in)
{
    // The private and local options are defined in a different file
    // than the resolver options type and so it cannot be a member of
    // the same enum therefore we cast 'in' to unsigned to switch
    switch (static_cast<uint32_t>(in))
    {
        StringifyCase(uint32_t, ResolverOptionKey::GetMinimumValue);
        StringifyCase(uint32_t, ResolverOptionKey::GetCancelHandleInteger);
        StringifyCase(uint32_t, ResolverOptionKey::GetMaximumValue);

        StringifyCase(uint32_t, ResolverOptionKey::SetMinimumValue);
        StringifyCase(uint32_t, ResolverOptionKey::SetCancelHandleInteger);
        StringifyCase(uint32_t, ResolverOptionKey::SetRemoveDomainnameFromCachePointer);
        StringifyCase(uint32_t, ResolverOptionKey::SetRemoveIpAddressFromCacheUnsigned32);
        StringifyCase(uint32_t, ResolverOptionKey::SetMaximumValue);

        StringifyCase(uint32_t, ResolverOptionKey::RequestMinimumValue);
        StringifyCase(uint32_t, ResolverOptionKey::RequestEnableServiceDiscoveryBoolean);
        StringifyCase(uint32_t, ResolverOptionKey::RequestCancelHandleInteger);
        StringifyCase(uint32_t, ResolverOptionKey::RequestEnableDnsCacheBoolean);
        StringifyCase(uint32_t, ResolverOptionKey::RequestMaximumValue);

#if ! defined(NN_SDK_BUILD_RELEASE)
        StringifyCase(uint32_t, ResolverOptionPrivateKey::GetMinimumValue);
        StringifyCase(uint32_t, ResolverOptionPrivateKey::GetDnsServerAddressesPointer);
        StringifyCase(uint32_t, ResolverOptionPrivateKey::GetCacheEntryCountForDomainnamePointer);
        StringifyCase(uint32_t, ResolverOptionPrivateKey::GetCacheEntryCountForIpUnsigned32);
        StringifyCase(uint32_t, ResolverOptionPrivateKey::GetMaximumValue);
        StringifyCase(uint32_t, ResolverOptionPrivateKey::SetMinimumValue);
        StringifyCase(uint32_t, ResolverOptionPrivateKey::SetFlushCacheBoolean);
        StringifyCase(uint32_t, ResolverOptionPrivateKey::SetTimeToLiveForDomainnamePointer);
        StringifyCase(uint32_t, ResolverOptionPrivateKey::SetDnsServerAddressesPointer);
        StringifyCase(uint32_t, ResolverOptionPrivateKey::SetMaximumValue);
        StringifyCase(uint32_t, ResolverOptionPrivateKey::RequestMinimumValue);
        StringifyCase(uint32_t, ResolverOptionPrivateKey::RequestMaximumValue);

        StringifyCase(uint32_t, ResolverOptionLocalKey::MinimumValue);
        StringifyCase(uint32_t, ResolverOptionLocalKey::GetHostByNameBufferSizeUnsigned64);
        StringifyCase(uint32_t, ResolverOptionLocalKey::GetHostByAddrBufferSizeUnsigned64);
        StringifyCase(uint32_t, ResolverOptionLocalKey::GetHostErrorStringBufferSizeUnsigned64);
        StringifyCase(uint32_t, ResolverOptionLocalKey::GaiErrorStringBufferSizeUnsigned64);
        StringifyCase(uint32_t, ResolverOptionLocalKey::GetAddrInfoBufferSizeUnsigned64);
        StringifyCase(uint32_t, ResolverOptionLocalKey::GetAddrInfoHintsBufferSizeUnsigned64);
        StringifyCase(uint32_t, ResolverOptionLocalKey::MaximumValue);
#endif
    default:
        ;
    };

    return "ResolverOptionKey_Undefined";
}; //NOLINT(impl/function_size)

const char* GetResolverOptionTypeString(const nn::socket::ResolverOptionType & in)
{
    switch (in)
    {
        StringifyCase(ResolverOptionType, ResolverOptionType::Boolean);
        StringifyCase(ResolverOptionType, ResolverOptionType::Integer);
        StringifyCase(ResolverOptionType, ResolverOptionType::Unsigned32);
        StringifyCase(ResolverOptionType, ResolverOptionType::Unsigned64);
        StringifyCase(ResolverOptionType, ResolverOptionType::Double);
        StringifyCase(ResolverOptionType, ResolverOptionType::Pointer);
    default:
        ;
    };

    return "ResolverOptionType::Undefined";
};

}}};
