﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "resolver_LibraryReferenceCountHelper.h"
#include <nn/os.h>
#include <mutex>

//#define LOG_LEVEL LOG_LEVEL_MAJOR
#define LOG_MODULE_NAME "resolver refcount" // NOLINT(preprocessor/const)
#include <nn/socket/resolver/private/resolver_DebugLog.h>

namespace nn { namespace socket { namespace resolver {

namespace {
bool g_ResolverEventActive = false;

nn::os::EventType  g_ResolverActiveEvent;
::std::atomic<unsigned int> g_NumSocketCallsActive(0);
}

void RefCountHelperInitialize() NN_NOEXCEPT
{
    if(!g_ResolverEventActive)
    {
        nn::os::InitializeEvent(&g_ResolverActiveEvent, false, nn::os::EventClearMode_AutoClear);
        g_ResolverEventActive = true;
    };
};

void RefCountHelperFinalize() NN_NOEXCEPT
{
    if(g_ResolverEventActive)
    {
        // This variable is a gate - make sure to mark false before
        // finalizing event.
        g_ResolverEventActive = false;
        nn::os::FinalizeEvent(&g_ResolverActiveEvent);
    };
};

bool IncrementSocketIfNeeded() NN_NOEXCEPT
{
    // If calls disabled before here, this add need not signal
    g_NumSocketCallsActive.fetch_add(1);

    // Key race spot here.

    bool returnVal = AreResolverCallsEnabled();

    if(!returnVal)
    {
        DecrementSocketAndSignalIfNeeded();
    };

    return returnVal;
};

void DecrementSocketAndSignalIfNeeded() NN_NOEXCEPT
{

    int curCount = g_NumSocketCallsActive.fetch_sub(1);

    // Value given is before subtract - take this into account!
    if(!AreResolverCallsEnabled() && curCount == 1)
    {
        if(g_ResolverEventActive)
        {
            nn::os::SignalEvent(&g_ResolverActiveEvent);
        };
    };
};

void WaitForAllResolverCallsToComplete() NN_NOEXCEPT
{
    int numCalls = 0;

    if(g_ResolverEventActive)
    {
        while((numCalls = g_NumSocketCallsActive) != 0)
        {
            LogDebug("%d calls still active\n", numCalls);
            nn::os::WaitEvent(&g_ResolverActiveEvent);
        };
    };
};

void EnableResolverCalls() NN_NOEXCEPT
{
    // Clear any old events.
    if(g_ResolverEventActive)
    {
        nn::os::ClearEvent(&g_ResolverActiveEvent);
    };

    // Zero lower count bits, just in case and enable high bit.
    g_NumSocketCallsActive = 0x80000000;
};

void DisableResolverCalls() NN_NOEXCEPT
{
    // High bit off - no more requests allowed.
    g_NumSocketCallsActive.fetch_and(~0x80000000);
};

bool AreResolverCallsEnabled() NN_NOEXCEPT
{
    return (g_NumSocketCallsActive & 0x80000000) != 0;
};

}}}; // nn::socket::resolver
