﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
/*
  Based on:

   Copyright (c) 2004 by Internet Systems Consortium, Inc. ("ISC")
   Copyright (c) 1996,1999 by Internet Software Consortium.

   Permission to use, copy, modify, and distribute this software for any
   purpose with or without fee is hereby granted, provided that the above
   copyright notice and this permission notice appear in all copies.

   THE SOFTWARE IS PROVIDED "AS IS" AND ISC DISCLAIMS ALL WARRANTIES
   WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
   MERCHANTABILITY AND FITNESS.  IN NO EVENT SHALL ISC BE LIABLE FOR
   ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
   WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
   ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT
   OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */

#include <cstring>
#include <nn/os.h>
#include <nn/socket.h>

namespace nn     {
namespace socket {
namespace detail {
namespace        {

const char Digits[]         = "0123456789";
const int  IP4AddressLength = 4;

int inetPton4(const char *pAddressString, uint8_t* pOutAddressBuffer)
NN_NOEXCEPT
{
    uint8_t  pAddress[IP4AddressLength] = {0};
    uint8_t* pOctet;
    bool     sawDigit;
    int      octetCount;
    uint8_t  currentChar;
    uint32_t addressValue;
    const char*    pDigitStart;

    sawDigit   = false;
    octetCount = 0;
    pOctet     = pAddress;
    if (pAddressString == nullptr || pOutAddressBuffer == nullptr)
    {
        SetLastError(Errno::EAfNoSupport);
        return 0;
    }
    while ((currentChar = *pAddressString++) != '\0')
    {
        if ((pDigitStart = strchr(Digits, currentChar)) != NULL)
        {
            addressValue = *pOctet * 10 + (pDigitStart - Digits);

            if ((sawDigit && *pOctet == 0) || addressValue > 255)
            {
                SetLastError(Errno::EAfNoSupport);
                return 0;
            }

            *pOctet = addressValue;

            if (!sawDigit)
            {
                if (++octetCount > 4)
                {
                    SetLastError(Errno::EAfNoSupport);
                    return 0;
                }

                sawDigit = true;
            }
        } else if (currentChar == '.' && sawDigit) {

            if (octetCount == 4)
            {
                SetLastError(Errno::EAfNoSupport);
                return 0;
            }

            pOctet++;

            sawDigit = false;
        } else {
            SetLastError(Errno::EAfNoSupport);
            return 0;
        }
    }

    if (octetCount < 4)
    {
        SetLastError(Errno::EAfNoSupport);
        return 0;
    }

    memcpy(pOutAddressBuffer, pAddress, IP4AddressLength);

    return 1;
}

} // namespace

int socketPton(nn::socket::Family family, const char* pAddressString, void* pOutAddressBuffer)
NN_NOEXCEPT
{
    switch (family)
    {
    case Family::Af_Inet:
        return inetPton4(pAddressString, (uint8_t*)pOutAddressBuffer);
    case Family::Af_Inet6:
        SetLastError(Errno::EAfNoSupport);
    default:
        return -1;
    }
}

int socketPton(int family, const char* pAddressString, void* pOutAddressBuffer)
NN_NOEXCEPT
{
    switch (family)
    {
    case AF_INET:
        return inetPton4(pAddressString, (uint8_t*)pOutAddressBuffer);
    case AF_INET6:
        SetLastError(Errno::EAfNoSupport);
    default:
        return -1;
    }
}

}}}
