﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
/*
  Based on:

   Copyright (c) 2004 by Internet Systems Consortium, Inc. ("ISC")
   Copyright (c) 1996-1999 by Internet Software Consortium.

   Permission to use, copy, modify, and distribute this software for any
   purpose with or without fee is hereby granted, provided that the above
   copyright notice and this permission notice appear in all copies.

   THE SOFTWARE IS PROVIDED "AS IS" AND ISC DISCLAIMS ALL WARRANTIES
   WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
   MERCHANTABILITY AND FITNESS.  IN NO EVENT SHALL ISC BE LIABLE FOR
   ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
   WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
   ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT
   OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */

#include <cstdio>
#include <cstring>
#include <nn/os.h>
#include <nn/socket/socket_Types.h>
#include <nn/socket/socket_Api.h>
#include <nn/util/util_FormatString.h>

namespace nn     {
namespace socket {
namespace detail {
namespace        {

char* inetNtop4(const uint8_t* pNetworkAddress, char* addressStringBuffer, socklen_t addressStringBufferLength)
NN_NOEXCEPT
{
    static const char format[] = "%u.%u.%u.%u";
    char buffer[INET_ADDRSTRLEN];
    int  length;
    char* rc = NULL;

    length = nn::util::SNPrintf(buffer,
                                sizeof(buffer),
                                format,
                                pNetworkAddress[0],
                                pNetworkAddress[1],
                                pNetworkAddress[2],
                                pNetworkAddress[3]);

    if (length >= static_cast<int>(addressStringBufferLength))
    {
        nn::socket::SetLastError(nn::socket::Errno::ENoSpc);
        goto bail;
    }

    strncpy(addressStringBuffer, buffer, length + 1);
    rc = addressStringBuffer;

bail:
    return rc;
}

} // namespace

char* socketNtop(Family family, const void* pNetworkAddress, char* addressStringBuffer, SockLenT addressStringBufferLength)
NN_NOEXCEPT
{
    char* rc = nullptr;
    if ( nullptr == pNetworkAddress )
    {
        nn::socket::SetLastError(nn::socket::Errno::EInval);
        goto bail;
    }
    else if ( nullptr == addressStringBuffer )
    {
        nn::socket::SetLastError(nn::socket::Errno::ENoSpc);
        goto bail;
    };

    switch (family)

    {
    case Family::Af_Inet:
        rc = inetNtop4((const uint8_t*)pNetworkAddress, addressStringBuffer, addressStringBufferLength);
        break;
    case Family::Af_Inet6:
    default:
        nn::socket::SetLastError(nn::socket::Errno::EAfNoSupport);
    };

bail:
    return rc;
}

char* socketNtop(int family, const void* pNetworkAddress, char* addressStringBuffer, socklen_t addressStringBufferLength)
NN_NOEXCEPT
{
    char* rc = NULL;
    if ( NULL == pNetworkAddress )
    {
        nn::socket::SetLastErrno(EINVAL);
        goto bail;
    }
    else if ( NULL == addressStringBuffer )
    {
        nn::socket::SetLastErrno(ENOSPC);
        goto bail;
    };

    switch (family)

    {
    case AF_INET:
        rc = inetNtop4((const uint8_t*)pNetworkAddress, addressStringBuffer, addressStringBufferLength);
        break;
    case AF_INET6:
    default:
        nn::socket::SetLastErrno(EAFNOSUPPORT);
    };

bail:
    return rc;
}

}}}
