﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>

#include <nn/socket/private/socket_PlatformTypesTranslation.h>

#include "socket_Api.h"
#include "socket_Allocator.h"

namespace nn     {
namespace socket {
namespace detail {

void CopyToPlatform(sockaddr_in* pDest, const SockAddrIn* pSrc)
NN_NOEXCEPT
{
    if( nullptr != pSrc )
    {
        pDest->sin_len = pSrc->sin_len;
        pDest->sin_family = static_cast<sa_family_t>(pSrc->sin_family);
        pDest->sin_port = pSrc->sin_port;
        std::memcpy(&pDest->sin_addr, &pSrc->sin_addr, sizeof(pDest->sin_addr));
        std::memcpy(pDest->sin_zero, pSrc->sin_zero, sizeof(pDest->sin_zero));
    }
}

void CopyFromPlatform(SockAddrIn* pDest, const sockaddr_in* pSrc)
NN_NOEXCEPT
{
    if( nullptr != pDest )
    {
        pDest->sin_len = pSrc->sin_len;
        pDest->sin_family = static_cast<Family>(pSrc->sin_family);
        pDest->sin_port = pSrc->sin_port;
        std::memcpy(&pDest->sin_addr, &pSrc->sin_addr, sizeof(pDest->sin_addr));
        std::memcpy(pDest->sin_zero, pSrc->sin_zero, sizeof(pDest->sin_zero));
    }
}

bool CopyFromPlatform(AddrInfo** pDest, const addrinfo* pSrc)
NN_NOEXCEPT
{
    bool bRet = true;
    AddrInfo* pNewAddrInfo = nullptr;

    if( nullptr != pDest )
    {
        pNewAddrInfo = reinterpret_cast<nn::socket::AddrInfo*>(nn::socket::detail::Calloc(1, sizeof(*pNewAddrInfo)));

        if( nullptr == pNewAddrInfo )
        {
            bRet = false;
            goto exit;
        }

        pNewAddrInfo->ai_flags = static_cast<AddrInfoFlag>(pSrc->ai_flags);
        pNewAddrInfo->ai_family = static_cast<Family>(pSrc->ai_family);
        pNewAddrInfo->ai_socktype = static_cast<Type>(pSrc->ai_socktype);
        pNewAddrInfo->ai_protocol = static_cast<Protocol>(pSrc->ai_protocol);
        pNewAddrInfo->ai_addrlen = static_cast<SockLenT>(sizeof(*(pNewAddrInfo->ai_addr)));

        if( nullptr != pSrc->ai_addr )
        {
            pNewAddrInfo->ai_addr = reinterpret_cast<nn::socket::SockAddr *>(nn::socket::detail::Calloc(1, sizeof(*pNewAddrInfo->ai_addr)));

            if( nullptr == pNewAddrInfo->ai_addr )
            {
                bRet = false;
                goto exit;
            }

            CopyFromPlatform(reinterpret_cast<SockAddrIn*>(pNewAddrInfo->ai_addr), reinterpret_cast<const sockaddr_in*>(pSrc->ai_addr));
        }

        if( nullptr != pSrc->ai_canonname )
        {
            size_t nameLen = strlen(pSrc->ai_canonname);
            pNewAddrInfo->ai_canonname = reinterpret_cast<char *>(nn::socket::detail::Calloc(1, nameLen + 1));

            if( nullptr == pNewAddrInfo->ai_canonname )
            {
                bRet = false;
                goto exit;
            }

            strncpy(pNewAddrInfo->ai_canonname, pSrc->ai_canonname, nameLen);
        }

        if( nullptr != pSrc->ai_next )
        {
            if( CopyFromPlatform(&pNewAddrInfo->ai_next, pSrc->ai_next) == false )
            {
                bRet = false;
                goto exit;
            }
        }

        *pDest = pNewAddrInfo;
    }

exit:
    if( false == bRet && nullptr != pNewAddrInfo )
    {
        nn::socket::detail::Free(pNewAddrInfo->ai_canonname);
        nn::socket::detail::Free(pNewAddrInfo->ai_addr);
        nn::socket::detail::Free(pNewAddrInfo);
        nn::socket::detail::SetLastError(Errno::ENoMem);
    }

    return bRet;
}

void FreeCopiedAddrInfo(AddrInfo *pAddrInfo)
NN_NOEXCEPT
{
    if( nullptr == pAddrInfo )
    {
        return;
    }

    nn::socket::detail::Free(pAddrInfo->ai_canonname);
    nn::socket::detail::Free(pAddrInfo->ai_addr);

    FreeCopiedAddrInfo(pAddrInfo->ai_next);

    nn::socket::detail::Free(pAddrInfo);
}

void CopyToPlatform(fd_set* pDest, const FdSet* pSrc)
NN_NOEXCEPT
{
    if( nullptr != pSrc )
    {
        std::memcpy(pDest->fds_bits, pSrc->fds_bits, sizeof(pDest->fds_bits));
    }
}

void CopyFromPlatform(FdSet* pDest, const fd_set* pSrc)
NN_NOEXCEPT
{
    if( nullptr != pDest )
    {
        std::memcpy(pDest->fds_bits, pSrc->fds_bits, sizeof(pDest->fds_bits));
    }
}

}}}

