﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "sfsmpl_CreateCalculator.h"

#include <nn/nn_SdkAssert.h>
#include <nn/nn_Abort.h>
#include <nn/sf/sf_Types.h> // for nn::sf::SharedPointer
#include <nn/sfsmpl/sfsmpl_ICalculator.sfdl.h> // for ICalculator

namespace nn { namespace sfsmpl {

namespace {

// Shim ライブラリ実装用のサービスオブジェクトへの共有ポインタ
// DFC と HIPC とで共通でよい。
nn::sf::SharedPointer<ICalculator> g_Calculator;

}

void InitializeCalculatorByDfc() NN_NOEXCEPT
{
    NN_SDK_ASSERT(!g_Calculator);
    g_Calculator = GetCalculatorByDfc();
}

void InitializeCalculatorByHipc() NN_NOEXCEPT
{
    NN_SDK_ASSERT(!g_Calculator);
    g_Calculator = GetCalculatorByHipc();
}

// 以下は DFC と HIPC とで共通コード

void FinalizeCalculator() NN_NOEXCEPT
{
    NN_SDK_ASSERT(g_Calculator);
    // 共有ポインタへの nullptr の代入で解放できる。
    g_Calculator = nullptr;
}

int32_t Negate(int32_t x) NN_NOEXCEPT
{
    int32_t ret;
    // g_Calculator の operator-> 経由でインターフェイスメソッド関数を呼び出せる。
    auto result = g_Calculator->Negate(&ret, x);
    // 今回の場合、インターフェイス実装においてエラーは返さないので、
    // 基本的に result は成功と仮定してよい。
    // IPC のセッションが何らかの理由で切れていた場合などにはエラーが返ることがあるが、
    // その場合、基本的にどうにもならないため abort としている。
    NN_ABORT_UNLESS(result.IsSuccess());
    return ret;
}

int32_t Add(int32_t x, int32_t y) NN_NOEXCEPT
{
    int32_t ret;
    auto result = g_Calculator->Add(&ret, x, y);
    NN_ABORT_UNLESS(result.IsSuccess());
    return ret;
}

int32_t Sub(int32_t x, int32_t y) NN_NOEXCEPT
{
    int32_t ret;
    auto result = g_Calculator->Sub(&ret, x, y);
    NN_ABORT_UNLESS(result.IsSuccess());
    return ret;
}

}}
