﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/sf/sf_DefaultAllocationPolicy.h>

#include <nn/os/os_SdkThreadLocalStorage.h>
#include <nn/os/os_Mutex.h>
#include <atomic>
#include <mutex>
#include <nn/nn_Abort.h>
#include <nn/util/util_Exchange.h>
#include <new>

namespace nn { namespace sf {

namespace {

    struct DefaultAllocatorImpl
    {
        os::MutexType tlsAllocationLock;
        bool isManuallyInitialized;
        std::atomic_bool tlsAllocated;
        os::TlsSlot currentMemoryResourceTlsSlot;
        MemoryResource* defaultMemoryResource;

        void EnsureCurrentMemoryResourceTlsSlotInitialized() NN_NOEXCEPT
        {
            if (!tlsAllocated.load(std::memory_order_acquire))
            {
                os::LockMutex(&tlsAllocationLock);
                if (!tlsAllocated.load(std::memory_order_relaxed))
                {
                    NN_ABORT_UNLESS_RESULT_SUCCESS(os::SdkAllocateTlsSlot(&currentMemoryResourceTlsSlot, nullptr));
                    tlsAllocated.store(true, std::memory_order_release);
                }
                os::UnlockMutex(&tlsAllocationLock);
            }
        }

        MemoryResource* GetDefaultMemoryResource() NN_NOEXCEPT
        {
            return defaultMemoryResource;
        }

        MemoryResource* SetDefaultMemoryResource(MemoryResource* pMemoryResource) NN_NOEXCEPT
        {
            return util::Exchange(&defaultMemoryResource, pMemoryResource);
        }

        MemoryResource* GetCurrentMemoryResource() NN_NOEXCEPT
        {
            EnsureCurrentMemoryResourceTlsSlotInitialized();
            return reinterpret_cast<MemoryResource*>(os::GetTlsValue(currentMemoryResourceTlsSlot));
        }

        MemoryResource* SetCurrentMemoryResource(MemoryResource* pMemoryResource) NN_NOEXCEPT
        {
            EnsureCurrentMemoryResourceTlsSlotInitialized();
            auto ret = reinterpret_cast<MemoryResource*>(os::GetTlsValue(currentMemoryResourceTlsSlot));
            os::SetTlsValue(currentMemoryResourceTlsSlot, reinterpret_cast<uintptr_t>(pMemoryResource));
            return ret;
        }

        MemoryResource* GetCurrentEffectiveMemoryResourceImpl() NN_NOEXCEPT
        {
            if (auto p = GetCurrentMemoryResource())
            {
                return p;
            }
            if (auto p = GetGlobalDefaultMemoryResource())
            {
                return p;
            }
            return nullptr;
        }

    };

    DefaultAllocatorImpl g_DefaultAllocatorImpl = { NN_OS_MUTEX_INITIALIZER(false) };

    inline void* DefaultAllocate(std::size_t bytes, std::size_t alignment) NN_NOEXCEPT
    {
        NN_UNUSED(alignment);
        return ::operator new(bytes, std::nothrow);
    }

    inline void DefaultDeallocate(void* p, std::size_t bytes, std::size_t alignment) NN_NOEXCEPT
    {
        NN_UNUSED(alignment);
        NN_UNUSED(bytes);
        return ::operator delete(p, std::nothrow);
    }

    class NewDeleteMemoryResource final
        : public MemoryResource
    {
    private:

        virtual void* do_allocate(std::size_t bytes, std::size_t alignment) NN_NOEXCEPT NN_OVERRIDE
        {
            return DefaultAllocate(bytes, alignment);
        }

        virtual void do_deallocate(void* p, std::size_t bytes, std::size_t alignment) NN_NOEXCEPT NN_OVERRIDE
        {
            return DefaultDeallocate(p, bytes, alignment);
        }

        virtual bool do_is_equal(const MemoryResource& other) const NN_NOEXCEPT NN_OVERRIDE
        {
            return this == &other;
        }

    public:

        static MemoryResource* GetInstance() NN_NOEXCEPT
        {
            static NewDeleteMemoryResource g_NewDeleteMemoryResource;
            return &g_NewDeleteMemoryResource;
        }

    };

}

MemoryResource* GetNewDeleteMemoryResource() NN_NOEXCEPT
{
    return NewDeleteMemoryResource::GetInstance();
}

MemoryResource* GetGlobalDefaultMemoryResource() NN_NOEXCEPT
{
    return g_DefaultAllocatorImpl.GetDefaultMemoryResource();
}

MemoryResource* SetGlobalDefaultMemoryResource(MemoryResource* pMemoryResource) NN_NOEXCEPT
{
    return g_DefaultAllocatorImpl.SetDefaultMemoryResource(pMemoryResource);
}

MemoryResource* GetCurrentMemoryResource() NN_NOEXCEPT
{
    return g_DefaultAllocatorImpl.GetCurrentMemoryResource();
}

MemoryResource* SetCurrentMemoryResource(MemoryResource* pMemoryResource) NN_NOEXCEPT
{
    return g_DefaultAllocatorImpl.SetCurrentMemoryResource(pMemoryResource);
}

MemoryResource* GetCurrentEffectiveMemoryResource() NN_NOEXCEPT
{
    if (auto p = g_DefaultAllocatorImpl.GetCurrentEffectiveMemoryResourceImpl())
    {
        return p;
    }
    return GetNewDeleteMemoryResource();
}

ScopedCurrentMemoryResourceSetter::ScopedCurrentMemoryResourceSetter(MemoryResource* pMemoryResource) NN_NOEXCEPT
    : m_Previous(g_DefaultAllocatorImpl.GetCurrentMemoryResource())
{
    os::SetTlsValue(g_DefaultAllocatorImpl.currentMemoryResourceTlsSlot, reinterpret_cast<uintptr_t>(pMemoryResource));
}

ScopedCurrentMemoryResourceSetter::~ScopedCurrentMemoryResourceSetter() NN_NOEXCEPT
{
    os::SetTlsValue(g_DefaultAllocatorImpl.currentMemoryResourceTlsSlot, reinterpret_cast<uintptr_t>(m_Previous));
}

namespace detail {

void* DefaultAllocateImpl(size_t size, size_t alignment, size_t offset) NN_NOEXCEPT
{
    auto mr = g_DefaultAllocatorImpl.GetCurrentEffectiveMemoryResourceImpl();
    auto q = mr ? mr->allocate(size, alignment) : DefaultAllocate(size, alignment);
    if (!q)
    {
        return nullptr;
    }
    *reinterpret_cast<MemoryResource**>(q) = mr;
    return static_cast<char*>(q) + offset;
}

void DefaultDeallocateImpl(void* p, size_t size, size_t alignment, size_t offset) NN_NOEXCEPT
{
    if (!p)
    {
        return;
    }
    auto q = static_cast<char*>(p) - offset;
    auto mr = *reinterpret_cast<MemoryResource**>(q);
    if (mr)
    {
        mr->deallocate(q, size, alignment);
    }
    else
    {
        DefaultDeallocate(q, size, alignment);
    }
}

}

}}
