﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Result.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/sf/sf_ISharedObject.h>
#include <nn/sf/hipc/sf_HipcHandleTypes.h>
#include <nn/sf/hipc/sf_HipcServiceResolutionApi.h>

#include "sf_HipcEmulatedPort.h"

namespace nn { namespace sf { namespace hipc {

template <typename PortType>
class HipcEmulatedServiceResolver
{
protected:

    virtual HipcClientPortHandle GetServicePortImpl(const char* name) NN_NOEXCEPT = 0;

public:

    // for Server
    virtual Result RegisterService(HipcServerPortHandle* pOut, const char* name, int32_t maxSessions) NN_NOEXCEPT = 0;
    virtual void UnregisterService(const char* name) NN_NOEXCEPT = 0;

    // for Client
    virtual bool IsServiceRegistered(const char* name) NN_NOEXCEPT = 0;
    Result ConnectToService(HipcClientSessionHandle* pOut, const char* name) NN_NOEXCEPT
    {
        SharedPointer<HipcEmulatedClientPort> port(GetServicePortImpl(name), false);
        *pOut = port->Connect(true);
        NN_RESULT_SUCCESS;
    }
    Result TryConnectToService(bool* pValid, HipcClientSessionHandle* pOut, const char* name) NN_NOEXCEPT
    {
        SharedPointer<HipcEmulatedClientPort> port(GetServicePortImpl(name), false);
        auto ret = port->Connect(false);
        *pValid = ret != nullptr;
        *pOut = ret;
        NN_RESULT_SUCCESS;
    }

protected:

    void RegisterPortImpl(PortType port, const char* name) NN_NOEXCEPT
    {
        for (auto&& e : m_Entries)
        {
            if (!e.valid)
            {
                e.valid = true;
                std::strcpy(e.name, name);
                e.port = port;
                return;
            }
        }
        NN_ABORT("[SF-Internal]");
    }

    PortType UnregisterPortImpl(const char* name) NN_NOEXCEPT
    {
        auto p = GetEntry(this, name);
        NN_ABORT_UNLESS(p);
        p->valid = false;
        return p->port;
    }

    bool GetPortImpl(PortType* pOut, const char* name) const NN_NOEXCEPT
    {
        auto p = GetEntry(this, name);
        if (!p)
        {
            return false;
        }
        *pOut = p->port;
        return true;
    }

private:

    template <typename This>
    static auto GetEntry(This* this_, const char* name) NN_NOEXCEPT -> decltype(&this_->m_Entries[0])
    {
        for (auto&& e : this_->m_Entries)
        {
            if (e.valid && std::strcmp(e.name, name) == 0)
            {
                return &e;
            }
        }
        return nullptr;
    }

private:

    static const int EntryCount = 256;

    struct Entry
    {
        bool valid;
        char name[HipcServiceNameLengthMax + 1];
        PortType port;
    };

    Entry m_Entries[EntryCount];
};

}}}
