﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Result.h>
#include <nn/util/util_BitUtil.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/sf/hipc/sf_HipcResult.h>
#include <nn/sf/detail/sf_InternalResult.h>
#include <nn/sf/cmif/server/sf_CmifServerObjectInfo.h>
#include <nn/sf/hipc/detail/sf_HipcMessageBufferAccessor.h>
#include <nn/sf/hipc/detail/sf_HipcMessageBufferAccessor2.h>
#include <nn/sf/hipc/sf_HipcDirectApi.h>
#include <nn/sf/hipc/server/sf_HipcServerApiModel.h>
#include <nn/sf/cmif/server/sf_CmifServerMessage.h>
#include <utility>
#include <cstring>
#include <nn/util/util_BitFlagSet.h>

#include "sf_IHipcObjectDomain.h"

namespace nn { namespace sf { namespace hipc { namespace server {

class Hipc2ServerMessage final
    : public cmif::server::CmifServerMessage
    , private HipcServerApiModelHolder
{
public:

    Hipc2ServerMessage(
        IHipcObjectDomain* pDomain,
        const detail::HipcMessageReader& reader,
        void* pointerBuffer,
        size_t pointerBufferSize,
        void* outMessageBuffer,
        size_t outMessageBufferSize,
        HipcServerApiModel* pServerApiModel
    ) NN_NOEXCEPT
        : HipcServerApiModelHolder(pServerApiModel)
        , m_pDomain(pDomain)
        , m_Reader(reader)
        , m_OutMessageBuffer(outMessageBuffer)
        , m_OutMessageBufferSize(outMessageBufferSize)
        , m_PointerBuffer(pointerBuffer)
        , m_PointerBufferSize(pointerBufferSize)
        , m_HandlesToCloseAfterReplyCount(0)
    {
    }

    virtual nn::Result PrepareForProcess(const cmif::CmifMessageMetaInfo& metaInfo) NN_NOEXCEPT NN_OVERRIDE
    {
        size_t inRawSize = 0;

        inRawSize += metaInfo.inRawDataSize + 16;

        inRawSize = util::align_up(inRawSize, sizeof(uint16_t));
        auto unfixedSizedPointerOutOffset = inRawSize;
        m_UnfixedSizedPointerOutSizeInfos = reinterpret_cast<uint16_t*>(m_Reader.GetRawPointer() + unfixedSizedPointerOutOffset);

        auto copyCount = 0;
        auto moveCount = 0;
        for (int i = 0; i < metaInfo.inNativeHandleCount; ++i)
        {
            auto attribute = metaInfo.inNativeHandleAttributes[i];
            if (attribute & cmif::NativeHandleAttribute_HipcCopy)
            {
                ++copyCount;
            }
            else if (attribute & cmif::NativeHandleAttribute_HipcMove)
            {
                ++moveCount;
            }
            else
            {
                NN_RESULT_THROW(hipc::ResultNotSupported());
            }
        }
        auto sendCount = 0;
        auto receiveCount = 0;
        auto pointerInCount = 0;
        auto pointerOutCount = 0;
        auto unfixedSizedPointerOutCount = 0;
        for (int i = 0; i < metaInfo.bufferCount; ++i)
        {
            auto attribute = metaInfo.bufferAttributes[i];
            if (attribute & cmif::BufferAttribute_HipcMapAlias)
            {
                m_IsMapAliasBuffers[i] = true;
                if (attribute & cmif::BufferAttribute_In)
                {
                    ++sendCount;
                    continue;
                }
                else if (attribute & cmif::BufferAttribute_Out)
                {
                    ++receiveCount;
                    continue;
                }
            }
            else if (attribute & cmif::BufferAttribute_HipcPointer)
            {
                m_IsMapAliasBuffers[i] = false;
                if (attribute & cmif::BufferAttribute_In)
                {
                    ++pointerInCount;
                    continue;
                }
                else if (attribute & cmif::BufferAttribute_Out)
                {
                    ++pointerOutCount;
                    if (!(attribute & cmif::BufferAttribute_FixedSize))
                    {
                        ++unfixedSizedPointerOutCount;
                    }
                    continue;
                }
            }
            else if (attribute & cmif::BufferAttribute_HipcAutoSelect)
            {
                if (attribute & cmif::BufferAttribute_In)
                {
                    auto mapPas = m_Reader.GetSend(sendCount);
                    m_IsMapAliasBuffers[i] = mapPas.pointer != 0;
                    ++sendCount;
                    ++pointerInCount;
                    continue;
                }
                else if (attribute & cmif::BufferAttribute_Out)
                {
                    auto mapPas = m_Reader.GetReceive(receiveCount);
                    m_IsMapAliasBuffers[i] = mapPas.pointer != 0;
                    ++receiveCount;
                    ++pointerOutCount;
                    if (!(attribute & cmif::BufferAttribute_FixedSize))
                    {
                        ++unfixedSizedPointerOutCount;
                    }
                    continue;
                }
            }
            NN_RESULT_THROW(hipc::ResultNotSupported());
        }

        inRawSize += unfixedSizedPointerOutCount * sizeof(uint16_t);
        inRawSize = util::align_up(inRawSize, sizeof(uint32_t));

        const auto& info = m_Reader.GetHeaderInfo().baseInfo;
        auto valid = true
            && info.hasPid == metaInfo.inProcessIdEnable
            && info.copyHandleCount == copyCount
            && info.moveHandleCount == moveCount
            && info.sendCount == sendCount
            && info.receiveCount == receiveCount
            && info.pointerCount == pointerInCount
            && info.exchangeCount == 0 // 未サポート
            // && info.receiveListCount == pointerOutCount 不要
            && info.rawDataByteSize >= inRawSize;
        NN_RESULT_THROW_UNLESS(valid, HandleRequestError(metaInfo));
        this->m_MetaInfo = metaInfo;
        this->m_PointerOutCount = pointerOutCount;
        NN_RESULT_SUCCESS;
    } // NOLINT[impl/function_size]

    static bool IsMapTransferAttributeAcceptable(uint32_t mapTransferAttribute, int bufferAttribute) NN_NOEXCEPT
    {
        return mapTransferAttribute == detail::GetMapTransferAttribute(bufferAttribute);
    }

    virtual nn::Result GetBuffers(cmif::PointerAndSize* buffers) const NN_NOEXCEPT NN_OVERRIDE
    {
        auto pointerOutBufferHead = reinterpret_cast<uintptr_t>(m_PointerBuffer) + m_PointerBufferSize;
        auto pointerInBufferTail = reinterpret_cast<uintptr_t>(m_PointerBuffer);

        auto mapTransferAttributesAreValid = true;
        auto sendIndex = 0;
        auto receiveIndex = 0;
        auto pointerInIndex = 0;
        auto pointerOutIndex = 0;
        auto unfixedSizedPointerOutIndex = 0;
        for (int i = 0; i < m_MetaInfo.bufferCount; ++i)
        {
            auto attribute = m_MetaInfo.bufferAttributes[i];
            if (m_IsMapAliasBuffers[i])
            {
                if (attribute & cmif::BufferAttribute_In)
                {
                    auto mapBufferInfo = m_Reader.GetSend(sendIndex);
                    buffers[i] = hipc::detail::UnregisterAddress(mapBufferInfo.ToPointerAndSize());
                    ++sendIndex;
                    if (attribute & cmif::BufferAttribute_HipcAutoSelect)
                    {
                        hipc::detail::UnregisterAddress(m_Reader.GetPointer(pointerInIndex));
                        ++pointerInIndex;
                    }
                    if (!IsMapTransferAttributeAcceptable(mapBufferInfo.mapTransferAttribute, attribute))
                    {
                        mapTransferAttributesAreValid = false;
                    }
                    continue;
                }
                else if (attribute & cmif::BufferAttribute_Out)
                {
                    auto mapBufferInfo = m_Reader.GetReceive(receiveIndex);
                    buffers[i] = hipc::detail::UnregisterAddress(mapBufferInfo.ToPointerAndSize());
                    ++receiveIndex;
                    if (attribute & cmif::BufferAttribute_HipcAutoSelect)
                    {
                        ++pointerOutIndex;
                        if (!(attribute & cmif::BufferAttribute_FixedSize))
                        {
                            ++unfixedSizedPointerOutIndex;
                        }
                    }
                    if (!IsMapTransferAttributeAcceptable(mapBufferInfo.mapTransferAttribute, attribute))
                    {
                        mapTransferAttributesAreValid = false;
                    }
                    continue;
                }
            }
            else
            {
                if (attribute & cmif::BufferAttribute_In)
                {
                    buffers[i] = hipc::detail::UnregisterAddress(m_Reader.GetPointer(pointerInIndex));
                    ++pointerInIndex;
                    auto pointerInTail = buffers[i].pointer + buffers[i].size;
                    if (pointerInBufferTail < pointerInTail && buffers[i].size > 0)
                    {
                        pointerInBufferTail = pointerInTail;
                    }
                    if (attribute & cmif::BufferAttribute_HipcAutoSelect)
                    {
                        hipc::detail::UnregisterAddress(m_Reader.GetSend(sendIndex).ToPointerAndSize());
                        ++sendIndex;
                    }
                    continue;
                }
                else if (attribute & cmif::BufferAttribute_Out)
                {
                    ++pointerOutIndex;
                    size_t pointerOutSize;
                    if (!(attribute & cmif::BufferAttribute_FixedSize))
                    {
                        uint16_t unfixedSizedPointerOutSize;
                        std::memcpy(&unfixedSizedPointerOutSize, &m_UnfixedSizedPointerOutSizeInfos[unfixedSizedPointerOutIndex], sizeof(unfixedSizedPointerOutSize));
                        pointerOutSize = unfixedSizedPointerOutSize;
                        buffers[i].size = pointerOutSize;
                        ++unfixedSizedPointerOutIndex;
                    }
                    else
                    {
                        pointerOutSize = buffers[i].size;
                    }
                    // pointer out 用のバッファアロケートは末尾から行う
                    pointerOutBufferHead -= pointerOutSize;
                    pointerOutBufferHead = util::align_down(pointerOutBufferHead, 16);
                    buffers[i].pointer = pointerOutBufferHead;
                    if (attribute & cmif::BufferAttribute_HipcAutoSelect)
                    {
                        hipc::detail::UnregisterAddress(m_Reader.GetReceive(receiveIndex).ToPointerAndSize());
                        ++receiveIndex;
                    }
                    continue;
                }
            }
        }
        NN_RESULT_THROW_UNLESS(mapTransferAttributesAreValid, sf::hipc::ResultInvalidCmifRequest());
        NN_RESULT_THROW_UNLESS(pointerInBufferTail <= pointerOutBufferHead, sf::hipc::ResultInsufficientPointerTransferBuffer());
        NN_RESULT_SUCCESS;
    }

    virtual nn::Result OverwriteClientProcessId(nn::Bit64 *pInOutPid) const NN_NOEXCEPT NN_OVERRIDE
    {
        NN_SDK_REQUIRES(m_MetaInfo.inProcessIdEnable);
        // MEMO: 入力 PID の検証は、 SF ユーザの利便性と効率化を考え行わない。
        *pInOutPid = m_Reader.GetProcessId(); // HIPC ではどんな PID が渡されても IPC ヘッダから PID を取得する。
        NN_RESULT_SUCCESS;
    }

    virtual nn::Result GetInNativeHandles(nn::sf::NativeHandle handles[]) const NN_NOEXCEPT NN_OVERRIDE
    {
        auto copyIndex = 0;
        auto moveIndex = 0;
        for (int i = 0; i < m_MetaInfo.inNativeHandleCount; ++i)
        {
            auto&& handle = handles[i];
            auto attribute = m_MetaInfo.inNativeHandleAttributes[i];
            if (attribute & cmif::NativeHandleAttribute_HipcCopy)
            {
                auto internalHandle = m_Reader.GetCopyHandle(copyIndex);
                handle = NativeHandle(hipc::detail::UnregisterOsHandle(internalHandle), true);
                ++copyIndex;
            }
            else if (attribute & cmif::NativeHandleAttribute_HipcMove)
            {
                auto internalHandle = m_Reader.GetMoveHandle(moveIndex);
                handle = NativeHandle(hipc::detail::UnregisterOsHandle(internalHandle), true);
                ++moveIndex;
            }
            else
            {
                NN_ABORT("[SF-CMIF-Unexpected]");
            }
        }
        NN_RESULT_SUCCESS;
    }

    virtual nn::Result GetInObjects(nn::sf::cmif::server::CmifServerObjectInfo* inObjects) const NN_NOEXCEPT NN_OVERRIDE
    {
        NN_UNUSED(inObjects);
        return ResultNotSupported();
    }

    virtual void BeginPreparingForReply(nn::sf::cmif::PointerAndSize* pOutRawData) NN_NOEXCEPT NN_OVERRIDE
    {
        auto copyCount = 0;
        auto moveCount = 0;
        for (int i = 0; i < m_MetaInfo.outNativeHandleCount; ++i)
        {
            auto attribute = m_MetaInfo.outNativeHandleAttributes[i];
            if (attribute & cmif::NativeHandleAttribute_HipcCopy)
            {
                ++copyCount;
            }
            else if (attribute & cmif::NativeHandleAttribute_HipcMove)
            {
                ++moveCount;
            }
            else
            {
                NN_ABORT("[SF-CMIF-Unexpected]");
            }
        }
        hipc::detail::HipcMessageHeaderInfo headerInfo = {
            false, // bool hasPid;
            copyCount, // int copyHandleCount;
            m_MetaInfo.outObjectCount + moveCount, // int moveHandleCount;
            m_PointerOutCount, // int pointerCount;
            0, // int sendCount;
            0, // int receiveCount;
            0, // int exchangeCount;
            util::align_up(m_MetaInfo.outRawDataSize, sizeof(uint32_t)) + 16, // int rawDataByteSize;
            0, // int receiveListCount;
            detail::HipcMessageReceiveBufferMode_None,
        };

        this->m_MessageDataInfo = hipc::detail::MakeHipcMessageDataInfo(headerInfo);
        NN_ABORT_UNLESS(m_MessageDataInfo.valid, "[SF-CMIF-Unexpected]");
        hipc::detail::HipcMessageWriter writer(m_MessageDataInfo, m_OutMessageBuffer, m_OutMessageBufferSize);
        writer.SetHeader(0);

        pOutRawData->pointer = nn::util::align_up(reinterpret_cast<uintptr_t>(writer.GetRawPointer()), 16);
        pOutRawData->size = m_MetaInfo.outRawDataSize;
    }

    virtual void BeginPreparingForErrorReply(nn::sf::cmif::PointerAndSize* pOutRawData, size_t outRawSize) NN_NOEXCEPT NN_OVERRIDE
    {
        hipc::detail::HipcMessageHeaderInfo headerInfo = {};
        headerInfo.rawDataByteSize = util::align_up(outRawSize, sizeof(uint32_t)) + 16;

        this->m_MessageDataInfo = hipc::detail::MakeHipcMessageDataInfo(headerInfo);
        NN_ABORT_UNLESS(m_MessageDataInfo.valid, "[SF-CMIF-Unexpected]");
        hipc::detail::HipcMessageWriter writer(m_MessageDataInfo, m_OutMessageBuffer, m_OutMessageBufferSize);
        writer.SetHeader(0);

        pOutRawData->pointer = nn::util::align_up(reinterpret_cast<uintptr_t>(writer.GetRawPointer()), 16);
        pOutRawData->size = outRawSize;
    }

    virtual void SetBuffers(cmif::PointerAndSize buffers[]) NN_NOEXCEPT NN_OVERRIDE
    {
        hipc::detail::HipcMessageWriter writer(m_MessageDataInfo, m_OutMessageBuffer, m_OutMessageBufferSize);
        auto pointerOutIndex = 0;
        for (int i = 0; i < m_MetaInfo.bufferCount; ++i)
        {
            auto attribute = m_MetaInfo.bufferAttributes[i];
            auto&& buffer = buffers[i];
            if (attribute & cmif::BufferAttribute_HipcPointer)
            {
                if (attribute & cmif::BufferAttribute_Out)
                {
                    writer.SetPointer(pointerOutIndex, hipc::detail::RegisterAddress(buffer.pointer), buffer.size, pointerOutIndex);
                    ++pointerOutIndex;
                }
            }
            else if (attribute & cmif::BufferAttribute_HipcAutoSelect)
            {
                if (attribute & cmif::BufferAttribute_Out)
                {
                    if (!m_IsMapAliasBuffers[i])
                    {
                        writer.SetPointer(pointerOutIndex, hipc::detail::RegisterAddress(buffer.pointer), buffer.size, pointerOutIndex);
                    }
                    else
                    {
                        writer.SetPointer(pointerOutIndex, 0, 0, pointerOutIndex);
                    }
                    ++pointerOutIndex;
                }
            }
        }
    }

    void SetOutObject(int i, cmif::server::CmifServerObjectInfo&& outObject) NN_NOEXCEPT
    {
        hipc::detail::HipcMessageWriter writer(m_MessageDataInfo, m_OutMessageBuffer, m_OutMessageBufferSize);
        if (!outObject)
        {
            writer.SetMoveHandle(i, detail::InvalidInternalHandleValue);
            return;
        }
        HipcServerSessionHandle serverSessionHandle;
        HipcClientSessionHandle clientSessionHandle;
        // TODO: PrepareForProcess に移動を検討
        NN_ABORT_UNLESS_RESULT_SUCCESS(HipcServerApiModelHolder::CreateSession(&serverSessionHandle, &clientSessionHandle));
        NN_ABORT_UNLESS_RESULT_SUCCESS(m_pDomain->Register(serverSessionHandle, std::move(outObject)));
        auto internalHandle = hipc::detail::RegisterSessionHandle(clientSessionHandle);
        writer.SetMoveHandle(i, internalHandle);
    }

    virtual void SetOutObjects(cmif::server::CmifServerObjectInfo* outObjects) NN_NOEXCEPT NN_OVERRIDE
    {
        for (int i = 0; i < m_MetaInfo.outObjectCount; i++)
        {
            SetOutObject(i, std::move(outObjects[i]));
        }
    }

    virtual void SetOutNativeHandles(nn::sf::NativeHandle outNativeHandles[]) NN_NOEXCEPT NN_OVERRIDE
    {
        hipc::detail::HipcMessageWriter writer(m_MessageDataInfo, m_OutMessageBuffer, m_OutMessageBufferSize);
        auto copyIndex = 0;
        auto moveIndex = 0;
        auto attributes = m_MetaInfo.outNativeHandleAttributes;
        for (int i = 0; i < m_MetaInfo.outNativeHandleCount; ++i)
        {
            auto attribute = attributes[i];
            auto&& handle = outNativeHandles[i];
            auto internalHandle = hipc::detail::RegisterOsHandle(handle.GetOsHandle());
            if (attribute & cmif::NativeHandleAttribute_HipcCopy)
            {
                if (handle.IsManaged())
                {
                    m_HandlesToCloseAfterReply[m_HandlesToCloseAfterReplyCount] = handle.GetOsHandle();
                    ++m_HandlesToCloseAfterReplyCount;
                }
                writer.SetCopyHandle(copyIndex, internalHandle);
                handle.Detach();
                ++copyIndex;
            }
            else if (attribute & cmif::NativeHandleAttribute_HipcMove)
            {
                NN_ABORT_UNLESS(handle.IsManaged(), "[SF-HIPC-InvalidHandle:ShouldBeManaged]");
                writer.SetMoveHandle(m_MetaInfo.outObjectCount + moveIndex, internalHandle);
                handle.Detach();
                ++moveIndex;
            }
            else
            {
                NN_ABORT("[SF-CMIF-Unexpected]");
            }
        }
    }

    void AfterReply() NN_NOEXCEPT
    {
        for (auto i = 0; i < m_HandlesToCloseAfterReplyCount; ++i)
        {
            // TODO: Win ではセッションハンドルをこの関数では破棄できないため、正しく動かない。
            //       ただし、現状、そのようなシーケンスがないため、とりあえずこのままにしておく。
            os::CloseNativeHandle(m_HandlesToCloseAfterReply[i]);
        }
    }

private:

    IHipcObjectDomain* m_pDomain;

    // request
    const hipc::detail::HipcMessageReader& m_Reader;

    // reply
    hipc::detail::HipcMessageDataInfo m_MessageDataInfo;
    void* m_OutMessageBuffer;
    size_t m_OutMessageBufferSize;

    cmif::CmifMessageMetaInfo m_MetaInfo;

    void* m_PointerBuffer;
    size_t m_PointerBufferSize;
    uint16_t* m_UnfixedSizedPointerOutSizeInfos;
    int m_PointerOutCount;
    util::BitFlagSet<sf::detail::ArgumentCountMax> m_IsMapAliasBuffers;

    int m_HandlesToCloseAfterReplyCount;
    os::NativeHandle m_HandlesToCloseAfterReply[8];

    Result HandleRequestError(const cmif::CmifMessageMetaInfo& metaInfo) NN_NOEXCEPT
    {
        // 必要に応じてより詳細な Result を返す
        NN_UNUSED(metaInfo);
        NN_RESULT_THROW(hipc::ResultInvalidCmifRequest());
    };

};

}}}}
