﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/sf/hipc/server/sf_HipcAllInOneServerManager.h>

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/nn_Abort.h>
#include <nn/nn_SdkAssert.h>
#include <nn/result/result_HandlingUtility.h>

#include <utility>
#include <mutex>
#include <nn/os/os_MultipleWaitApi.h>
#include <nn/sf/hipc/sf_HipcDirectApi.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_Exchange.h>
#include <nn/sf/hipc/sf_HipcServiceResolutionApi.h>
#include <nn/sf/detail/sf_InternalResult.h>
#include <cstring>

namespace nn { namespace sf { namespace hipc { namespace server { inline namespace v1 {

namespace detail {

HipcAllInOneServerManagerBase::HipcAllInOneServerManagerBase(DomainEntryStorage* entryBuffer, int entryBufferCount) NN_NOEXCEPT
    : HipcServerSessionManagerWithDomain(entryBuffer, entryBufferCount)
    , m_StopRequestEvent(nn::os::EventClearMode_ManualClear)
    , m_NotificationEvent(nn::os::EventClearMode_ManualClear)
{
    os::InitializeMultiWait(&m_MultiWait);
    os::InitializeMultiWaitHolder(&m_StopRequestEventHolder, m_StopRequestEvent.GetBase());
    os::LinkMultiWaitHolder(&m_MultiWait, &m_StopRequestEventHolder);
    os::InitializeMultiWaitHolder(&m_NotificationEventHolder, m_NotificationEvent.GetBase());
    os::LinkMultiWaitHolder(&m_MultiWait, &m_NotificationEventHolder);
    os::InitializeMultiWait(&m_DeferredList);
}

HipcAllInOneServerManagerBase::~HipcAllInOneServerManagerBase() NN_NOEXCEPT
{
    for (auto&& port : m_AllPortList)
    {
        os::UnlinkMultiWaitHolder(&port);
        os::FinalizeMultiWaitHolder(&port);
        if (port.m_ServiceNameManaged)
        {
            HipcServerApiModelHolder::UnregisterHipcService(port.m_ServiceName);
            hipc::FinalizeHipcServiceResolution();
            hipc::CloseServerPortHandle(port.m_PortHandle);
        }
    }
    os::UnlinkMultiWaitHolder(&m_NotificationEventHolder);
    os::FinalizeMultiWaitHolder(&m_NotificationEventHolder);
    os::UnlinkMultiWaitHolder(&m_StopRequestEventHolder);
    os::FinalizeMultiWaitHolder(&m_StopRequestEventHolder);
    os::UnlinkAllMultiWaitHolder(&m_MultiWait);
    os::FinalizeMultiWait(&m_DeferredList);
    os::FinalizeMultiWait(&m_MultiWait);
}

void HipcAllInOneServerManagerBase::RegisterPortImpl(PortForAllInOne* pPort, HipcServerPortHandle portHandle) NN_NOEXCEPT
{
    pPort->m_PortHandle = portHandle;
    hipc::AttachWaitHolderForAccept(pPort, portHandle);
    os::SetMultiWaitHolderUserData(pPort, PortTag);
    os::LinkMultiWaitHolder(&m_MultiWait, pPort);
    m_AllPortList.push_back(*pPort);
}

void HipcAllInOneServerManagerBase::InitializePort(int portIndex, HipcServerPortHandle portHandle) NN_NOEXCEPT
{
    InitializePortImpl(portIndex, cmif::server::CmifServerObjectInfo(), portHandle);
}

Result HipcAllInOneServerManagerBase::InitializePort(int portIndex, int32_t maxSessions, const char* serviceName) NN_NOEXCEPT
{
    return InitializePortImpl(portIndex, cmif::server::CmifServerObjectInfo(), maxSessions, serviceName);
}

void HipcAllInOneServerManagerBase::InitializePortImpl(int portIndex, cmif::server::CmifServerObjectInfo&& x, HipcServerPortHandle portHandle) NN_NOEXCEPT
{
    auto pPort = AllocatePort();
    pPort->m_ServiceNameManaged = false;
    if (x)
    {
        pPort->m_Object = std::move(x);
    }
    else
    {
        pPort->m_Index = portIndex;
    }
    RegisterPortImpl(pPort, portHandle);
}

Result HipcAllInOneServerManagerBase::InitializePortImpl(int portIndex, cmif::server::CmifServerObjectInfo&& x, int32_t maxSessions, const char* serviceName) NN_NOEXCEPT
{
    hipc::InitializeHipcServiceResolution();
    auto success = false;
    NN_UTIL_SCOPE_EXIT
    {
        if (!success)
        {
            hipc::FinalizeHipcServiceResolution();
        }
    };
    HipcServerPortHandle portHandle;
    NN_RESULT_DO(HipcServerApiModelHolder::RegisterHipcService(&portHandle, maxSessions, serviceName));
    auto pPort = AllocatePort();
    std::strcpy(pPort->m_ServiceName, serviceName);
    pPort->m_ServiceNameManaged = true;
    if (x)
    {
        pPort->m_Object = std::move(x);
    }
    else
    {
        pPort->m_Index = portIndex;
    }
    RegisterPortImpl(pPort, portHandle);
    success = true;
    NN_RESULT_SUCCESS;
}

void HipcAllInOneServerManagerBase::Start() NN_NOEXCEPT
{
    m_StopRequestEvent.Clear();
}

void HipcAllInOneServerManagerBase::RequestStop() NN_NOEXCEPT
{
    m_StopRequestEvent.Signal();
}

inline void HipcAllInOneServerManagerBase::LinkToDeferredList(os::MultiWaitHolderType * p) NN_NOEXCEPT
{
    std::lock_guard<decltype(m_DeferredListMutex)> lk(m_DeferredListMutex);
    os::LinkMultiWaitHolder(&m_DeferredList, p);
    m_NotificationEvent.Signal();
}

inline void HipcAllInOneServerManagerBase::LinkDeferred() NN_NOEXCEPT
{
    std::lock_guard<decltype(m_DeferredListMutex)> lk(m_DeferredListMutex);
    os::MoveAllMultiWaitHolder(&m_MultiWait, &m_DeferredList);
}

os::MultiWaitHolderType* HipcAllInOneServerManagerBase::WaitSignaled() NN_NOEXCEPT
{
    std::lock_guard<decltype(m_SelectMutex)> lk(m_SelectMutex);
    for (;;)
    {
        LinkDeferred();
        auto ret = os::WaitAny(&m_MultiWait);
        if (ret == &m_StopRequestEventHolder)
        {
            return nullptr;
        }
        if (!(ret == &m_NotificationEventHolder))
        {
            os::UnlinkMultiWaitHolder(ret);
            return ret;
        }
        m_NotificationEvent.Clear();
    }
}

Result HipcAllInOneServerManagerBase::ReceiveAsSession(os::MultiWaitHolderType* p) NN_NOEXCEPT
{
    auto messageBuffer = hipc::GetMessageBufferOnTls();
    auto messageBufferSize = hipc::MessageBufferSizeOnTls;
    return ReceiveAsSession(p, messageBuffer, messageBufferSize);
}

Result HipcAllInOneServerManagerBase::ReceiveAsSession(os::MultiWaitHolderType* p, void* messageBuffer, size_t messageBufferSize) NN_NOEXCEPT
{
    NN_ABORT_UNLESS(os::GetMultiWaitHolderUserData(p) == SessionTag, "[SF-HIPC-InvalidProcessTag:NeedsSession]");
    auto pSession = static_cast<SessionForAllInOne*>(p);
    NN_ABORT_UNLESS(!pSession->m_Received, "[SF-HIPC-InvalidSession:AlreadyReceived]");
    NN_RESULT_DO(this->ReceiveRequest(pSession, messageBuffer, messageBufferSize));
    pSession->m_Received = true;
    NN_RESULT_SUCCESS;
}

Result HipcAllInOneServerManagerBase::ProcessAsSession(os::MultiWaitHolderType* p) NN_NOEXCEPT
{
    auto messageBuffer = hipc::GetMessageBufferOnTls();
    auto messageBufferSize = hipc::MessageBufferSizeOnTls;
    return ProcessAsSession(p, messageBuffer, messageBufferSize);
}

Result HipcAllInOneServerManagerBase::ProcessInvokeRequest(os::MultiWaitHolderType* p) NN_NOEXCEPT
{
    NN_ABORT_UNLESS(os::GetMultiWaitHolderUserData(p) == SessionTag, "[SF-HIPC-InvalidProcessTag:NeedsSession]");
    auto pSession = static_cast<SessionForAllInOne*>(p);
    auto messageBuffer = hipc::GetMessageBufferOnTls();
    auto messageBufferSize = hipc::MessageBufferSizeOnTls;
    if (!pSession->m_Received)
    {
        NN_RESULT_DO(this->ReceiveRequest(pSession, messageBuffer, messageBufferSize));
        pSession->m_Received = true;
    }
    else
    {
        auto savedMessageBuffer = GetSavedMessageBuffer(messageBufferSize, pSession);
        NN_ABORT_UNLESS_NOT_NULL(savedMessageBuffer);
        std::memcpy(messageBuffer, savedMessageBuffer, messageBufferSize);
    }
    NN_SDK_ASSERT(pSession->m_Received);
    NN_RESULT_TRY(this->ProcessRequest(pSession, messageBuffer, messageBufferSize))
        NN_RESULT_CATCH(ResultProcessDeferred)
        {
            NN_ABORT("[SF-HIPC-InvalidUserImplementation:ProcessInvokeRequestWithDefer] Please call ProcessInvokeRequestWithDefer instead.");
        }
        NN_RESULT_CATCH(sf::detail::ResultCurrentContextInvalidated)
        {
            NN_ABORT("[SF-HIPC-InvalidUserImplementation:ProcessInvokeRequestWithDefer] Please call ProcessInvokeRequestWithDefer instead.");
        }
    NN_RESULT_END_TRY
    NN_RESULT_SUCCESS;
}

Result HipcAllInOneServerManagerBase::ProcessInvokeRequestWithDefer(os::MultiWaitHolderType* p) NN_NOEXCEPT
{
    NN_ABORT_UNLESS(os::GetMultiWaitHolderUserData(p) == SessionTag, "[SF-HIPC-InvalidProcessTag:NeedsSession]");
    auto pSession = static_cast<SessionForAllInOne*>(p);
    NN_UNUSED(pSession->m_SavedMessageBuffer); // 警告回避。メンバそのものは互換性維持のため残している。
    NN_UNUSED(pSession->m_SavedMessageBufferSize); // 警告回避。メンバそのものは互換性維持のため残している。
    auto messageBuffer = hipc::GetMessageBufferOnTls();
    auto messageBufferSize = hipc::MessageBufferSizeOnTls;
    auto savedMessageBuffer = GetSavedMessageBuffer(messageBufferSize, pSession);
    NN_ABORT_UNLESS_NOT_NULL(savedMessageBuffer);
    if (!pSession->m_Received)
    {
        NN_RESULT_DO(this->ReceiveRequest(pSession, messageBuffer, messageBufferSize));
        pSession->m_Received = true;
        std::memcpy(savedMessageBuffer, messageBuffer, messageBufferSize);
    }
    else
    {
        std::memcpy(messageBuffer, savedMessageBuffer, messageBufferSize);
    }
    NN_SDK_ASSERT(pSession->m_Received);
    NN_SDK_ASSERT_NOT_NULL(savedMessageBuffer);
    NN_SDK_ASSERT(std::memcmp(savedMessageBuffer, messageBuffer, messageBufferSize) == 0);
    NN_RESULT_TRY(this->ProcessRequest(pSession, messageBuffer, messageBufferSize))
        NN_RESULT_CATCH(sf::detail::ResultCurrentContextInvalidated)
        {
            // 処理の責任は別のコンテキストに移っているため、これ以上の処理は不要
            NN_RESULT_SUCCESS;
        }
    NN_RESULT_END_TRY
    NN_RESULT_SUCCESS;
}

Result HipcAllInOneServerManagerBase::ProcessAsSession(os::MultiWaitHolderType* p, void* messageBuffer, size_t messageBufferSize) NN_NOEXCEPT
{
    NN_ABORT_UNLESS(os::GetMultiWaitHolderUserData(p) == SessionTag, "[SF-HIPC-InvalidProcessTag:NeedsSession]");
    auto pSession = static_cast<SessionForAllInOne*>(p);
    if (!pSession->m_Received)
    {
        NN_RESULT_DO(this->ReceiveRequest(pSession, messageBuffer, messageBufferSize));
        pSession->m_Received = true;
    }
    NN_RESULT_DO(this->ProcessRequest(pSession, messageBuffer, messageBufferSize));
    NN_RESULT_SUCCESS;
}

Result HipcAllInOneServerManagerBase::ProcessAsPort(os::MultiWaitHolderType* p) NN_NOEXCEPT
{
    NN_ABORT_UNLESS(os::GetMultiWaitHolderUserData(p) == PortTag, "[SF-HIPC-InvalidProcessTag:NeedsPort]");
    auto pPort = static_cast<PortForAllInOne*>(p);
    NN_UTIL_SCOPE_EXIT
    {
        LinkToDeferredList(p);
    };
    if (pPort->m_Object)
    {
        return this->Accept(pPort->m_PortHandle, pPort->m_Object.Clone());
    }
    else
    {
        return OnNeedsToAccept(pPort->m_Index, pPort);
    }
}

void HipcAllInOneServerManagerBase::RegisterServerSessionToWait(HipcServerSession* pSession) NN_NOEXCEPT
{
    auto p = static_cast<SessionForAllInOne*>(pSession);
    os::SetMultiWaitHolderUserData(p, SessionTag);
    p->m_Received = false;
    LinkToDeferredList(p);
}

// utility
Result HipcAllInOneServerManagerBase::ProcessAuto(os::MultiWaitHolderType* p) NN_NOEXCEPT
{
    auto tag = os::GetMultiWaitHolderUserData(p);
    if (tag == SessionTag)
    {
        return ProcessAsSession(p);
    }
    if (tag == PortTag)
    {
        return ProcessAsPort(p);
    }
    NN_ABORT("[SF-HIPC-InvalidProcessTag]");
}

// utility
bool HipcAllInOneServerManagerBase::WaitAndProcessAutoImpl() NN_NOEXCEPT
{
    auto p = WaitSignaled();
    if (!p)
    {
        return false;
    }
    auto result = ProcessAuto(p);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    return true;
}

// utility
void HipcAllInOneServerManagerBase::WaitAndProcessAuto() NN_NOEXCEPT
{
    WaitAndProcessAutoImpl();
}

// utility
void HipcAllInOneServerManagerBase::LoopAuto() NN_NOEXCEPT
{
    while (WaitAndProcessAutoImpl())
    {
    }
}

// ユーザシグナル
void HipcAllInOneServerManagerBase::AddUserWaitHolder(os::MultiWaitHolderType * p) NN_NOEXCEPT
{
    auto tag = os::GetMultiWaitHolderUserData(p);
    NN_UNUSED(tag);
    NN_SDK_REQUIRES(tag != SessionTag, "[SF-USER-PreConditionViolation] os::GetMultiWaitHolderUserData(p) must not equal SessionTag");
    NN_SDK_REQUIRES(tag != PortTag, "[SF-USER-PreConditionViolation] os::GetMultiWaitHolderUserData(p) must not equal AcceptTag");
    LinkToDeferredList(p);
}

}

}}}}}
