﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/sf/hipc/detail/sf_HipcMessageBufferAccessor.h>

#include <nn/nn_Common.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Result.h>
#include <nn/util/util_BitPack.h>
#include <nn/util/util_BitUtil.h>
#include <nn/os/os_NativeHandle.h>
#include <nn/sf/detail/sf_CommonUtil.h>

#include <climits>
#include <cstring>
#include <algorithm>

namespace nn { namespace sf { namespace hipc { namespace detail {

namespace
{

const size_t IpcMessageChunkSize = sizeof(uint32_t);
const size_t PidSize = 2 * IpcMessageChunkSize;
const size_t HandleDataSize = sizeof(InternalHandleValue);
const size_t PointerDataSize = 2 * IpcMessageChunkSize;
const size_t MapDataSize = 3 * IpcMessageChunkSize;
const size_t ReceiveListEntrySize = 2 * IpcMessageChunkSize;

//
typedef nn::util::BitPack32::Field<0,                   16, Bit16>  Tag;
typedef nn::util::BitPack32::Field<Tag::Next,           4,  int>    PointerNum;
typedef nn::util::BitPack32::Field<PointerNum::Next,    4,  int>    SendNum;
typedef nn::util::BitPack32::Field<SendNum::Next,       4,  int>    ReceiveNum;
typedef nn::util::BitPack32::Field<ReceiveNum::Next,    4,  int>    ExchangeNum;

typedef nn::util::BitPack32::Field<0,                   10, int>    RawNum;
typedef nn::util::BitPack32::Field<RawNum::Next,        4,  int>    ReceiveListNum;
typedef nn::util::BitPack32::Field<31,                  1,  int>    SpecialNum;

// 特殊データ
typedef nn::util::BitPack32::Field<0,                   1,  bool>   Pid;
typedef nn::util::BitPack32::Field<Pid::Next,           4,  int>    CopyHandleNum;
typedef nn::util::BitPack32::Field<CopyHandleNum::Next, 4,  int>    MoveHandleNum;

// ポインタデータ
typedef nn::util::BitPack32::Field<0,                       4, int> PointerIndex;
typedef nn::util::BitPack32::Field<6,                       3, uint32_t> PointerAddress36;
typedef nn::util::BitPack32::Field<12,                      4, uint32_t> PointerAddress32;
typedef nn::util::BitPack32::Field<PointerAddress32::Next, 16, uint32_t> PointerSize;

// Map データ
typedef nn::util::BitPack32::Field<0,               2, uint32_t> MapTransferAttribute;
typedef nn::util::BitPack32::Field<2,               3, uint32_t> MapAddress36;
typedef nn::util::BitPack32::Field<24,              4, uint32_t> MapSizeHi;
typedef nn::util::BitPack32::Field<MapSizeHi::Next, 4, uint32_t> MapAddress32;

// 受信指定リスト
typedef nn::util::BitPack32::Field<0,   7,  uint32_t> ReceiveAddressHi;
typedef nn::util::BitPack32::Field<16,  16, uint32_t> ReceiveSize;

}

Bit16 HipcMessageBufferAccessor::GetTag(void *messageBuffer, size_t messageBufferSize) NN_NOEXCEPT
{
    auto headerHead = reinterpret_cast<nn::util::BitPack32*>(messageBuffer);
    NN_ABORT_UNLESS(sizeof(headerHead[0]) < messageBufferSize);
    return headerHead[0].Get<Tag>();
}

bool HipcMessageBufferAccessor::SetHeadPointers(const HipcMessageHeaderInfo& headerInfo) NN_NOEXCEPT
{
    auto p = m_MessageBuffer;

    this->m_HeaderHead = reinterpret_cast<nn::util::BitPack32*>(p);
    p += 8;

    if (headerInfo.hasPid || headerInfo.copyHandleCount > 0 || headerInfo.moveHandleCount > 0)
    {
        this->m_SpecialHeaderHead = reinterpret_cast<nn::util::BitPack32*>(p);
        p += 4;

        if (headerInfo.hasPid)
        {
            this->m_PidHead = reinterpret_cast<nn::util::BitPack32*>(p);
            p += PidSize;
        }

        if (headerInfo.copyHandleCount > 0)
        {
            this->m_CopyHandleHead = reinterpret_cast<nn::util::BitPack32*>(p);
            p += HandleDataSize * headerInfo.copyHandleCount;
        }

        if (headerInfo.moveHandleCount > 0)
        {
            this->m_MoveHandleHead = reinterpret_cast<nn::util::BitPack32*>(p);
            p += HandleDataSize * headerInfo.moveHandleCount;
        }
    }
    else
    {
        this->m_SpecialHeaderHead = nullptr;
    }

    if (headerInfo.pointerCount > 0)
    {
        this->m_PointerHead = reinterpret_cast<nn::util::BitPack32*>(p);
        p += PointerDataSize * headerInfo.pointerCount;
    }

    if (headerInfo.sendCount > 0)
    {
        this->m_SendHead = reinterpret_cast<nn::util::BitPack32*>(p);
        p += MapDataSize * headerInfo.sendCount;
    }

    if (headerInfo.receiveCount > 0)
    {
        this->m_ReceiveHead = reinterpret_cast<nn::util::BitPack32*>(p);
        p += MapDataSize * headerInfo.receiveCount;
    }

    if (headerInfo.exchangeCount > 0)
    {
        this->m_ExchangeHead = reinterpret_cast<nn::util::BitPack32*>(p);
        p += MapDataSize * headerInfo.exchangeCount;
    }

    NN_SDK_ASSERT(util::is_aligned(headerInfo.rawDataByteSize, sizeof(uint32_t)), "[SF-Internal]");
    if (headerInfo.rawDataByteSize > 0)
    {
        this->m_RawDataHead = p;
        p += headerInfo.rawDataByteSize;
    }

    if (headerInfo.receiveListCount > 0)
    {
        this->m_ReceiveListHead = reinterpret_cast<nn::util::BitPack32*>(p);
        p += ReceiveListEntrySize * headerInfo.receiveListCount;
    }

    auto messageSize = static_cast<size_t>(p - m_MessageBuffer);
    if (!(messageSize <= m_MessageBufferSize))
    {
        return false;
    }

    this->m_MessageSize = messageSize;
    return true;
} // NOLINT(readability/fn_size)

bool HipcMessageBufferAccessor::SetupHeader(void *messageBuffer, size_t messageBufferSize, const HipcMessageHeaderInfo& headerInfo) NN_NOEXCEPT
{
    this->m_MessageBuffer = reinterpret_cast<char*>(messageBuffer);
    this->m_MessageBufferSize = messageBufferSize;

    if (!SetHeadPointers(headerInfo))
    {
        return false;
    }

    int receiveListCountBits;
    switch (headerInfo.receiveBufferMode)
    {
        case HipcMessageReceiveBufferMode_None:
        {
            NN_ABORT_UNLESS(headerInfo.receiveListCount == 0);
            receiveListCountBits = 0;
            break;
        }
        case HipcMessageReceiveBufferMode_MessageBuffer:
        {
            NN_ABORT_UNLESS(headerInfo.receiveListCount == 0);
            receiveListCountBits = 1;
            break;
        }
        case HipcMessageReceiveBufferMode_Single:
        {
            NN_ABORT_UNLESS(headerInfo.receiveListCount == 1);
            receiveListCountBits = 2;
            break;
        }
        case HipcMessageReceiveBufferMode_Multi:
        {
            NN_ABORT_UNLESS(headerInfo.receiveListCount <= 13);
            receiveListCountBits = headerInfo.receiveListCount > 0 ? headerInfo.receiveListCount + 2 : 0;

            break;
        }
        default: NN_UNEXPECTED_DEFAULT;
    }

    m_HeaderHead[0].storage = 0;
    m_HeaderHead[0].Set<PointerNum>(headerInfo.pointerCount);
    m_HeaderHead[0].Set<SendNum>(headerInfo.sendCount);
    m_HeaderHead[0].Set<ReceiveNum>(headerInfo.receiveCount);
    m_HeaderHead[0].Set<ExchangeNum>(headerInfo.exchangeCount);
    m_HeaderHead[1].storage = 0;
    auto rawNum = headerInfo.rawDataByteSize / sizeof(uint32_t);
    NN_SDK_ASSERT(rawNum <= INT_MAX, "[SF-Internal]");
    m_HeaderHead[1].Set<RawNum>(static_cast<int>(rawNum));
    m_HeaderHead[1].Set<ReceiveListNum>(receiveListCountBits);

    if (m_SpecialHeaderHead)
    {
        m_HeaderHead[1].Set<SpecialNum>(1);
        m_SpecialHeaderHead[0].storage = 0;
        m_SpecialHeaderHead[0].Set<Pid>(headerInfo.hasPid);
        m_SpecialHeaderHead[0].Set<CopyHandleNum>(headerInfo.copyHandleCount);
        m_SpecialHeaderHead[0].Set<MoveHandleNum>(headerInfo.moveHandleCount);
    }
    else
    {
        m_HeaderHead[1].Set<SpecialNum>(0);
    }

    return true;
}

bool HipcMessageBufferAccessor::ParseHeader(HipcMessageHeaderInfo* pHeaderInfo, void *messageBuffer, size_t messageBufferSize) NN_NOEXCEPT
{
    this->m_MessageBuffer = reinterpret_cast<char*>(messageBuffer);
    this->m_MessageBufferSize = messageBufferSize;

    auto& headerInfo = *pHeaderInfo;

    auto* headerHead = reinterpret_cast<util::BitPack32*>(m_MessageBuffer);
    headerInfo.pointerCount = headerHead[0].Get<PointerNum>();
    headerInfo.sendCount = headerHead[0].Get<SendNum>();
    headerInfo.receiveCount = headerHead[0].Get<ReceiveNum>();
    headerInfo.exchangeCount = headerHead[0].Get<ExchangeNum>();
    headerInfo.rawDataByteSize = headerHead[1].Get<RawNum>() * sizeof(uint32_t);
    headerInfo.receiveListCount = headerHead[1].Get<ReceiveListNum>();

    auto receiveListCountBits = headerHead[1].Get<ReceiveListNum>();
    switch (headerHead[1].Get<ReceiveListNum>())
    {
        case 0:
        {
            headerInfo.receiveBufferMode = HipcMessageReceiveBufferMode_None;
            headerInfo.receiveListCount = 0;
            break;
        }
        case 1:
        {
            headerInfo.receiveBufferMode = HipcMessageReceiveBufferMode_MessageBuffer;
            headerInfo.receiveListCount = 0;
            break;
        }
        case 2:
        {
            headerInfo.receiveBufferMode = HipcMessageReceiveBufferMode_Single;
            headerInfo.receiveListCount = 1;
            break;
        }
        default:
        {
            NN_ABORT_UNLESS(receiveListCountBits <= 15);
            headerInfo.receiveBufferMode = HipcMessageReceiveBufferMode_Multi;
            headerInfo.receiveListCount = receiveListCountBits - 2;
            break;
        }
    }

    if (headerHead[1].Get<SpecialNum>() > 0)
    {
        auto* specialHeaderHead = reinterpret_cast<util::BitPack32*>(m_MessageBuffer + 8);
        headerInfo.hasPid = specialHeaderHead[0].Get<Pid>();
        headerInfo.copyHandleCount = specialHeaderHead[0].Get<CopyHandleNum>();
        headerInfo.moveHandleCount = specialHeaderHead[0].Get<MoveHandleNum>();
    }
    else
    {
        headerInfo.hasPid = false;
        headerInfo.copyHandleCount = 0;
        headerInfo.moveHandleCount = 0;
    }

    return SetHeadPointers(headerInfo);
}

void HipcMessageBufferAccessor::SetTag(Bit16 tag) NN_NOEXCEPT
{
    m_HeaderHead[0].Set<Tag>(tag);
}

Bit16 HipcMessageBufferAccessor::GetTag() const NN_NOEXCEPT
{
    return m_HeaderHead[0].Get<Tag>();
}

void HipcMessageBufferAccessor::SetProcessId(uint64_t processId) NN_NOEXCEPT
{
    std::memcpy(m_PidHead, &processId, sizeof(processId));
}

uint64_t HipcMessageBufferAccessor::GetProcessId() const NN_NOEXCEPT
{
    uint64_t processId;
    std::memcpy(&processId, m_PidHead, sizeof(processId));
    return processId;
}

void HipcMessageBufferAccessor::SetCopyHandle(int index, InternalHandleValue handle) NN_NOEXCEPT
{
    std::memcpy(m_CopyHandleHead + index * 1, &handle, sizeof(handle));
}

InternalHandleValue HipcMessageBufferAccessor::GetCopyHandle(int index) const NN_NOEXCEPT
{
    InternalHandleValue handle;
    std::memcpy(&handle, m_CopyHandleHead + index * 1, sizeof(handle));
    return handle;
}

void HipcMessageBufferAccessor::SetMoveHandle(int index, InternalHandleValue handle) NN_NOEXCEPT
{
    std::memcpy(m_MoveHandleHead + index * 1, &handle, sizeof(handle));
}

InternalHandleValue HipcMessageBufferAccessor::GetMoveHandle(int index) const NN_NOEXCEPT
{
    InternalHandleValue handle;
    std::memcpy(&handle, m_MoveHandleHead + index * 1, sizeof(handle));
    return handle;
}

void HipcMessageBufferAccessor::SetPointer(int index, InternalAddressValue addressValue, size_t size, int receiveIndex) NN_NOEXCEPT
{
    auto data = m_PointerHead + index * 2;
    auto address64 = static_cast<uint64_t>(addressValue);
    data[0].storage = 0;
    data[0].Set<PointerIndex>(receiveIndex);
    data[0].Set<PointerAddress32>(sf::detail::ExtractBits<uint32_t>(address64, 32, PointerAddress32::Width));
    data[0].Set<PointerAddress36>(sf::detail::ExtractBits<uint32_t>(address64, 36, PointerAddress36::Width));
    NN_SDK_ASSERT(size <= UINT16_MAX, "[SF-Internal]");
    data[0].Set<PointerSize>(static_cast<uint32_t>(size));
    data[1].storage = static_cast<uint32_t>(address64 & 0xFFFFFFFF);
}

AddressValueAndSize HipcMessageBufferAccessor::GetPointer(int index) const NN_NOEXCEPT
{
    auto data = m_PointerHead + index * 2;
    uint64_t address32 = data[0].Get<PointerAddress32>();
    uint64_t address36 = data[0].Get<PointerAddress36>();
    auto size = data[0].Get<PointerSize>();
    uint64_t address0 = data[1].storage;
    auto address = static_cast<std::uintptr_t>(0
        | address36 << 36
        | address32 << 32
        | address0 << 0
    );
    AddressValueAndSize ret = { address, size };
    return ret;
}

int HipcMessageBufferAccessor::GetPointerReceiveIndex(int index) const NN_NOEXCEPT
{
    auto data = m_PointerHead + index * 2;
    return data[0].Get<PointerIndex>();
}

namespace {

inline void SetMapData(util::BitPack32* head, int index, InternalAddressValue addressValue, size_t size, uint32_t mapTransferAttribute) NN_NOEXCEPT
{
    auto data = head + index * 3;
    auto address64 = static_cast<uint64_t>(addressValue);
    auto size64 = static_cast<uint64_t>(size);
    auto sizeHi32 = static_cast<uint32_t>(size64 >> 32);
    auto sizeLow32 = static_cast<uint32_t>(size64 & 0xFFFFFFFF);
    data[0].storage = sizeLow32;
    data[1].storage = static_cast<uint32_t>(address64 & 0xFFFFFFFF);
    data[2].Set<MapTransferAttribute>(mapTransferAttribute);
    data[2].Set<MapAddress32>(sf::detail::ExtractBits<uint32_t>(address64, 32, MapAddress32::Width));
    data[2].Set<MapAddress36>(sf::detail::ExtractBits<uint32_t>(address64, 36, MapAddress36::Width));
    data[2].Set<MapSizeHi>(sizeHi32);
}

inline MapTransferBufferInfo GetMapData(const util::BitPack32* head, int index) NN_NOEXCEPT
{
    auto data = head + index * 3;
    uint64_t sizeLow32 = data[0].storage;
    uint64_t address0 = data[1].storage;
    uint64_t address32 = data[2].Get<MapAddress32>();
    uint64_t address36 = data[2].Get<MapAddress36>();
    uint64_t sizeHi32 = data[2].Get<MapSizeHi>();
    auto mapTransferAttribute = data[2].Get<MapTransferAttribute>();
    auto address = static_cast<std::uintptr_t>(0
        | address36 << 36
        | address32 << 32
        | address0 << 0
    );
    auto size = static_cast<size_t>(sizeHi32 << 32 | sizeLow32);
    MapTransferBufferInfo ret = { address, size, mapTransferAttribute };
    return ret;
}

}

void HipcMessageBufferAccessor::SetSend(int index, InternalAddressValue addressValue, size_t size, uint32_t mapTransferAttribute) NN_NOEXCEPT
{
    SetMapData(m_SendHead, index, addressValue, size, mapTransferAttribute);
}

MapTransferBufferInfo HipcMessageBufferAccessor::GetSend(int index) const NN_NOEXCEPT
{
    return GetMapData(m_SendHead, index);
}

void HipcMessageBufferAccessor::SetReceive(int index, InternalAddressValue addressValue, size_t size, uint32_t mapTransferAttribute) NN_NOEXCEPT
{
    SetMapData(m_ReceiveHead, index, addressValue, size, mapTransferAttribute);
}

MapTransferBufferInfo HipcMessageBufferAccessor::GetReceive(int index) const NN_NOEXCEPT
{
    return GetMapData(m_ReceiveHead, index);
}

void HipcMessageBufferAccessor::SetExchange(int index, InternalAddressValue addressValue, size_t size, uint32_t mapTransferAttribute) NN_NOEXCEPT
{
    SetMapData(m_ExchangeHead, index, addressValue, size, mapTransferAttribute);
}

MapTransferBufferInfo HipcMessageBufferAccessor::GetExchange(int index) const NN_NOEXCEPT
{
    return GetMapData(m_ExchangeHead, index);
}

uintptr_t HipcMessageBufferAccessor::GetRawPointer() const NN_NOEXCEPT
{
    return reinterpret_cast<uintptr_t>(m_RawDataHead);
}

void HipcMessageBufferAccessor::SetReceiveList(int index, InternalAddressValue addressValue, size_t size) NN_NOEXCEPT
{
    auto data = m_ReceiveListHead + index * 2;
    auto address64 = static_cast<uint64_t>(addressValue);
    auto addressHi32 = static_cast<uint32_t>(address64 >> 32);
    auto addressLow32 = static_cast<uint32_t>(address64 & 0xFFFFFFFF);
    data[0].storage = addressLow32;
    data[1].storage = 0;
    data[1].Set<ReceiveAddressHi>(addressHi32);
    NN_SDK_ASSERT(size <= UINT32_MAX, "[SF-Internal]");
    data[1].Set<ReceiveSize>(static_cast<uint32_t>(size));
}

AddressValueAndSize HipcMessageBufferAccessor::GetReceiveList(int index) const NN_NOEXCEPT
{
    auto data = m_ReceiveListHead + index * 2;
    uint64_t addressHi32 = data[1].Get<ReceiveAddressHi>();
    auto size = data[1].Get<ReceiveSize>();
    uint64_t addressLow32 = data[0].storage;
    auto address = static_cast<std::uintptr_t>(addressHi32 << 32 | addressLow32);
    AddressValueAndSize ret = { address, size };
    return ret;
}

}}}}
