﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#if (defined(NN_DETAIL_SDMMC_USE_NN_OS_FOR_TIMER))
    #include <nn/os.h>
#elif (defined(NN_DETAIL_SDMMC_USE_NN_UTIL_FOR_TIMER))
    #include <nn/util.h>
#endif

namespace nn { namespace sdmmc {
namespace detail {

void WaitMicroseconds(uint32_t us) NN_NOEXCEPT;
void WaitClocks(uint32_t numClocks, uint32_t clockFrequencyKHz) NN_NOEXCEPT;

#if (defined(NN_DETAIL_SDMMC_USE_NN_OS_FOR_TIMER))
    class ManualTimer
    {
    private:
        nn::os::Tick m_TimeoutTick;
        bool m_IsTimeout;

    public:
        explicit ManualTimer(uint32_t ms) NN_NOEXCEPT
        {
            m_TimeoutTick = nn::os::GetSystemTick() + nn::os::ConvertToTick(nn::TimeSpan::FromMilliSeconds(ms));
            m_IsTimeout = false;
        }

        bool Update() NN_NOEXCEPT
        {
            if (m_IsTimeout)
            {
                return false;   // すでにタイムアウトで更新できない
            }

            if (nn::os::GetSystemTick() > m_TimeoutTick)
            {
                m_IsTimeout = true;
            }
            else
            {
                m_IsTimeout = false;
            }
            return true;
        }
    };
#elif (defined(NN_DETAIL_SDMMC_USE_NN_UTIL_FOR_TIMER))
    class ManualTimer
    {
    private:
        uint32_t m_TimeoutMicroSeconds;
        bool m_IsTimeout;

    public:
        explicit ManualTimer(uint32_t ms) NN_NOEXCEPT
        {
            m_TimeoutMicroSeconds = nn::util::GetMicroSeconds() + (ms * 1000);
            m_IsTimeout = false;
        }

        bool Update() NN_NOEXCEPT
        {
            if (m_IsTimeout)
            {
                return false;   // すでにタイムアウトで更新できない
            }

            if (nn::util::GetMicroSeconds() > m_TimeoutMicroSeconds)
            {
                m_IsTimeout = true;
            }
            else
            {
                m_IsTimeout = false;
            }
            return true;
        }
    };
#else   // タイムアウト機能なし
    class ManualTimer
    {
    public:
        explicit ManualTimer(uint32_t ms) NN_NOEXCEPT
        {
            NN_UNUSED(ms);
        }

        bool Update() NN_NOEXCEPT
        {
            return true;
        }
    };
#endif

} // namespace detail {
}} // namespace nn { namespace sdmmc {
