﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "sdmmc_SdCardDeviceAccessor.h"
#include "sdmmc_Timer.h"
#include "sdmmc_Log.h"
#if (defined(NN_DETAIL_SDMMC_THREAD_SAFE_ENABLE))
    #include <mutex>
#endif
#include <nn/nn_Abort.h>

namespace nn { namespace sdmmc {
namespace detail {

#if (defined(NN_DETAIL_SDMMC_THREAD_SAFE_ENABLE))
    #define NN_DETAIL_SDMMC_DEVICE_LOCK_GUARD   std::lock_guard<nn::os::Mutex> lock(m_SdCardDevice.m_DeviceMutex)
#else
    #define NN_DETAIL_SDMMC_DEVICE_LOCK_GUARD
#endif

#if (defined(NN_DETAIL_SDMMC_SD_CARD_DETECTOR_ENABLE))
    #define NN_DETAIL_SDMMC_RESULT_THROW_IF_REMOVED     if (m_SdCardDevice.IsRemoved()) { return ResultDeviceRemoved(); }
#else
    #define NN_DETAIL_SDMMC_RESULT_THROW_IF_REMOVED
#endif

namespace
{
    // 許容できる消費電流 (mA), http://ird-mine.nintendo.co.jp/redmine/issues/26583
    const uint16_t AcceptableCurrentConsumption = 800;

    // CMD8 SEND_IF_COND の check pattern (任意の値)
    const uint8_t CheckPatternOfCommandSendIfCond = 0xAA;

    // OCR Register ビット定義
    const uint32_t OcrCardPowerUpStatus = 0x1U << 31;
    const uint32_t OcrCardCapacityStatus = 0x1U << 30;
    const uint32_t OcrSwitchingTo18VAccepted = 0x1U << 24;

    uint8_t GetSdBusWidths(uint8_t* pScr) NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_NOT_NULL(pScr);

        // SCR[51:48]: SD_BUS_WIDTHS
        uint8_t sdBusWidths = pScr[1] & 0xF;
        NN_DETAIL_SDMMC_DEBUG_LOG("SD_BUS_WIDTHS of SCR: 0x%X\n", sdBusWidths);
        return sdBusWidths;
    }

    bool IsSupportedBusWidth4bit(uint8_t sdBusWidths) NN_NOEXCEPT
    {
        if (sdBusWidths & 0x4)
        {
            NN_DETAIL_SDMMC_DEBUG_LOG("SD_BUS_WIDTHS indicates that 4 bit is supported\n");
            return true;
        }
        else
        {
            NN_DETAIL_SDMMC_DEBUG_LOG("SD_BUS_WIDTHS indicates that 4 bit is NOT supported\n");
            return false;
        }
    }

    bool IsLessThanSpecVersion1_10(uint8_t* pScr) NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_NOT_NULL(pScr);

        // SCR[59:56] = SD_SPEC
        uint8_t sdSpec = pScr[0] & 0xF;
        NN_DETAIL_SDMMC_DEBUG_LOG("SD_SPEC of SCR: 0x%X\n", sdSpec);
        if (sdSpec < 1)
        {
            NN_DETAIL_SDMMC_DEBUG_LOG("SD_SPEC is less than Version 1.10\n");
            return true;
        }
        else
        {
            NN_DETAIL_SDMMC_DEBUG_LOG("SD_SPEC indicates Version 1.10 or higher\n");
            return false;
        }
    }

    bool IsSupportedAccessMode(uint8_t* pSwitchFunctionStatus, SwitchFunctionAccessMode accessMode) NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_NOT_NULL(pSwitchFunctionStatus);

        // Switch Function Status[415:400] = Support Bits of Functions in Function Group 1
        NN_DETAIL_SDMMC_DEBUG_LOG("Switch Function Status[407:400]: 0x%X\n", pSwitchFunctionStatus[13]);
        if (pSwitchFunctionStatus[13] & (0x01U << accessMode))
        {
            return true;
        }
        else
        {
            return false;
        }
    }

    bool IsAccessModeOnFunctionSelection(uint8_t* pSwitchFunctionStatus, SwitchFunctionAccessMode accessMode)
    {
        NN_ABORT_UNLESS_NOT_NULL(pSwitchFunctionStatus);

        // Switch Function Status[379:376] = Function Selection of Function Group 1
        uint8_t selection = pSwitchFunctionStatus[16] & 0x0F;
        NN_DETAIL_SDMMC_DEBUG_LOG("Switch Function Status[379:376]: 0x%X\n", selection);
        if (selection == static_cast<uint8_t>(accessMode))
        {
            return true;
        }
        else
        {
            return false;
        }
    }

    bool CanBeSwitchedAccessMode(uint8_t* pSwitchFunctionStatus, SwitchFunctionAccessMode accessMode) NN_NOEXCEPT
    {
        return IsAccessModeOnFunctionSelection(pSwitchFunctionStatus, accessMode);
    }

    bool IsSuccessSwitchAccessMode(uint8_t* pSwitchFunctionStatus, SwitchFunctionAccessMode accessMode) NN_NOEXCEPT
    {
        // CanBeSwitchedAccessMode() と同じ判定
        return IsAccessModeOnFunctionSelection(pSwitchFunctionStatus, accessMode);
    }

    uint16_t GetMaximumCurrentConsumption(uint8_t* pSwitchFunctionStatus) NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_NOT_NULL(pSwitchFunctionStatus);

        // Switch Function Status[511:496] = Maximum Current/Power Consumption
        uint16_t maximumCurrentConsumption = pSwitchFunctionStatus[0];
        maximumCurrentConsumption = (maximumCurrentConsumption << 8) | pSwitchFunctionStatus[1];
        NN_DETAIL_SDMMC_DEBUG_LOG("Switch Function Status[511:496]: %u\n", maximumCurrentConsumption);
        return maximumCurrentConsumption;
    }

    Result GetCurrentSpeedMode(SpeedMode* pOutSpeedMode, uint8_t* pSwitchFunctionStatus, bool isUhsIMode) NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_NOT_NULL(pOutSpeedMode);
        NN_ABORT_UNLESS_NOT_NULL(pSwitchFunctionStatus);

        // Switch Function Status[379:376] = function group 1
        SwitchFunctionAccessMode accessMode = static_cast<SwitchFunctionAccessMode>(pSwitchFunctionStatus[16] & 0x0F);
        switch (accessMode)
        {
        case SwitchFunctionAccessMode_Default:
            if (isUhsIMode)
            {
                *pOutSpeedMode = SpeedMode_SdCardSdr12;
            }
            else
            {
                *pOutSpeedMode = SpeedMode_SdCardDefaultSpeed;
            }
            NN_RESULT_SUCCESS;
        case SwitchFunctionAccessMode_HighSpeed:
            if (isUhsIMode)
            {
                *pOutSpeedMode = SpeedMode_SdCardSdr25;
            }
            else
            {
                *pOutSpeedMode = SpeedMode_SdCardHighSpeed;
            }
            NN_RESULT_SUCCESS;
        case SwitchFunctionAccessMode_Sdr50:
            *pOutSpeedMode = SpeedMode_SdCardSdr50;
            NN_RESULT_SUCCESS;
        case SwitchFunctionAccessMode_Sdr104:
            *pOutSpeedMode = SpeedMode_SdCardSdr104;
            NN_RESULT_SUCCESS;
        case SwitchFunctionAccessMode_Ddr50:
            *pOutSpeedMode = SpeedMode_SdCardDdr50;
            NN_RESULT_SUCCESS;
        default:
            NN_DETAIL_SDMMC_ERROR_LOG("Unexpected Switch Function Status[379:376]: 0x%X\n", accessMode);
            return ResultUnexpectedSdCardSwitchFunctionStatus();
        }
    }

    bool IsLessThanCsdVersion2_0(uint8_t* pCsd)
    {
        NN_ABORT_UNLESS_NOT_NULL(pCsd);

        // CSD[127:126]: CSD_STRUCTURE
        uint8_t csdStructure = (pCsd[14] & 0xC0U) >> 6;
        NN_DETAIL_SDMMC_DEBUG_LOG("CSD_STRUCTURE: 0x%X\n", csdStructure);
        if (csdStructure == 0)
        {
            NN_DETAIL_SDMMC_DEBUG_LOG("CSD Version is 1.0\n");
            return true;
        }
        else
        {
            NN_DETAIL_SDMMC_DEBUG_LOG("CSD Version is 2.0 or higher\n");
            return false;
        }
    }

    uint32_t GetMemoryCapacity(uint16_t* pCsd) NN_NOEXCEPT
    {
        // pCsd[3]: CSD[71:56], pCsd[2]: CSD[55:40]
        // CSD[69:48]: C_SIZE の取得
        uint32_t cSize = ((pCsd[3] & 0x3FFFU) << 8) | ((pCsd[2] & 0xFF00U) >> 8);

        // 桁あふれしないように、べき乗をまとめて計算
        // memory capacity(byte) = (C_SIZE+1) * 512K byte;
        // memory capacity(sector) = (C_SIZE+1) * 1024;
        return (cSize + 1) << 10;
    }

    uint32_t GetSizeOfProtectedArea(uint8_t* pSdStatus) NN_NOEXCEPT
    {
        // SD Status[479:448]: SIZE_OF_PROTECTED_AREA
        return (static_cast<uint32_t>(pSdStatus[4]) << 24) | (static_cast<uint32_t>(pSdStatus[5]) << 16) | (static_cast<uint32_t>(pSdStatus[6]) << 8) | static_cast<uint32_t>(pSdStatus[7]);
    }
}

void SdCardDevice::SetOcrAndHighCapacity(uint32_t ocr) NN_NOEXCEPT
{
    BaseDevice::SetOcr(ocr);
    if ((ocr & OcrCardCapacityStatus) != 0)
    {
        BaseDevice::SetHighCapacity(true);
    }
    else
    {
        BaseDevice::SetHighCapacity(false);
    }
}

#if (defined(NN_DETAIL_SDMMC_SD_CARD_DETECTOR_ENABLE))
    void SdCardDeviceAccessor::RemovedCallback() NN_NOEXCEPT
    {
        // HostController を含め、現在動作している処理があれば中断を促す
        m_SdCardDevice.SignalRemovedEvent();

        // 現在動作している処理があれば終わるのを待つ
        NN_DETAIL_SDMMC_DEVICE_LOCK_GUARD;

        // HostController を落としておく（SD カード電源 OFF も含まれる）
        IHostController* pHostController = BaseDeviceAccessor::GetHostController();
        pHostController->Shutdown();
    }
#endif

Result SdCardDeviceAccessor::IssueCommandSendRelativeAddr(uint16_t* pOutRca) const NN_NOEXCEPT
{
    // argument[31:0] stuff bit
    ResponseType responseType = ResponseType_R6;
    Command command(3, 0, responseType, false);

    IHostController* pHostController = BaseDeviceAccessor::GetHostController();
    NN_RESULT_DO(pHostController->IssueCommand(&command));

    // 上位 16bit RCA, 下位 16bit Device Status (チェックしない)
    uint32_t response;
    pHostController->GetLastResponse(&response, sizeof(response), responseType);
    NN_ABORT_UNLESS_NOT_NULL(pOutRca);
    *pOutRca = static_cast<uint16_t>((response >> 16) & 0xFFFF);

    NN_RESULT_SUCCESS;
}

Result SdCardDeviceAccessor::IssueCommandSendIfCond() const NN_NOEXCEPT
{
    // argument[31:12] reserved bits : 0
    // argument[11: 8] supply voltage(VHS) : 0x01(2.7-3.6V)
    // argument[ 7: 0] check pattern : 任意の値
    uint32_t argument = (0x01U << 8) | static_cast<uint32_t>(CheckPatternOfCommandSendIfCond);
    ResponseType responseType = ResponseType_R7;
    Command command(8, argument, responseType, false);

    IHostController* pHostController = BaseDeviceAccessor::GetHostController();
    NN_RESULT_DO(pHostController->IssueCommand(&command));

    uint32_t response;
    pHostController->GetLastResponse(&response, sizeof(response), responseType);
    // Echo Back ビットを比較
    const uint32_t EchoBackBits = 0xFFFU;   // response[11: 0]
    if ((response & EchoBackBits) != (argument & EchoBackBits))
    {
        NN_DETAIL_SDMMC_ERROR_LOG("CMD8 Unexpected Response: 0x%04X (Expected Response: 0x%04X)\n", response, argument);
        return ResultSdCardValidationError();
    }

    NN_RESULT_SUCCESS;
}

Result SdCardDeviceAccessor::IssueCommandCheckSupportedFunction(void* pOutSwitchFunctionStatusBuffer, size_t switchFunctionStatusBufferSize) const NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(pOutSwitchFunctionStatusBuffer);
    NN_ABORT_UNLESS(switchFunctionStatusBufferSize >= SdCardSwitchFunctionStatusSize);

    IHostController* pHostController = BaseDeviceAccessor::GetHostController();

    // argument[31] Mode : 0(Check function), 1(Switch function)
    // argument[30:24] reserved (0)
    // argument[23:20] function group 6 : F(Current)
    // argument[19:16] function group 5 : F(Current)
    // argument[15:12] function group 4 : F(Current)
    // argument[11: 8] function group 3 : F(Current)
    // Argument[ 7: 4] function group 2 : F(Current)
    // Argument[ 3: 0] function group 1 : F(Current)
    ResponseType responseType = ResponseType_R1;
    Command command(6, 0x00FFFFFF, responseType, false);
    TransferData transferData(pOutSwitchFunctionStatusBuffer, SdCardSwitchFunctionStatusSize, 1, TransferDirection_ReadFromDevice);
    NN_RESULT_DO(pHostController->IssueCommand(&command, &transferData));

    uint32_t response;
    pHostController->GetLastResponse(&response, sizeof(response), responseType);
    NN_RESULT_DO(m_SdCardDevice.CheckDeviceStatus(response));

    NN_DETAIL_SDMMC_DEBUG_LOG("Switch Function Status:\n");
    NN_DETAIL_SDMMC_DATA_LOG(pOutSwitchFunctionStatusBuffer, SdCardSwitchFunctionStatusSize);

    NN_RESULT_SUCCESS;
}

Result SdCardDeviceAccessor::IssueCommandSwitchAccessMode(void* pOutSwitchFunctionStatusBuffer, size_t switchFunctionStatusBufferSize, bool isSetFunction, SwitchFunctionAccessMode accessMode) const NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(pOutSwitchFunctionStatusBuffer);
    NN_ABORT_UNLESS(switchFunctionStatusBufferSize >= SdCardSwitchFunctionStatusSize);

    IHostController* pHostController = BaseDeviceAccessor::GetHostController();

    // argument[31] Mode : 0(Check function), 1(Switch function)
    // argument[30:24] reserved (0)
    // argument[23:20] function group 6 : F(Current)
    // argument[19:16] function group 5 : F(Current)
    // argument[15:12] function group 4 : F(Current)
    // argument[11: 8] function group 3 : F(Current)
    // Argument[ 7: 4] function group 2 : F(Current)
    // Argument[ 3: 0] function group 1 : Access mode
    uint32_t mode = isSetFunction ? 1 : 0;
    uint32_t argument = (mode << 31) | 0x00FFFFF0U | static_cast<uint32_t>(accessMode);
    ResponseType responseType = ResponseType_R1;
    Command command(6, argument, responseType, false);
    TransferData transferData(pOutSwitchFunctionStatusBuffer, SdCardSwitchFunctionStatusSize, 1, TransferDirection_ReadFromDevice);
    NN_RESULT_DO(pHostController->IssueCommand(&command, &transferData));

    uint32_t response;
    pHostController->GetLastResponse(&response, sizeof(response), responseType);
    NN_RESULT_DO(m_SdCardDevice.CheckDeviceStatus(response));

    NN_DETAIL_SDMMC_DEBUG_LOG("Switch Function Status:\n");
    NN_DETAIL_SDMMC_DATA_LOG(pOutSwitchFunctionStatusBuffer, SdCardSwitchFunctionStatusSize);

    NN_RESULT_SUCCESS;
}

Result SdCardDeviceAccessor::IssueCommandVoltageSwitch() const NN_NOEXCEPT
{
    // argument[31: 0] reserved bits (all 0)
    return BaseDeviceAccessor::IssueCommandAndCheckR1(11, 0, false, DeviceState_Ready);
}

Result SdCardDeviceAccessor::IssueCommandAppCmd(DeviceState expectedState, uint32_t deviceStatusBitsIgnored) const NN_NOEXCEPT
{
    IHostController* pHostController = BaseDeviceAccessor::GetHostController();

    // argument[31:16] RCA
    // argument[15: 0] stuff bits
    ResponseType responseType = ResponseType_R1;
    Command command(55, static_cast<uint32_t>(m_SdCardDevice.GetRca()) << 16, responseType, false);
    NN_RESULT_DO(pHostController->IssueCommand(&command));

    uint32_t response;
    pHostController->GetLastResponse(&response, sizeof(response), responseType);
    if (deviceStatusBitsIgnored != 0)
    {
        response &= (~deviceStatusBitsIgnored);
    }
    NN_RESULT_DO(m_SdCardDevice.CheckDeviceStatus(response));
    if ((response & DeviceStatus_Bit_AppCmd) == 0)
    {
        return ResultUnexpectedSdCardAcmdDisabled();
    }
    if (expectedState != DeviceState_Unknown)
    {
        if (m_SdCardDevice.GetDeviceState(response) != expectedState)
        {
            return ResultUnexpectedDeviceState();
        }
    }

    NN_RESULT_SUCCESS;
}

Result SdCardDeviceAccessor::IssueCommandSetBusWidth4bit() const NN_NOEXCEPT
{
    // argument[31: 2] stuff bits
    // argument[ 1: 0] 10b = 4 bits bus
    return BaseDeviceAccessor::IssueCommandAndCheckR1(6, 0x2, false, DeviceState_Tran);
}

Result SdCardDeviceAccessor::IssueCommandSdStatus(void* pOutSdStatusBuffer, size_t sdStatusBufferSize) const NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(pOutSdStatusBuffer);
    NN_ABORT_UNLESS(sdStatusBufferSize >= SdCardSdStatusSize);

    IHostController* pHostController = BaseDeviceAccessor::GetHostController();

    // argument[31:0] stuff bits
    ResponseType responseType = ResponseType_R1;
    Command command(13, 0, responseType, false);
    TransferData transferData(pOutSdStatusBuffer, SdCardSdStatusSize, 1, TransferDirection_ReadFromDevice);
    NN_RESULT_DO(pHostController->IssueCommand(&command, &transferData));

    uint32_t response;
    pHostController->GetLastResponse(&response, sizeof(response), responseType);
    NN_RESULT_DO(m_SdCardDevice.CheckDeviceStatus(response));

    NN_DETAIL_SDMMC_DEBUG_LOG("SD Status:\n");
    NN_DETAIL_SDMMC_DATA_LOG(pOutSdStatusBuffer, SdCardSdStatusSize);

    NN_RESULT_SUCCESS;
}

Result SdCardDeviceAccessor::IssueCommandSendOpCond(uint32_t* pOutOcr, bool isLessThanSpecVersion2_00, bool isHostSupportUhsIMode) const NN_NOEXCEPT
{
    // argument[31] reserved bit
    // argument[30] HCS : 1(Host Capacity Support) or 0(Not Support)
    // argument[29] reserved for eSD
    // argument[28] XPC : 1(150mA, High Speed対応) SDXC専用,
    // argument[27;25] reserved bits
    // argument[24] S18R : 0(1.8Vにしない), 1(1.8Vにできる)
    // argument[23:0] Voltage : 0x100000(3.2-3.3V),
    uint32_t hcs = isLessThanSpecVersion2_00 ? 0 : 1;
    uint32_t xpc = isLessThanSpecVersion2_00 ? 0 : 1;
    uint32_t s18r = ((!isLessThanSpecVersion2_00) && isHostSupportUhsIMode) ? 1 : 0;
    uint32_t argument = (hcs << 30) | (xpc << 28) | (s18r << 24) | 0x100000U;
    ResponseType responseType = ResponseType_R3;
    Command command(41, argument, responseType, false);

    IHostController* pHostController = BaseDeviceAccessor::GetHostController();
    NN_RESULT_DO(pHostController->IssueCommand(&command));

    pHostController->GetLastResponse(pOutOcr, sizeof(uint32_t), responseType);

    NN_RESULT_SUCCESS;
}

Result SdCardDeviceAccessor::IssueCommandClearCardDetect() const NN_NOEXCEPT
{
    // argument[31:1] = stuff bits
    // argument[0] set_cd : 1(Connect), 0(Disconnect)
    return BaseDeviceAccessor::IssueCommandAndCheckR1(42, 0, false, DeviceState_Tran);
}

Result SdCardDeviceAccessor::IssueCommandSendScr(void* pOutScrBuffer, size_t scrBufferSize) const NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(pOutScrBuffer);
    NN_ABORT_UNLESS(scrBufferSize >= SdCardScrSize);

    IHostController* pHostController = BaseDeviceAccessor::GetHostController();

    // argument[31:0] stuff bits
    ResponseType responseType = ResponseType_R1;
    Command command(51, 0, responseType, false);
    TransferData transferData(pOutScrBuffer, SdCardScrSize, 1, TransferDirection_ReadFromDevice);
    NN_RESULT_DO(pHostController->IssueCommand(&command, &transferData));

    uint32_t response;
    pHostController->GetLastResponse(&response, sizeof(response), responseType);
    NN_RESULT_DO(m_SdCardDevice.CheckDeviceStatus(response));

    NN_DETAIL_SDMMC_DEBUG_LOG("SCR:\n");
    NN_DETAIL_SDMMC_DATA_LOG(pOutScrBuffer, SdCardScrSize);

    NN_RESULT_SUCCESS;
}

Result SdCardDeviceAccessor::EnterUhsIMode() NN_NOEXCEPT
{
    NN_RESULT_DO(IssueCommandVoltageSwitch());

    IHostController* pHostController = BaseDeviceAccessor::GetHostController();
    NN_RESULT_DO(pHostController->SwitchToSdr12());

    NN_RESULT_SUCCESS;
}

Result SdCardDeviceAccessor::ChangeToReadyState(bool isLessThanSpecVersion2_00, bool isHostSupportUhsIMode) NN_NOEXCEPT
{
    uint32_t deviceStatusBitsIgnored = 0;
    if (isLessThanSpecVersion2_00)
    {
        // Physical Spec Version 2.00 未満に対しては未対応コマンドを発行したため、
        // 次に ILLEGAL_COMMAND が通知される可能性がある
        deviceStatusBitsIgnored = DeviceStatus_Bit_IllegalCommand;
    }

    // ready state へ遷移、High Capacity 有無取得
    ManualTimer timer(3000);  // 規格上は 1 秒だが、超えるカードもあるため 3 秒とする
    while (true)
    {
        NN_RESULT_DO(IssueCommandAppCmd(DeviceState_Unknown, deviceStatusBitsIgnored)); // Device State のチェックは除外する
        deviceStatusBitsIgnored = 0;    // Physical Spec Version 2.00 未満でも無視するのは初回のみ

        uint32_t ocr;
        NN_RESULT_DO(IssueCommandSendOpCond(&ocr, isLessThanSpecVersion2_00, isHostSupportUhsIMode));
        if ((ocr & OcrCardPowerUpStatus) != 0)
        {
            m_SdCardDevice.SetOcrAndHighCapacity(ocr);

            m_SdCardDevice.SetUhsIMode(false);
            if (isHostSupportUhsIMode && ((ocr & OcrSwitchingTo18VAccepted) != 0))
            {
                NN_RESULT_DO(EnterUhsIMode());
                m_SdCardDevice.SetUhsIMode(true);
            }

            NN_RESULT_SUCCESS;  // 完了
        }

        if (!(timer.Update()))
        {
            // power up routine が完了しないまま、タイムアウトを過ぎている
            return ResultSdCardInitializationSwTimeout();
        }

        // 最大 1 秒待つ処理に対して 1ms の Sleep を入れる
        WaitMicroseconds(1000);
    }
}

Result SdCardDeviceAccessor::ChangeToStbyStateAndGetRca() NN_NOEXCEPT
{
    ManualTimer timer(1000);  // 規定はないため、1 秒とする
    while (true)
    {
        uint16_t rca;
        NN_RESULT_DO(IssueCommandSendRelativeAddr(&rca));
        if (rca != 0)
        {
            m_SdCardDevice.SetRca(rca);

            NN_RESULT_SUCCESS;  // 完了
        }

        if (!(timer.Update()))
        {
            // 有効な RCA を取得できないまま、タイムアウトを過ぎている
            return ResultSdCardGetValidRcaSwTimeout();
        }
        // デバイスの変化を待つ処理ではないため Sleep はしない
    }
}

Result SdCardDeviceAccessor::SetMemoryCapacity(void* pCsd) NN_NOEXCEPT
{
    if (IsLessThanCsdVersion2_0(reinterpret_cast<uint8_t*>(pCsd)))
    {
        NN_RESULT_DO(m_SdCardDevice.SetLegacyMemoryCapacity());
    }
    else
    {
        NN_ABORT_UNLESS((reinterpret_cast<uintptr_t>(pCsd) % sizeof(uint16_t)) == 0);
        m_SdCardDevice.SetMemoryCapacity(::nn::sdmmc::detail::GetMemoryCapacity(reinterpret_cast<uint16_t*>(pCsd)));
    }

    NN_RESULT_SUCCESS;
}

void SdCardDeviceAccessor::TryDisconnectDat3PullUpResistor() const NN_NOEXCEPT
{
    // DAT3 の pull-up 抵抗を外さなくても実害はないと推測されるため、失敗しても無視する

    Result result = IssueCommandAppCmd(DeviceState_Tran);
    if (result.IsFailure())
    {
        NN_DETAIL_SDMMC_DEBUG_LOG("IssueCommandAppCmd() is failure(Module:%d, Description:%d)\n", result.GetModule(), result.GetDescription());
    }
    else
    {
        result = IssueCommandClearCardDetect();
        if (result.IsFailure())
        {
            NN_DETAIL_SDMMC_DEBUG_LOG("IssueCommandClearCardDetect() is failure(Module:%d, Description:%d)\n", result.GetModule(), result.GetDescription());
        }
    }

    // 成功の有無にかかわらず、Card Status にエラー残っている可能性があるため、コマンド発行によりクリアする
    result = BaseDeviceAccessor::IssueCommandSendStatus();
    if (result.IsFailure())
    {
        NN_DETAIL_SDMMC_DEBUG_LOG("IssueCommandSendStatus() in TryDisconnectDat3PullUpResistor() is failure(Module:%d, Description:%d)\n", result.GetModule(), result.GetDescription());
    }
}

Result SdCardDeviceAccessor::GetScr(void* pOutScrBuffer, size_t scrBufferSize) const NN_NOEXCEPT
{
    NN_RESULT_DO(IssueCommandAppCmd(DeviceState_Tran));
    NN_RESULT_DO(IssueCommandSendScr(pOutScrBuffer, scrBufferSize));

    NN_RESULT_SUCCESS;
}

Result SdCardDeviceAccessor::ExtendBusWidth(BusWidth maxBusWidth, uint8_t sdBusWidths) NN_NOEXCEPT
{
    if (maxBusWidth == BusWidth_1Bit)
    {
        // 1bit バス幅のまま
        NN_RESULT_SUCCESS;
    }

    if (!IsSupportedBusWidth4bit(sdBusWidths))
    {
        // 1bit バス幅のまま
        NN_DETAIL_SDMMC_DEBUG_LOG("Bus Witdh is 1 bit (SD Card doesn't support 4bit)\n");
        NN_RESULT_SUCCESS;
    }

    IHostController* pHostController = BaseDeviceAccessor::GetHostController();
    if (!(pHostController->IsSupportedBusWidth(BusWidth_4Bit)))
    {
        // 1bit バス幅のまま
        NN_DETAIL_SDMMC_DEBUG_LOG("Bus Witdh is 1 bit (Host Controller doesn't support 4bit)\n");
        NN_RESULT_SUCCESS;
    }

    NN_DETAIL_SDMMC_DEBUG_LOG("Set 4bit Bus\n");
    NN_RESULT_DO(IssueCommandAppCmd(DeviceState_Tran));
    NN_RESULT_DO(IssueCommandSetBusWidth4bit());

    pHostController->SetBusWidth(BusWidth_4Bit);

    NN_RESULT_SUCCESS;
}

Result SdCardDeviceAccessor::SwitchAccessMode(SwitchFunctionAccessMode accessMode, void* pSdCardWorkBuffer, size_t sdCardWorkBufferSize) NN_NOEXCEPT
{
    // 切り替え確認
    NN_RESULT_DO(IssueCommandSwitchAccessMode(pSdCardWorkBuffer, sdCardWorkBufferSize, false, accessMode));
    if (!CanBeSwitchedAccessMode(reinterpret_cast<uint8_t*>(pSdCardWorkBuffer), accessMode))
    {
        return ResultSdCardCannotSwitchedAccessMode();
    }
    // 消費電流チェック
    uint16_t maximumCurrentConsumption = GetMaximumCurrentConsumption(reinterpret_cast<uint8_t*>(pSdCardWorkBuffer));
    if (maximumCurrentConsumption > AcceptableCurrentConsumption)
    {
        NN_DETAIL_SDMMC_DEBUG_LOG("Maximum Current Consumption(mA): %u > %u\n", maximumCurrentConsumption, AcceptableCurrentConsumption);
        return ResultSdCardUnacceptableCurrentConsumption();
    }

    // 切り替え
    NN_RESULT_DO(IssueCommandSwitchAccessMode(pSdCardWorkBuffer, sdCardWorkBufferSize, true, accessMode));
    if (!IsSuccessSwitchAccessMode(reinterpret_cast<uint8_t*>(pSdCardWorkBuffer), accessMode))
    {
        return ResultSdCardFailedSwitchedAccessMode();
    }

    NN_RESULT_SUCCESS;
}

Result SdCardDeviceAccessor::ExtendBusSpeedAtUhsIMode(SpeedMode maxSpeedMode, void* pSdCardWorkBuffer, size_t sdCardWorkBufferSize) NN_NOEXCEPT
{
    IHostController* pHostController = BaseDeviceAccessor::GetHostController();
    BusWidth currentBusWidth = pHostController->GetBusWidth();
    if (currentBusWidth != BusWidth_4Bit)
    {
        // 4bit バス幅でなければ UHS-I Mode で Tuning できない。
        NN_DETAIL_SDMMC_DEBUG_LOG("Bus Width isn't 4 bit (%d) at UHS-I Mode\n", currentBusWidth);
        return ResultSdCardNot4BitBusWidthAtUhsIMode();
    }

    // 対応確認
    NN_RESULT_DO(IssueCommandCheckSupportedFunction(pSdCardWorkBuffer, sdCardWorkBufferSize));
    SwitchFunctionAccessMode targetAccessMode;
    SpeedMode targetSpeedMode;
    if ((maxSpeedMode == SpeedMode_SdCardSdr104)
        && IsSupportedAccessMode(reinterpret_cast<uint8_t*>(pSdCardWorkBuffer), SwitchFunctionAccessMode_Sdr104))
    {
        NN_DETAIL_SDMMC_DEBUG_LOG("Switch to SDR104\n");
        targetAccessMode = SwitchFunctionAccessMode_Sdr104;
        targetSpeedMode = SpeedMode_SdCardSdr104;
    }
    else if (((maxSpeedMode == SpeedMode_SdCardSdr104) || (maxSpeedMode == SpeedMode_SdCardSdr50))
        && IsSupportedAccessMode(reinterpret_cast<uint8_t*>(pSdCardWorkBuffer), SwitchFunctionAccessMode_Sdr50))
    {
        NN_DETAIL_SDMMC_DEBUG_LOG("Switch to SDR50\n");
        targetAccessMode = SwitchFunctionAccessMode_Sdr50;
        targetSpeedMode = SpeedMode_SdCardSdr50;
    }
    else
    {
        // 電源 OFF しない限りは UHS-I Mode 内の Speed Mode しか選択できない。
        // SDR25, SDR12 は選択せず、エラーリトライで High Speed を試みる。
        NN_DETAIL_SDMMC_DEBUG_LOG("SD Card doesn't support SDR104 and SDR50\n");
        return ResultSdCardNotSupportSdr104AndSdr50();
    }

    // 切り替え
    NN_RESULT_DO(SwitchAccessMode(targetAccessMode, pSdCardWorkBuffer, sdCardWorkBufferSize));

    NN_RESULT_DO(pHostController->SetSpeedMode(targetSpeedMode));
    NN_RESULT_DO(pHostController->Tuning(targetSpeedMode, 19));

    // SD カードとのコマンド通信確認
    // (レスポンス取得にも Tuning が必要なため、CMD6 のエラーをチェックするタイミングはない)
    NN_RESULT_DO(BaseDeviceAccessor::IssueCommandSendStatus());

    NN_RESULT_SUCCESS;
}

Result SdCardDeviceAccessor::ExtendBusSpeedAtNonUhsIMode(SpeedMode maxSpeedMode, bool isLessThanSpecVersion1_10, void* pSdCardWorkBuffer, size_t sdCardWorkBufferSize) NN_NOEXCEPT
{
    if (maxSpeedMode == SpeedMode_SdCardDefaultSpeed)
    {
        // Default Speed のまま
        NN_RESULT_SUCCESS;
    }

    if (isLessThanSpecVersion1_10)
    {
        // Default Speed のまま
        NN_DETAIL_SDMMC_DEBUG_LOG("Speed Mode is Default (SD Card is less than Version 1.10)\n");
        NN_RESULT_SUCCESS;
    }

    // 対応確認
    NN_RESULT_DO(IssueCommandCheckSupportedFunction(pSdCardWorkBuffer, sdCardWorkBufferSize));
    if (!(IsSupportedAccessMode(reinterpret_cast<uint8_t*>(pSdCardWorkBuffer), SwitchFunctionAccessMode_HighSpeed)))
    {
        // High Speed 未対応であれば、Default Speed のまま
        NN_DETAIL_SDMMC_DEBUG_LOG("Speed Mode is Default (SD Card doesn't support High Speed)\n");
        NN_RESULT_SUCCESS;
    }

    // 切り替え
    NN_DETAIL_SDMMC_DEBUG_LOG("Switch to High Speed Mode\n");
    NN_RESULT_DO(SwitchAccessMode(SwitchFunctionAccessMode_HighSpeed, pSdCardWorkBuffer, sdCardWorkBufferSize));

    // SD カード側にエラーが起こっていないか確認してからクロックを上げる
    NN_RESULT_DO(BaseDeviceAccessor::IssueCommandSendStatus());

    IHostController* pHostController = BaseDeviceAccessor::GetHostController();
    NN_RESULT_DO(pHostController->SetSpeedMode(SpeedMode_SdCardHighSpeed));

    NN_RESULT_SUCCESS;
}

Result SdCardDeviceAccessor::GetSdStatus(void* pOutSdStatusBuffer, size_t sdStatusBufferSize) const NN_NOEXCEPT
{
    NN_RESULT_DO(IssueCommandAppCmd(DeviceState_Tran));
    NN_RESULT_DO(IssueCommandSdStatus(pOutSdStatusBuffer, sdStatusBufferSize));

    NN_RESULT_SUCCESS;
}

Result SdCardDeviceAccessor::StartupSdCardDevice(BusWidth maxBusWidth, SpeedMode maxSpeedMode, void* pSdCardWorkBuffer, size_t sdCardWorkBufferSize) NN_NOEXCEPT
{
    // HostController 起動
    IHostController* pHostController = BaseDeviceAccessor::GetHostController();
    NN_RESULT_DO(pHostController->Startup(BusPower_3_3V, BusWidth_1Bit, SpeedMode_SdCardIdentification, false));

    // 1ms + 74 clock cycles 待ち
    WaitMicroseconds(1000);
    WaitClocks(74, pHostController->GetDeviceClockFrequencyKHz());

    // idle state へ遷移
    NN_RESULT_DO(BaseDeviceAccessor::IssueCommandGoIdleState());

    // Physical Spec Version 2.00 未満かチェック
    bool isLessThanSpecVersion2_00 = false;
    Result result = IssueCommandSendIfCond();
    if (result.IsSuccess())
    {
        // Physical Spec Version 2.00 以上のまま続ける
    }
    else if (ResultResponseTimeoutError::Includes(result))
    {
        // レスポンスがなかった場合は Physical Spec Version 2.00 未満と判断して続ける
        NN_DETAIL_SDMMC_DEBUG_LOG("SD Card is less than Physical Spec Version 2.00\n");
        isLessThanSpecVersion2_00 = true;
    }
    else
    {
        // 上記以外は失敗とする
        return result;
    }

    // 取得できるまで RCA は 0 とする
    m_SdCardDevice.SetRca(0);

    // ready state へ遷移、High Capacity 有無取得、可能な場合は電圧変更 (DDR50 未対応, SDR50 未満は High Speed を試みる)
    bool isUhsIModeEnable = (maxBusWidth != BusWidth_1Bit) && ((maxSpeedMode == SpeedMode_SdCardSdr104) || (maxSpeedMode == SpeedMode_SdCardSdr50));
    bool isHostSupportUhsIMode = pHostController->IsSupportedTuning() && pHostController->IsSupportedBusPower(BusPower_1_8V);
    NN_RESULT_DO(ChangeToReadyState(isLessThanSpecVersion2_00, isUhsIModeEnable && isHostSupportUhsIMode));

    // ident state へ遷移、CID 取得
    NN_RESULT_DO(BaseDeviceAccessor::IssueCommandAllSendCid(pSdCardWorkBuffer, sdCardWorkBufferSize));
    m_SdCardDevice.SetCid(pSdCardWorkBuffer, sdCardWorkBufferSize);

    // stby state へ遷移、RCA 取得
    NN_RESULT_DO(ChangeToStbyStateAndGetRca());

    // CSD 取得
    NN_RESULT_DO(BaseDeviceAccessor::IssueCommandSendCsd(pSdCardWorkBuffer, sdCardWorkBufferSize));
    m_SdCardDevice.SetCsd(pSdCardWorkBuffer, sdCardWorkBufferSize);
    NN_RESULT_DO(SetMemoryCapacity(pSdCardWorkBuffer));

    if (!(m_SdCardDevice.IsUhsIMode()))
    {
        // Default Speed にクロックアップ
        NN_RESULT_DO(pHostController->SetSpeedMode(SpeedMode_SdCardDefaultSpeed));
    }

    // tran state へ遷移
    NN_RESULT_DO(BaseDeviceAccessor::IssueCommandSelectCard());

    // ブロックサイズ設定
    NN_RESULT_DO(BaseDeviceAccessor::IssueCommandSetBlockLenToSectorSize());

    // DAT3 pull-up 抵抗切断試行
    TryDisconnectDat3PullUpResistor();

    // SCR 取得
    NN_RESULT_DO(GetScr(pSdCardWorkBuffer, sdCardWorkBufferSize));
    uint8_t sdBusWidths = GetSdBusWidths(reinterpret_cast<uint8_t*>(pSdCardWorkBuffer));
    bool isLessThanSpecVersion1_10 = IsLessThanSpecVersion1_10(reinterpret_cast<uint8_t*>(pSdCardWorkBuffer));

    // バス幅拡張
    NN_RESULT_DO(ExtendBusWidth(maxBusWidth, sdBusWidths));

    // バススピード拡張
    if (m_SdCardDevice.IsUhsIMode())
    {
        NN_RESULT_DO(ExtendBusSpeedAtUhsIMode(maxSpeedMode, pSdCardWorkBuffer, sdCardWorkBufferSize));
    }
    else
    {
        NN_RESULT_DO(ExtendBusSpeedAtNonUhsIMode(maxSpeedMode, isLessThanSpecVersion1_10, pSdCardWorkBuffer, sdCardWorkBufferSize));

        // バス幅拡張後、一度もデータ転送しないパスがあるため、SD Status 取得によるデータ転送チェック
        NN_RESULT_DO(GetSdStatus(pSdCardWorkBuffer, sdCardWorkBufferSize));
    }

    // 以降、クロック供給を抑える
    pHostController->SetPowerSaving(true);

    NN_RESULT_SUCCESS;
}

Result SdCardDeviceAccessor::OnActivate() NN_NOEXCEPT
{
    struct StartupParameters
    {
        BusWidth busWidth;
        SpeedMode maxSpeedMode;
    };

    StartupParameters startupParameters[] =
    {
        #if (defined(NN_DETAIL_SDMMC_SD_CARD_UHS_I_ENABLE))
            { BusWidth_4Bit, SpeedMode_SdCardSdr104 },          // 1 回目
            { BusWidth_4Bit, SpeedMode_SdCardSdr104 },          // 2 回目: 1 回目と同じ
            { BusWidth_4Bit, SpeedMode_SdCardHighSpeed },       // 3 回目: Speed Mode を落とす
            { BusWidth_4Bit, SpeedMode_SdCardDefaultSpeed },    // 4 回目: Speed Mode をさらに落とす
            { BusWidth_1Bit, SpeedMode_SdCardHighSpeed }        // 5 回目: バス幅を減らす
        #else
            { BusWidth_4Bit, SpeedMode_SdCardHighSpeed },       // 1 回目
            { BusWidth_4Bit, SpeedMode_SdCardHighSpeed },       // 2 回目: 1 回目と同じ
            { BusWidth_4Bit, SpeedMode_SdCardDefaultSpeed },    // 3 回目: Speed Mode を落とす
            { BusWidth_1Bit, SpeedMode_SdCardHighSpeed }        // 4 回目: バス幅を減らす
        #endif
    };

    Result result = ResultInternalError();
    for (int i = 0; i < static_cast<int>(sizeof(startupParameters) / sizeof(startupParameters[0])); i++)
    {
        m_MaxBusWidth = startupParameters[i].busWidth;
        m_MaxSpeedMode = startupParameters[i].maxSpeedMode;

        result = StartupSdCardDevice(m_MaxBusWidth, m_MaxSpeedMode, m_pWorkBuffer, m_WorkBufferSize);
        if (result.IsSuccess())
        {
            if (i != 0)
            {
                BaseDeviceAccessor::PushErrorLog(true, "S %d %d:0", m_MaxBusWidth, m_MaxSpeedMode);
                BaseDeviceAccessor::CountUpActivationErrorCorrections();
            }

            NN_RESULT_SUCCESS;  // 成功したら終了
        }

        NN_DETAIL_SDMMC_RESULT_THROW_IF_REMOVED;

        NN_DETAIL_SDMMC_ERROR_LOG("StartupSdCardDevice(%d, %d) is failure(Module:%d, Description:%d)\n",
            m_MaxBusWidth, m_MaxSpeedMode, result.GetModule(), result.GetDescription());
        BaseDeviceAccessor::PushErrorLog(false, "S %d %d:%X", m_MaxBusWidth, m_MaxSpeedMode, result.GetInnerValueForDebug());

        // 一旦 HostController を落とす（SD カード電源 OFF も含まれる）
        IHostController* pHostController = BaseDeviceAccessor::GetHostController();
        pHostController->Shutdown();
    }

    if (ResultUnexpectedDeviceCsdValue::Includes(result))
    {
        uint32_t csd[DeviceCsdSize / sizeof(uint32_t)];
        m_SdCardDevice.GetCsd(csd, sizeof(csd));
        BaseDeviceAccessor::PushErrorLog(false, "%06X%08X%08X%08X", csd[3] & 0xFFFFFFU, csd[2], csd[1], csd[0]);
    }
    BaseDeviceAccessor::PushErrorTimeStamp();

    #if (defined(NN_DETAIL_SDMMC_SD_CARD_DETECTOR_ENABLE))
        if (ResultCommunicationNotAttained::Includes(result))
        {
            // 通信が成立していなかった場合は Debounce 期間待ち、抜けイベントがあれば Result を置き換える
            WaitMicroseconds(m_pSdCardDetector->GetDebounceMilliSeconds() * 1000);

            NN_DETAIL_SDMMC_RESULT_THROW_IF_REMOVED;
        }
    #endif

    return result;
}

Result SdCardDeviceAccessor::OnReadWrite(uint32_t sectorIndex, uint32_t numSectors, void* pBuffer, size_t bufferSize, bool isRead) NN_NOEXCEPT
{
    Result result = BaseDeviceAccessor::ReadWriteMultiply(sectorIndex, numSectors, 0, pBuffer, bufferSize, isRead);

    #if (defined(NN_DETAIL_SDMMC_SD_CARD_DETECTOR_ENABLE))
        if (ResultCommunicationNotAttained::Includes(result))
        {
            // 通信が成立していなかった場合は Debounce 期間待ち、抜けイベントがあれば Result を置き換える
            WaitMicroseconds(m_pSdCardDetector->GetDebounceMilliSeconds() * 1000);

            NN_DETAIL_SDMMC_RESULT_THROW_IF_REMOVED;
        }
    #endif

    return result;
}

Result SdCardDeviceAccessor::ReStartup() NN_NOEXCEPT
{
    // 一旦 HostController を落とす（SD カード電源 OFF も含まれる）
    IHostController* pHostController = BaseDeviceAccessor::GetHostController();
    pHostController->Shutdown();

    // 再初期化
    Result result = StartupSdCardDevice(m_MaxBusWidth, m_MaxSpeedMode, m_pWorkBuffer, m_WorkBufferSize);
    if (result.IsFailure())
    {
        NN_DETAIL_SDMMC_RESULT_THROW_IF_REMOVED;

        NN_DETAIL_SDMMC_ERROR_LOG("StartupSdCardDevice(%d, %d) is failure(Module:%d, Description:%d)\n",
            m_MaxBusWidth, m_MaxSpeedMode, result.GetModule(), result.GetDescription());
        BaseDeviceAccessor::PushErrorLog(false, "S %d %d:%X", m_MaxBusWidth, m_MaxSpeedMode, result.GetInnerValueForDebug());

        return result;
    }

    NN_RESULT_SUCCESS;
}

void SdCardDeviceAccessor::Initialize() NN_NOEXCEPT
{
    NN_DETAIL_SDMMC_DEVICE_LOCK_GUARD;
    if (m_IsInitialized)
    {
        return;
    }

    BaseDeviceAccessor::SetDevice(&m_SdCardDevice);

    IHostController* pHostController = BaseDeviceAccessor::GetHostController();
    #if (defined(NN_DETAIL_SDMMC_SD_CARD_DETECTOR_ENABLE))
        m_SdCardDevice.InitializeRemovedEvent();
        pHostController->PreSetRemovedEvent(m_SdCardDevice.GetRemovedEvent());
        CallbackInfos callbackInfos;
        callbackInfos.insertedCallback = nullptr;   // 挿入では何もしない
        callbackInfos.pInsertedCallbackParameter = this;
        callbackInfos.removedCallback = RemovedCallbackEntry;
        callbackInfos.pRemovedCallbackParameter = this;
        m_pSdCardDetector->Initialize(&callbackInfos);
    #endif
    pHostController->Initialize();

    m_IsInitialized = true;
}

void SdCardDeviceAccessor::Finalize() NN_NOEXCEPT
{
    NN_DETAIL_SDMMC_DEVICE_LOCK_GUARD;
    if (!m_IsInitialized)
    {
        return;
    }
    m_IsInitialized = false;

    BaseDeviceAccessor::Deactivate();

    IHostController* pHostController = BaseDeviceAccessor::GetHostController();
    pHostController->Finalize();

    #if (defined(NN_DETAIL_SDMMC_SD_CARD_DETECTOR_ENABLE))
        m_pSdCardDetector->Finalize();
        m_SdCardDevice.FinalizeRemovedEvent();
    #endif
}

Result SdCardDeviceAccessor::Activate() NN_NOEXCEPT
{
    #if (defined(NN_DETAIL_SDMMC_SD_CARD_DETECTOR_ENABLE))
        {
            NN_DETAIL_SDMMC_DEVICE_LOCK_GUARD;
            if (!(m_SdCardDevice.IsAwake()))
            {
                // Sleep 中
                return ResultNotAwakened();
            }
            if (m_SdCardDevice.IsActive())
            {
                // Activate 済みならば何もしない
                NN_RESULT_SUCCESS;
            }

            // 以降、RemovedCallback() より RemovedEvent が Signal されれば ResultDeviceRemoved() とする
            m_SdCardDevice.ClearRemovedEvent();
        }

        // 挿入状態の最終確認
        // (RemovedCallback() 内の NN_DETAIL_SDMMC_DEVICE_LOCK_GUARD とデットロックしないよう
        //  NN_DETAIL_SDMMC_DEVICE_LOCK_GUARD 中を避ける)
        if (!(m_pSdCardDetector->IsInserted()))
        {
            return ResultNoDevice();
        }
    #endif

    return BaseDeviceAccessor::Activate();
}

Result SdCardDeviceAccessor::GetSpeedMode(SpeedMode* pOutSpeedMode) const NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(pOutSpeedMode);

    NN_DETAIL_SDMMC_DEVICE_LOCK_GUARD;
    NN_RESULT_DO(m_SdCardDevice.CheckAccessible());

    // SD カード側 Speed Mode の取得
    NN_RESULT_DO(GetScr(m_pWorkBuffer, m_WorkBufferSize));
    bool isLessThanSpecVersion1_10 = IsLessThanSpecVersion1_10(reinterpret_cast<uint8_t*>(m_pWorkBuffer));
    if (isLessThanSpecVersion1_10)
    {
        *pOutSpeedMode = SpeedMode_SdCardDefaultSpeed;
        NN_RESULT_SUCCESS;
    }

    NN_RESULT_DO(IssueCommandCheckSupportedFunction(m_pWorkBuffer, m_WorkBufferSize));
    NN_RESULT_DO(GetCurrentSpeedMode(pOutSpeedMode, reinterpret_cast<uint8_t*>(m_pWorkBuffer), m_SdCardDevice.IsUhsIMode()));

    NN_RESULT_SUCCESS;
}

void SdCardDeviceAccessor::PutSdCardToSleep() NN_NOEXCEPT
{
    NN_DETAIL_SDMMC_DEVICE_LOCK_GUARD;
    if (!(m_SdCardDevice.IsAwake()))
    {
        // Sleep 済みならば何もしない
        return;
    }

    m_SdCardDevice.PutToSleep();

    #if (defined(NN_DETAIL_SDMMC_SD_CARD_DETECTOR_ENABLE))
        m_pSdCardDetector->PutToSleep();
    #endif

    #if (defined(NN_DETAIL_SDMMC_SD_CARD_DETECTOR_ENABLE))
        if (m_SdCardDevice.IsActive() && (!(m_SdCardDevice.IsRemoved())))
    #else
        if (m_SdCardDevice.IsActive())
    #endif
    {
        IHostController* pHostController = BaseDeviceAccessor::GetHostController();
        pHostController->PutToSleep();
    }
}

void SdCardDeviceAccessor::AwakenSdCard() NN_NOEXCEPT
{
    NN_DETAIL_SDMMC_DEVICE_LOCK_GUARD;
    if (m_SdCardDevice.IsAwake())
    {
        // Awake 済みならば何もしない
        return;
    }

    bool isForcedDetectionEvent = false;

    #if (defined(NN_DETAIL_SDMMC_SD_CARD_DETECTOR_ENABLE))
        if (m_SdCardDevice.IsActive() && (!(m_SdCardDevice.IsRemoved())))
    #else
        if (m_SdCardDevice.IsActive())
    #endif
    {
        IHostController* pHostController = BaseDeviceAccessor::GetHostController();
        Result result = pHostController->Awaken();
        if (result.IsSuccess())
        {
            // SD カードが抜き差しされておらず tran-state のままであれば成功する
            result = BaseDeviceAccessor::IssueCommandSendStatus();
            if (result.IsFailure())
            {
                NN_DETAIL_SDMMC_DEBUG_LOG("IssueCommandSendStatus() is failure(Module:%d, Description:%d)\n", result.GetModule(), result.GetDescription());
                isForcedDetectionEvent = true;
            }
        }
        else
        {
            NN_DETAIL_SDMMC_ERROR_LOG("Awaken() is failure(Module:%d, Description:%d)\n", result.GetModule(), result.GetDescription());
            BaseDeviceAccessor::PushErrorLog(true, "A:%X", result.GetInnerValueForDebug());
            isForcedDetectionEvent = true;  // HostController 故障時も抜けたことにする
        }
    }

    #if (defined(NN_DETAIL_SDMMC_SD_CARD_DETECTOR_ENABLE))
        m_pSdCardDetector->Awaken(isForcedDetectionEvent);
    #else
        NN_UNUSED(isForcedDetectionEvent);
    #endif

    m_SdCardDevice.Awaken();
}

Result SdCardDeviceAccessor::GetSdCardProtectedAreaCapacity(uint32_t* pOutNumSectors) const NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(pOutNumSectors);

    NN_RESULT_DO(GetSdCardSdStatus(m_pWorkBuffer, m_WorkBufferSize));
    uint32_t sizeOfProtectedArea = GetSizeOfProtectedArea(reinterpret_cast<uint8_t*>(m_pWorkBuffer));

    uint8_t csd[DeviceCsdSize];
    m_SdCardDevice.GetCsd(csd, sizeof(csd));
    if (IsLessThanCsdVersion2_0(csd))
    {
        uint8_t cSizeMult = 0;
        uint8_t readBlLen = 0;
        m_SdCardDevice.GetLegacyCapacityParameters(&cSizeMult, &readBlLen);
        NN_DETAIL_SDMMC_DEBUG_LOG("Legacy SIZE_OF_PROTECTED_AREA: %u, C_SIZE_MULT: %u, READ_BL_LEN: %u\n", sizeOfProtectedArea, static_cast<uint32_t>(cSizeMult), static_cast<uint32_t>(readBlLen));
        if (((cSizeMult + 2) + readBlLen) >= 9)
        {
            // 桁あふれしないように、べき乗をまとめて計算
            // MULT = 2^(C_SIZE_MULT+2);
            // BLOCK_LEN = 2^READ_BL_LEN;
            // Protected Area(byte) = SIZE_OF_PROTECTED_AREA * MULT * BLOCK_LEN;
            // Protected Area(sector) = Protected Area(byte) / 512;
            *pOutNumSectors = sizeOfProtectedArea << ((cSizeMult + 2) + readBlLen - 9);
            NN_RESULT_SUCCESS;
        }
        else
        {
            return ResultUnexpectedDeviceCsdValue();
        }
    }
    else
    {
        // Protected Area(byte) = SIZE_OF_PROTECTED_AREA;
        // Protected Area(sector) = Protected Area(byte) / 512;
        *pOutNumSectors = sizeOfProtectedArea >> 9;
        NN_RESULT_SUCCESS;
    }
}

Result SdCardDeviceAccessor::GetSdCardScr(void* pOutScrBuffer, size_t scrBufferSize) const NN_NOEXCEPT
{
    NN_DETAIL_SDMMC_DEVICE_LOCK_GUARD;
    NN_RESULT_DO(m_SdCardDevice.CheckAccessible());

    NN_RESULT_DO(GetScr(pOutScrBuffer, scrBufferSize));

    NN_RESULT_SUCCESS;
}

Result SdCardDeviceAccessor::GetSdCardSwitchFunctionStatus(void* pOutSwitchFunctionStatusBuffer, size_t switchFunctionStatusBufferSize,
    SdCardSwitchFunction sdCardSwitchFunction) const NN_NOEXCEPT
{
    NN_DETAIL_SDMMC_DEVICE_LOCK_GUARD;
    NN_RESULT_DO(m_SdCardDevice.CheckAccessible());

    // pOutSwitchFunctionStatusBuffer 領域を活用して、一時的に SCR を格納する
    NN_RESULT_DO(GetScr(pOutSwitchFunctionStatusBuffer, switchFunctionStatusBufferSize));
    bool isLessThanSpecVersion1_10 = IsLessThanSpecVersion1_10(reinterpret_cast<uint8_t*>(pOutSwitchFunctionStatusBuffer));
    if (isLessThanSpecVersion1_10)
    {
        // Switch Function Status を保持しない
        return ResultSdCardNotSupportSwitchFunctionStatus();
    }

    if (sdCardSwitchFunction == SdCardSwitchFunction_CheckSupportedFunction)
    {
        NN_RESULT_DO(IssueCommandCheckSupportedFunction(pOutSwitchFunctionStatusBuffer, switchFunctionStatusBufferSize));
    }
    else
    {
        SwitchFunctionAccessMode accessMode;
        switch (sdCardSwitchFunction)
        {
        case SdCardSwitchFunction_CheckDefault:
            accessMode = SwitchFunctionAccessMode_Default;
            break;
        case SdCardSwitchFunction_CheckHighSpeed:
            accessMode = SwitchFunctionAccessMode_HighSpeed;
            break;
        case SdCardSwitchFunction_CheckSdr50:
            accessMode = SwitchFunctionAccessMode_Sdr50;
            break;
        case SdCardSwitchFunction_CheckSdr104:
            accessMode = SwitchFunctionAccessMode_Sdr104;
            break;
        case SdCardSwitchFunction_CheckDdr50:
            accessMode = SwitchFunctionAccessMode_Ddr50;
            break;
        default:
            NN_UNEXPECTED_DEFAULT;
        }
        NN_RESULT_DO(IssueCommandSwitchAccessMode(pOutSwitchFunctionStatusBuffer, switchFunctionStatusBufferSize, false, accessMode));
    }

    NN_RESULT_SUCCESS;
}

Result SdCardDeviceAccessor::GetSdCardCurrentConsumption(uint16_t* pOutCurrentConsumption, SpeedMode speedMode) const NN_NOEXCEPT
{
    NN_DETAIL_SDMMC_DEVICE_LOCK_GUARD;
    NN_RESULT_DO(m_SdCardDevice.CheckAccessible());

    NN_RESULT_DO(GetScr(m_pWorkBuffer, m_WorkBufferSize));
    bool isLessThanSpecVersion1_10 = IsLessThanSpecVersion1_10(reinterpret_cast<uint8_t*>(m_pWorkBuffer));
    if (isLessThanSpecVersion1_10)
    {
        // Switch Function Status を保持しない
        return ResultSdCardNotSupportSwitchFunctionStatus();
    }

    SwitchFunctionAccessMode accessMode;
    switch (speedMode)
    {
    case SpeedMode_SdCardSdr12:
        NN_FALL_THROUGH;    // Access mode は Default と同じ
    case SpeedMode_SdCardDefaultSpeed:
        accessMode = SwitchFunctionAccessMode_Default;
        break;
    case SpeedMode_SdCardSdr25:
        NN_FALL_THROUGH;    // Access mode は High Speed と同じ
    case SpeedMode_SdCardHighSpeed:
        accessMode = SwitchFunctionAccessMode_HighSpeed;
        break;
    case SpeedMode_SdCardSdr50:
        accessMode = SwitchFunctionAccessMode_Sdr50;
        break;
    case SpeedMode_SdCardSdr104:
        accessMode = SwitchFunctionAccessMode_Sdr104;
        break;
    case SpeedMode_SdCardDdr50:
        accessMode = SwitchFunctionAccessMode_Ddr50;
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
    NN_RESULT_DO(IssueCommandSwitchAccessMode(m_pWorkBuffer, m_WorkBufferSize, false, accessMode));
    if (!IsSupportedAccessMode(reinterpret_cast<uint8_t*>(m_pWorkBuffer), accessMode))
    {
        return ResultSdCardNotSupportAccessMode();
    }

    NN_ABORT_UNLESS_NOT_NULL(pOutCurrentConsumption);
    *pOutCurrentConsumption = GetMaximumCurrentConsumption(reinterpret_cast<uint8_t*>(m_pWorkBuffer));

    NN_RESULT_SUCCESS;
}

Result SdCardDeviceAccessor::GetSdCardSdStatus(void* pOutSdStatusBuffer, size_t sdStatusBufferSize) const NN_NOEXCEPT
{
    NN_DETAIL_SDMMC_DEVICE_LOCK_GUARD;
    NN_RESULT_DO(m_SdCardDevice.CheckAccessible());

    NN_RESULT_DO(GetSdStatus(pOutSdStatusBuffer, sdStatusBufferSize));

    NN_RESULT_SUCCESS;
}

} // namespace detail {
}} // namespace nn { namespace sdmmc {
