﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "sdmmc_MmcDeviceAccessor.h"
#include "sdmmc_Timer.h"
#include "sdmmc_Log.h"
#if (defined(NN_DETAIL_SDMMC_THREAD_SAFE_ENABLE))
    #include <mutex>
#endif
#include <nn/nn_Abort.h>

namespace nn { namespace sdmmc {
namespace detail {

#if (defined(NN_DETAIL_SDMMC_THREAD_SAFE_ENABLE))
    #define NN_DETAIL_SDMMC_DEVICE_LOCK_GUARD   std::lock_guard<nn::os::Mutex> lock(m_MmcDevice.m_DeviceMutex)
#else
    #define NN_DETAIL_SDMMC_DEVICE_LOCK_GUARD
#endif

namespace
{
    // OCR Register ビット定義
    const uint32_t OcrCardPowerUpStatus = 0x1U << 31;
    const uint32_t OcrAccessModeShift = 29;
    const uint32_t OcrAccessModeMask = 0x3U << OcrAccessModeShift;
    const uint32_t OcrAccessModeSectorMode = 0x2U << OcrAccessModeShift;

    // Extended CSD Register DEVICE_TYPE ビット定義
    const uint8_t DeviceTypeHs400__200MHz_1_8V = 0x1U << 6;
    const uint8_t DeviceTypeHs200__200MHz_1_8V = 0x1U << 4;
    const uint8_t DeviceTypeHighSpeed_52MHz = 0x1U << 1;

    bool IsToshibaMmc(uint8_t* pCid) NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_NOT_NULL(pCid);
        // pCid: CID[127:8] が格納されている
        // pCid[14]: CID[127:120]
        // CID[127:120]: MID
        uint8_t mid = pCid[14];
        NN_DETAIL_SDMMC_DEBUG_LOG("MID of CID: 0x%02X\n", mid);
        if (mid == 0x11)    // 東芝製
        {
            NN_DETAIL_SDMMC_DEBUG_LOG("Toshiba eMMC\n");
            return true;
        }
        else
        {
            NN_DETAIL_SDMMC_DEBUG_LOG("NOT Toshiba eMMC\n");
            return false;
        }
    }

    bool IsLessThanStandardVersion(uint8_t* pCsd) NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_NOT_NULL(pCsd);
        // pCsd: CSD[127:8] が格納されている
        // pCsd[14]: CSD[127:120]
        // CSD[125:122]: SPEC_VERS
        uint8_t specVers = (pCsd[14] >> 2) & 0xF;
        NN_DETAIL_SDMMC_DEBUG_LOG("SPEC_VERS of CSD: %u\n", specVers);
        if (specVers < 4)
        {
            NN_DETAIL_SDMMC_DEBUG_LOG("SPEC_VERS indicates non standerd version\n");
            return true;
        }
        else
        {
            NN_DETAIL_SDMMC_DEBUG_LOG("SPEC_VERS indicates version 4.0 or higher\n");
            return false;
        }
    }

    bool IsBkopsAutoEnable(uint8_t* pExtendedCsd) NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_NOT_NULL(pExtendedCsd);
        // Extended CSD[163] = BKOPS_EN
        uint8_t bkopsEn = pExtendedCsd[163];
        NN_DETAIL_SDMMC_DEBUG_LOG("BKOPS_EN of Extended CSD: 0x%02X\n", bkopsEn);
        if ((bkopsEn & (0x1U << 1)) != 0)   // AUTO_EN bit
        {
            NN_DETAIL_SDMMC_DEBUG_LOG("AUTO_EN is enabled\n");
            return true;
        }
        else
        {
            NN_DETAIL_SDMMC_DEBUG_LOG("AUTO_EN is disabled\n");
            return false;
        }
    }

    uint8_t GetMmcDeviceType(uint8_t* pExtendedCsd) NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_NOT_NULL(pExtendedCsd);
        // Extended CSD[196] = DEVICE_TYPE
        uint8_t mmcDevcieType = pExtendedCsd[196];
        NN_DETAIL_SDMMC_DEBUG_LOG("DEVICE_TYPE of Extended CSD: 0x%02X\n", mmcDevcieType);
        return mmcDevcieType;
    }

    bool IsSupportedHighSpeedMode(uint8_t mmcDevcieType) NN_NOEXCEPT
    {
        if (mmcDevcieType & DeviceTypeHighSpeed_52MHz)
        {
            NN_DETAIL_SDMMC_DEBUG_LOG("DEVICE_TYPE indicates that High-Speed 52MHz is supported\n");
            return true;
        }
        else
        {
            NN_DETAIL_SDMMC_DEBUG_LOG("DEVICE_TYPE indicates that High-Speed 52MHz is NOT supported\n");
            return false;
        }
    }

    bool IsSupportedHs200Mode(uint8_t mmcDevcieType) NN_NOEXCEPT
    {
        if (mmcDevcieType & DeviceTypeHs200__200MHz_1_8V)
        {
            NN_DETAIL_SDMMC_DEBUG_LOG("DEVICE_TYPE indicates that HS200 - 1.8V is supported\n");
            return true;
        }
        else
        {
            NN_DETAIL_SDMMC_DEBUG_LOG("DEVICE_TYPE indicates that HS200 - 1.8V is NOT supported\n");
            return false;
        }
    }

    bool IsSupportedHs400Mode(uint8_t mmcDevcieType) NN_NOEXCEPT
    {
        if (mmcDevcieType & DeviceTypeHs400__200MHz_1_8V)
        {
            NN_DETAIL_SDMMC_DEBUG_LOG("DEVICE_TYPE indicates that HS400 - 1.8V is supported\n");
            return true;
        }
        else
        {
            NN_DETAIL_SDMMC_DEBUG_LOG("DEVICE_TYPE indicates that HS400 - 1.8V is NOT supported\n");
            return false;
        }
    }

    uint32_t GetMemoryCapacity(uint32_t* pExtendedCsd) NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_NOT_NULL(pExtendedCsd);
        // pExtendedCsd[53]: Extended CSD[215:212]
        // Extended CSD[215:212]: SEC_COUNT
        return pExtendedCsd[53];
    }

    uint32_t GetBootPartitionCapacity(uint8_t* pExtendedCsd) NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_NOT_NULL(pExtendedCsd);
        // Extended CSD[226]: BOOT_SIZE_MULT
        return pExtendedCsd[226] * ((128 * 1024) / SectorSize);
    }

    Result GetCurrentSpeedMode(SpeedMode* pOutSpeedMode, uint8_t* pExtendedCsd) NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_NOT_NULL(pOutSpeedMode);
        NN_ABORT_UNLESS_NOT_NULL(pExtendedCsd);
        // Extended CSD[185]: HS_TIMING,
        switch (pExtendedCsd[185] & 0xF)    // Bit3:0 Timing Interface
        {
        case 0x0:   // Selecting backwards compatibility interfae timing
            *pOutSpeedMode = SpeedMode_MmcLegacySpeed;
            NN_RESULT_SUCCESS;
        case 0x1:   // High Speed
            *pOutSpeedMode = SpeedMode_MmcHighSpeed;
            NN_RESULT_SUCCESS;
        case 0x2:   // HS200
            *pOutSpeedMode = SpeedMode_MmcHs200;
            NN_RESULT_SUCCESS;
        case 0x3:   // HS400
            *pOutSpeedMode = SpeedMode_MmcHs400;
            NN_RESULT_SUCCESS;
        default:
            NN_DETAIL_SDMMC_ERROR_LOG("Unexpected Extended CSD[185]: 0x%02X\n", pExtendedCsd[185]);
            return ResultUnexpectedMmcExtendedCsdValue();
        }
    }
}

void MmcDevice::SetOcrAndHighCapacity(uint32_t ocr) NN_NOEXCEPT
{
    BaseDevice::SetOcr(ocr);
    if ((ocr & OcrAccessModeMask) == OcrAccessModeSectorMode)
    {
        BaseDevice::SetHighCapacity(true);
    }
    else
    {
        BaseDevice::SetHighCapacity(false);
    }
}

Result MmcDeviceAccessor::IssueCommandSendOpCond(uint32_t* pOutOcr, BusPower busPower) const NN_NOEXCEPT
{
    // argument[31]    reserved
    // argument[30:29] Host support access mode, 10b : sector mode
    // argument[28:24] reserved
    // argument[23:15] 111 1111b : 2.7-3.6V
    // argument[14: 8] 1 1111 1111b : 2.0-2,6V
    // argument[7]     1b : 1.70-1.95V
    // argument[6:0]   reserved
    uint32_t argument = OcrAccessModeSectorMode;
    switch (busPower)
    {
    case BusPower_1_8V:
        argument = argument | (0x1U << 7);
        break;
    case BusPower_3_3V:
        argument = argument | (0x7FU << 15);
        break;
    case BusPower_Off:
        NN_FALL_THROUGH;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
    ResponseType responseType = ResponseType_R3;
    Command command(1, argument, responseType, false);

    IHostController* pHostController = BaseDeviceAccessor::GetHostController();
    NN_RESULT_DO(pHostController->IssueCommand(&command));

    pHostController->GetLastResponse(pOutOcr, sizeof(uint32_t), responseType);

    NN_RESULT_SUCCESS;
}

Result MmcDeviceAccessor::IssueCommandSetRelativeAddr() const NN_NOEXCEPT
{
    uint32_t rca = m_MmcDevice.GetRca();
    NN_ABORT_UNLESS(rca > 0);

    // argument[31:16] RCA
    // argument[15: 0] stuff bits
    // R1, このコマンドで状態遷移するため、state チェックは除外する
    uint32_t argument = static_cast<uint32_t>(rca) << 16;
    return BaseDeviceAccessor::IssueCommandAndCheckR1(3, argument, false, DeviceState_Unknown);
}

Result MmcDeviceAccessor::IssueCommandSwitch(CommandSwitch commandSwitch) const NN_NOEXCEPT
{
    // argument[31:26] Set to 0
    // argument[25:24] Access
    // argument[23:16] Index of Extended CSD
    // argument[15: 8] Value
    // argument[ 7: 3] Set to 0
    // argument[ 2: 0] Cmd Set
    uint32_t argument;
    switch (commandSwitch)
    {
    case CommandSwitch_SetBitsProductionStateAwarenessEnable:
        argument = 0x01111000;
        break;
    case CommandSwitch_ClearBitsAutoModeEnable:
        argument = 0x02112000;
        break;
    case CommandSwitch_WriteProductionStateAwarenessNormal:
        argument = 0x03850000;
        break;
    case CommandSwitch_WriteProductionStateAwarenessPreSolderingWrites:
        argument = 0x03850100;
        break;
    case CommandSwitch_WriteProductionStateAwarenessPreSolderingPostWrites:
        argument = 0x03850200;
        break;
    case CommandSwitch_SetBitsBkopsEnAutoEn:
        argument = 0x01A30200;
        break;
    case CommandSwitch_WriteBusWidth1Bit:
        argument = 0x03B70000;
        break;
    case CommandSwitch_WriteBusWidth4Bit:
        argument = 0x03B70100;
        break;
    case CommandSwitch_WriteBusWidth8Bit:
        argument = 0x03B70200;
        break;
    case CommandSwitch_WriteBusWidth8BitDdr:
        argument = 0x03B70600;
        break;
    case CommandSwitch_WriteHsTimingLegacySpeed:
        argument = 0x03B90000;
        break;
    case CommandSwitch_WriteHsTimingHighSpeed:
        argument = 0x03B90100;
        break;
    case CommandSwitch_WriteHsTimingHs200:
        argument = 0x03B90200;
        break;
    case CommandSwitch_WriteHsTimingHs400:
        argument = 0x03B90300;
        break;
    case CommandSwitch_WritePartitionAccessDefault:
        argument = 0x03B30000;
        break;
    case CommandSwitch_WritePartitionAccessRwBootPartition1:
        argument = 0x03B30100;
        break;
    case CommandSwitch_WritePartitionAccessRwBootPartition2:
        argument = 0x03B30200;
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
    return BaseDeviceAccessor::IssueCommandAndCheckR1(6, argument, true, DeviceState_Unknown);
}

Result MmcDeviceAccessor::IssueCommandSendExtCsd(void* pOutExtendedCsdBuffer, size_t extendedCsdBufferSize) const NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(pOutExtendedCsdBuffer);
    NN_ABORT_UNLESS(extendedCsdBufferSize >= MmcExtendedCsdSize);

    IHostController* pHostController = BaseDeviceAccessor::GetHostController();

    // argument[31:0] stuff bits
    ResponseType responseType = ResponseType_R1;
    Command command(8, 0, responseType, false);
    TransferData transferData(pOutExtendedCsdBuffer, MmcExtendedCsdSize, 1, TransferDirection_ReadFromDevice);
    NN_RESULT_DO(pHostController->IssueCommand(&command, &transferData));

    uint32_t response;
    pHostController->GetLastResponse(&response, sizeof(response), responseType);
    NN_RESULT_DO(m_MmcDevice.CheckDeviceStatus(response));

    NN_DETAIL_SDMMC_DEBUG_LOG("Extended CSD:\n");
    NN_DETAIL_SDMMC_DATA_LOG(pOutExtendedCsdBuffer, MmcExtendedCsdSize);

    NN_RESULT_SUCCESS;
}

Result MmcDeviceAccessor::IssueCommandEraseGroupStart(uint32_t sectorIndex) const NN_NOEXCEPT
{
    // argument[31:0] data address of the first erase group
    uint32_t argument;
    if (m_MmcDevice.IsHighCapacity())
    {
        argument = sectorIndex;
    }
    else
    {
        argument = sectorIndex * SectorSize;
    }
    return BaseDeviceAccessor::IssueCommandAndCheckR1(35, argument, false, DeviceState_Unknown);
}

Result MmcDeviceAccessor::IssueCommandEraseGroupEnd(uint32_t sectorIndex) const NN_NOEXCEPT
{
    // argument[31:0] data address of the last erase group
    uint32_t argument;
    if (m_MmcDevice.IsHighCapacity())
    {
        argument = sectorIndex;
    }
    else
    {
        argument = sectorIndex * SectorSize;
    }
    return BaseDeviceAccessor::IssueCommandAndCheckR1(36, argument, false, DeviceState_Tran);
}

Result MmcDeviceAccessor::IssueCommandErase() const NN_NOEXCEPT
{
    // argument[31:0] Command Description, 0: Erase
    return BaseDeviceAccessor::IssueCommandAndCheckR1(38, 0, true, DeviceState_Tran);
}

Result MmcDeviceAccessor::ChangeToReadyState(BusPower busPower) NN_NOEXCEPT
{
    // ready state へ遷移、High Capacity 有無取得
    ManualTimer timer(1500);  // 規格上は 1 秒だが、余裕を見て 1.5 秒とする
    while (true)
    {
        uint32_t ocr;
        NN_RESULT_DO(IssueCommandSendOpCond(&ocr, busPower));
        if ((ocr & OcrCardPowerUpStatus) != 0)
        {
            m_MmcDevice.SetOcrAndHighCapacity(ocr);

            NN_RESULT_SUCCESS;  // 完了
        }

        if (!(timer.Update()))
        {
            // power up routine が完了しないまま、タイムアウトを過ぎている
            return ResultMmcInitializationSwTimeout();
        }

        // 最大 1 秒待つ処理に対して 1ms 間隔でチェック
        WaitMicroseconds(1000);
    }
}

Result MmcDeviceAccessor::ExtendBusWidth(BusWidth maxBusWidth) NN_NOEXCEPT
{
    if (maxBusWidth == BusWidth_1Bit)
    {
        // 1bit バス幅のまま
        NN_RESULT_SUCCESS;
    }

    BusWidth busWidth = BusWidth_1Bit;
    CommandSwitch commandSwitch;
    IHostController* pHostController = BaseDeviceAccessor::GetHostController();
    if ((maxBusWidth == BusWidth_8Bit)
        && pHostController->IsSupportedBusWidth(BusWidth_8Bit))
    {
        NN_DETAIL_SDMMC_DEBUG_LOG("Set 8bit Bus\n");
        busWidth = BusWidth_8Bit;
        commandSwitch = CommandSwitch_WriteBusWidth8Bit;
    }
    else if (((maxBusWidth == BusWidth_8Bit) || (maxBusWidth == BusWidth_4Bit))
        && pHostController->IsSupportedBusWidth(BusWidth_4Bit))
    {
        NN_DETAIL_SDMMC_DEBUG_LOG("Set 4bit Bus\n");
        busWidth = BusWidth_4Bit;
        commandSwitch = CommandSwitch_WriteBusWidth4Bit;
    }
    else
    {
        // 1bit バス幅のまま
        NN_DETAIL_SDMMC_DEBUG_LOG("Bus Witdh is 1 bit (Host Controller supports only 1 bit)\n");
        NN_RESULT_SUCCESS;
    }

    NN_RESULT_DO(IssueCommandSwitch(commandSwitch));
    NN_RESULT_DO(BaseDeviceAccessor::IssueCommandSendStatus());

    pHostController->SetBusWidth(busWidth);

    NN_RESULT_SUCCESS;
}

Result MmcDeviceAccessor::EnableBkopsAuto() NN_NOEXCEPT
{
    NN_RESULT_DO(IssueCommandSwitch(CommandSwitch_SetBitsBkopsEnAutoEn));
    NN_RESULT_DO(BaseDeviceAccessor::IssueCommandSendStatus());

    NN_RESULT_SUCCESS;
}

Result MmcDeviceAccessor::ChangeToHighSpeed(bool isClockup) NN_NOEXCEPT
{
    NN_DETAIL_SDMMC_DEBUG_LOG("Set MMC to High Speed\n");
    NN_RESULT_DO(IssueCommandSwitch(CommandSwitch_WriteHsTimingHighSpeed));

    if (isClockup)
    {
        // SWITCH_ERROR が起こっていないか確認してから、クロックを上げる
        NN_RESULT_DO(BaseDeviceAccessor::IssueCommandSendStatus());
    }

    NN_DETAIL_SDMMC_DEBUG_LOG("Set HostController to MMC High Speed\n");
    IHostController* pHostController = BaseDeviceAccessor::GetHostController();
    NN_RESULT_DO(pHostController->SetSpeedMode(SpeedMode_MmcHighSpeed));

    if (!isClockup)
    {
        // クロックを下げてから、SWITCH_ERROR が起こっていないか確認する
        NN_RESULT_DO(BaseDeviceAccessor::IssueCommandSendStatus());
    }

    NN_RESULT_SUCCESS;
}

Result MmcDeviceAccessor::ChangeToHs200() NN_NOEXCEPT
{
    NN_DETAIL_SDMMC_DEBUG_LOG("Set MMC to HS200\n");
    NN_RESULT_DO(IssueCommandSwitch(CommandSwitch_WriteHsTimingHs200));

    NN_DETAIL_SDMMC_DEBUG_LOG("Set HostController to HS200\n");
    IHostController* pHostController = BaseDeviceAccessor::GetHostController();
    NN_RESULT_DO(pHostController->SetSpeedMode(SpeedMode_MmcHs200));

    NN_RESULT_DO(pHostController->Tuning(SpeedMode_MmcHs200, 21));

    // MMC とのコマンド通信確認
    // (レスポンス取得にも Tuning が必要なため、SWITCH_ERROR をチェックするタイミングはない)
    NN_RESULT_DO(BaseDeviceAccessor::IssueCommandSendStatus());

    NN_RESULT_SUCCESS;
}

Result MmcDeviceAccessor::ChangeToHs400() NN_NOEXCEPT
{
    // HS400 のレスポンス受信のために Tuning を行う
    NN_RESULT_DO(ChangeToHs200());

    // Tuning 結果のバックアップ
    IHostController* pHostController = BaseDeviceAccessor::GetHostController();
    pHostController->SaveTuningStatusForHs400();

    // HS400 に設定するコマンド通信のために High Speed に下げる
    NN_RESULT_DO(ChangeToHighSpeed(false));

    NN_DETAIL_SDMMC_DEBUG_LOG("Set MMC to HS400\n");
    NN_RESULT_DO(IssueCommandSwitch(CommandSwitch_WriteBusWidth8BitDdr));
    // 直前の SWITCH_ERROR チェックも含む
    NN_RESULT_DO(IssueCommandSwitch(CommandSwitch_WriteHsTimingHs400));

    NN_DETAIL_SDMMC_DEBUG_LOG("Set HostController to HS400\n");
    NN_RESULT_DO(pHostController->SetSpeedMode(SpeedMode_MmcHs400));

    // レスポンス取得タイミングを合わせてから、SWITCH_ERROR が起こっていないか確認する
    NN_RESULT_DO(BaseDeviceAccessor::IssueCommandSendStatus());

    NN_RESULT_SUCCESS;
}

Result MmcDeviceAccessor::ExtendBusSpeed(uint8_t mmcDevcieType, SpeedMode maxSpeedMode) NN_NOEXCEPT
{
    IHostController* pHostController = BaseDeviceAccessor::GetHostController();
    if (pHostController->IsSupportedTuning()
        && (pHostController->GetBusPower() == BusPower_1_8V))
    {
        if ((pHostController->GetBusWidth() == BusWidth_8Bit)
            && IsSupportedHs400Mode(mmcDevcieType)
            && (maxSpeedMode == SpeedMode_MmcHs400))
        {
            return ChangeToHs400();
        }
        else if (((pHostController->GetBusWidth() == BusWidth_8Bit) || (pHostController->GetBusWidth() == BusWidth_4Bit))
            && IsSupportedHs200Mode(mmcDevcieType)
            && ((maxSpeedMode == SpeedMode_MmcHs400) || (maxSpeedMode == SpeedMode_MmcHs200)))
        {
            return ChangeToHs200();
        }
        else
        {
            // 以下で High Speed にトライする
        }
    }

    if (IsSupportedHighSpeedMode(mmcDevcieType))
    {
        // Default Speed から High Speed に上げる
        return ChangeToHighSpeed(true);
    }

    // Default Speed のまま
    NN_DETAIL_SDMMC_DEBUG_LOG("Speed Mode is Default (MMC doesn't support High Speed)\n");
    NN_RESULT_SUCCESS;
}

Result MmcDeviceAccessor::StartupMmcDevice(BusWidth maxBusWidth, SpeedMode maxSpeedMode, void* pMmcWorkBuffer, size_t mmcWorkBufferSize) NN_NOEXCEPT
{
    // HostController 起動
    IHostController* pHostController = BaseDeviceAccessor::GetHostController();
    BusPower busPower = BusPower_1_8V;  // 1.8V 優先
    if (!(pHostController->IsSupportedBusPower(BusPower_1_8V)))
    {
        busPower = BusPower_3_3V;
    }
    NN_RESULT_DO(pHostController->Startup(busPower, BusWidth_1Bit, SpeedMode_MmcIdentification, false));

    // 1ms + 74 clock cycles 待ち
    WaitMicroseconds(1000);
    WaitClocks(74, pHostController->GetDeviceClockFrequencyKHz());

    // idle state へ遷移
    NN_RESULT_DO(BaseDeviceAccessor::IssueCommandGoIdleState());
    m_MmcPartition = MmcPartition_UserData;

    // ready state へ遷移、High Capacity 有無取得
    NN_RESULT_DO(ChangeToReadyState(busPower));

    // ident state へ遷移、CID 取得
    NN_RESULT_DO(BaseDeviceAccessor::IssueCommandAllSendCid(pMmcWorkBuffer, mmcWorkBufferSize));
    m_MmcDevice.SetCid(pMmcWorkBuffer, mmcWorkBufferSize);
    bool isToshibaMmc = IsToshibaMmc(reinterpret_cast<uint8_t*>(pMmcWorkBuffer));

    // RCA 設定、stby state へ遷移
    NN_RESULT_DO(IssueCommandSetRelativeAddr());

    // CSD 取得
    NN_RESULT_DO(BaseDeviceAccessor::IssueCommandSendCsd(pMmcWorkBuffer, mmcWorkBufferSize));
    m_MmcDevice.SetCsd(pMmcWorkBuffer, mmcWorkBufferSize);
    bool isLessThanStandardVersion = IsLessThanStandardVersion(reinterpret_cast<uint8_t*>(pMmcWorkBuffer));

    // Legacy Speed にクロックアップ
    NN_RESULT_DO(pHostController->SetSpeedMode(SpeedMode_MmcLegacySpeed));

    // tran state へ遷移
    NN_RESULT_DO(BaseDeviceAccessor::IssueCommandSelectCard());

    // ブロックサイズ設定
    NN_RESULT_DO(BaseDeviceAccessor::IssueCommandSetBlockLenToSectorSize());

    if (isLessThanStandardVersion)
    {
        // 1bit, Default Speed のまま終了
        NN_DETAIL_SDMMC_DEBUG_LOG("1bit, Default Speed (MMC is less than Standard Version)\n");

        NN_RESULT_DO(m_MmcDevice.SetLegacyMemoryCapacity());

        m_MmcDevice.SetActive();
        NN_RESULT_SUCCESS;
    }

    // バス幅拡張
    NN_RESULT_DO(ExtendBusWidth(maxBusWidth));

    // Extended CSD 取得
    NN_RESULT_DO(IssueCommandSendExtCsd(pMmcWorkBuffer, mmcWorkBufferSize));
    NN_ABORT_UNLESS((reinterpret_cast<uintptr_t>(pMmcWorkBuffer) % sizeof(uint32_t)) == 0);
    m_MmcDevice.SetMemoryCapacity(::nn::sdmmc::detail::GetMemoryCapacity(reinterpret_cast<uint32_t*>(pMmcWorkBuffer)));
    if (isToshibaMmc)
    {
        if (!IsBkopsAutoEnable(reinterpret_cast<uint8_t*>(pMmcWorkBuffer)))
        {
            // 東芝製 eMMC の Read Disturb 対策を有効にする
            Result result = EnableBkopsAuto();
            if (result.IsFailure())
            {
                NN_DETAIL_SDMMC_WARNING_LOG("EnableBkopsAuto() is failure (Module:%d, Description:%d)\n", result.GetModule(), result.GetDescription());
                NN_DETAIL_SDMMC_WARNING_LOG("This MMC may be legacy.\n");
                // エラーにはせず、以降の処理を続ける
            }
        }
    }
    uint8_t mmcDevcieType = GetMmcDeviceType(reinterpret_cast<uint8_t*>(pMmcWorkBuffer));

    // バススピード拡張
    NN_RESULT_DO(ExtendBusSpeed(mmcDevcieType, maxSpeedMode));

    // 以降、クロック供給を抑える
    pHostController->SetPowerSaving(true);

    NN_RESULT_SUCCESS;
}

Result MmcDeviceAccessor::CancelToshibaMmcMode1() NN_NOEXCEPT
{
    // http://ird-mine.nintendo.co.jp/redmine/issues/28724
    NN_RESULT_DO(IssueCommandSwitch(CommandSwitch_SetBitsProductionStateAwarenessEnable));
    NN_RESULT_DO(BaseDeviceAccessor::IssueCommandSendStatus());
    NN_RESULT_DO(IssueCommandSwitch(CommandSwitch_ClearBitsAutoModeEnable));
    NN_RESULT_DO(BaseDeviceAccessor::IssueCommandSendStatus());
    NN_RESULT_DO(IssueCommandSwitch(CommandSwitch_WriteProductionStateAwarenessPreSolderingWrites));
    NN_RESULT_DO(BaseDeviceAccessor::IssueCommandSendStatus());
    NN_RESULT_DO(IssueCommandSwitch(CommandSwitch_WriteProductionStateAwarenessPreSolderingPostWrites));
    NN_RESULT_DO(BaseDeviceAccessor::IssueCommandSendStatus());
    NN_RESULT_DO(IssueCommandSwitch(CommandSwitch_WriteProductionStateAwarenessNormal));
    NN_RESULT_DO(BaseDeviceAccessor::IssueCommandSendStatus());
    NN_RESULT_SUCCESS;
}

Result MmcDeviceAccessor::OnActivate() NN_NOEXCEPT
{
    struct StartupParameters
    {
        BusWidth busWidth;
        SpeedMode maxSpeedMode;
    };

    StartupParameters startupParameters[] =
    {
        #if (defined(NN_DETAIL_SDMMC_MMC_HS400_ENABLE))
            { BusWidth_8Bit, SpeedMode_MmcHs400 },          // 1 回目
        #elif (defined(NN_DETAIL_SDMMC_MMC_HS200_ENABLE))
            { BusWidth_8Bit, SpeedMode_MmcHs200 },          // 1 回目
        #else
            { BusWidth_8Bit, SpeedMode_MmcHighSpeed },      // 1 回目
        #endif
        { BusWidth_8Bit, SpeedMode_MmcHighSpeed },          // 2 回目: Speed Mode を落とす
        { BusWidth_1Bit, SpeedMode_MmcHighSpeed }           // 3 回目: バス幅を減らす
    };

    Result result = ResultInternalError();
    for (int i = 0; i < static_cast<int>(sizeof(startupParameters) / sizeof(startupParameters[0])); i++)
    {
        m_MaxBusWidth = startupParameters[i].busWidth;
        m_MaxSpeedMode = startupParameters[i].maxSpeedMode;

        result = StartupMmcDevice(m_MaxBusWidth, m_MaxSpeedMode, m_pWorkBuffer, m_WorkBufferSize);
        if (result.IsSuccess())
        {
            if (i != 0)
            {
                BaseDeviceAccessor::PushErrorLog(true, "S %d %d:0", m_MaxBusWidth, m_MaxSpeedMode);
                BaseDeviceAccessor::CountUpActivationErrorCorrections();
            }

            NN_RESULT_SUCCESS;  // 成功したら終了
        }

        NN_DETAIL_SDMMC_ERROR_LOG("StartupMmcDevice(%d, %d) is failure(Module:%d, Description:%d)\n",
            m_MaxBusWidth, m_MaxSpeedMode, result.GetModule(), result.GetDescription());
        BaseDeviceAccessor::PushErrorLog(false, "S %d %d:%X", m_MaxBusWidth, m_MaxSpeedMode, result.GetInnerValueForDebug());

        // 一旦 HostController を落とす
        IHostController* pHostController = BaseDeviceAccessor::GetHostController();
        pHostController->Shutdown();
    }

    BaseDeviceAccessor::PushErrorTimeStamp();

    return result;
}

Result MmcDeviceAccessor::OnReadWrite(uint32_t sectorIndex, uint32_t numSectors, void* pBuffer, size_t bufferSize, bool isRead) NN_NOEXCEPT
{
    uint32_t divisionSectorIndexAlignment = 0;
    if (!isRead)
    {
        #if (defined(NN_DETAIL_SDMMC_MMC_WRITE_ALIGN_WARNING_ENABLE) || defined(NN_DETAIL_SDMMC_MMC_WRITE_ALIGN_ABORT_ENABLE))
            const uint32_t MmcWriteAddressKiBAlignment = 16;    // (KiB) http://spdlybra.nintendo.co.jp/jira/browse/IAAA-723
            const uint32_t MmcWriteSectorIndexAlignment = (MmcWriteAddressKiBAlignment * 1024) / SectorSize;
            divisionSectorIndexAlignment = MmcWriteSectorIndexAlignment;
        #endif
        #if (defined(NN_DETAIL_SDMMC_MMC_WRITE_ALIGN_WARNING_ENABLE))
            if ((sectorIndex % MmcWriteSectorIndexAlignment) != 0)
            {
                NN_DETAIL_SDMMC_LOG("[sdmmc] Warning: Write MMC address 0x%08X%08X(byte) is not %u(KiB) alignment.\n",
                    static_cast<uint32_t>((static_cast<uint64_t>(sectorIndex) * SectorSize) >> 32),
                    static_cast<uint32_t>((static_cast<uint64_t>(sectorIndex) * SectorSize)), MmcWriteAddressKiBAlignment);
            }
        #endif
        #if (defined(NN_DETAIL_SDMMC_MMC_WRITE_ALIGN_ABORT_ENABLE))
            NN_ABORT_UNLESS((sectorIndex % MmcWriteSectorIndexAlignment) == 0,
                "[sdmmc] Error: Write MMC address 0x%08X%08X(byte) is not %u(KiB) alignment.\n",
                static_cast<uint32_t>((static_cast<uint64_t>(sectorIndex) * SectorSize) >> 32),
                static_cast<uint32_t>(static_cast<uint64_t>(sectorIndex) * SectorSize), MmcWriteAddressKiBAlignment);
        #endif
    }
    return BaseDeviceAccessor::ReadWriteMultiply(sectorIndex, numSectors, divisionSectorIndexAlignment, pBuffer, bufferSize, isRead);
}

Result MmcDeviceAccessor::ReStartup() NN_NOEXCEPT
{
    // 一旦 HostController を落とす
    IHostController* pHostController = BaseDeviceAccessor::GetHostController();
    pHostController->Shutdown();

    // 再初期化
    Result result = StartupMmcDevice(m_MaxBusWidth, m_MaxSpeedMode, m_pWorkBuffer, m_WorkBufferSize);
    if (result.IsFailure())
    {
        NN_DETAIL_SDMMC_ERROR_LOG("StartupMmcDevice(%d, %d) is failure(Module:%d, Description:%d)\n",
            m_MaxBusWidth, m_MaxSpeedMode, result.GetModule(), result.GetDescription());
        BaseDeviceAccessor::PushErrorLog(false, "S %d %d:%X", m_MaxBusWidth, m_MaxSpeedMode, result.GetInnerValueForDebug());

        return result;
    }

    NN_RESULT_SUCCESS;
}

void MmcDeviceAccessor::Initialize() NN_NOEXCEPT
{
    NN_DETAIL_SDMMC_DEVICE_LOCK_GUARD;
    if (m_IsInitialized)
    {
        return;
    }

    BaseDeviceAccessor::SetDevice(&m_MmcDevice);

    IHostController* pHostController = BaseDeviceAccessor::GetHostController();
    pHostController->Initialize();

    m_IsInitialized = true;
}

void MmcDeviceAccessor::Finalize() NN_NOEXCEPT
{
    NN_DETAIL_SDMMC_DEVICE_LOCK_GUARD;
    if (!m_IsInitialized)
    {
        return;
    }
    m_IsInitialized = false;

    BaseDeviceAccessor::Deactivate();

    IHostController* pHostController = BaseDeviceAccessor::GetHostController();
    pHostController->Finalize();
}

Result MmcDeviceAccessor::GetSpeedMode(SpeedMode* pOutSpeedMode) const NN_NOEXCEPT
{
    NN_RESULT_DO(GetMmcExtendedCsd(m_pWorkBuffer, m_WorkBufferSize));
    NN_ABORT_UNLESS_NOT_NULL(pOutSpeedMode);
    NN_RESULT_DO(GetCurrentSpeedMode(pOutSpeedMode, reinterpret_cast<uint8_t*>(m_pWorkBuffer)));

    NN_RESULT_SUCCESS;
}

void MmcDeviceAccessor::PutMmcToSleep() NN_NOEXCEPT
{
    NN_DETAIL_SDMMC_DEVICE_LOCK_GUARD;
    if (!(m_MmcDevice.IsAwake()))
    {
        // Sleep 済みならば何もしない
        return;
    }

    m_MmcDevice.PutToSleep();

    if (m_MmcDevice.IsActive())
    {
        IHostController* pHostController = BaseDeviceAccessor::GetHostController();
        pHostController->PutToSleep();
    }
}

void MmcDeviceAccessor::AwakenMmc() NN_NOEXCEPT
{
    NN_DETAIL_SDMMC_DEVICE_LOCK_GUARD;
    if (m_MmcDevice.IsAwake())
    {
        // Awake 済みならば何もしない
        return;
    }

    if (m_MmcDevice.IsActive())
    {
        IHostController* pHostController = BaseDeviceAccessor::GetHostController();
        Result result = pHostController->Awaken();
        if (result.IsFailure())
        {
            NN_DETAIL_SDMMC_ERROR_LOG("Awaken() is failure(Module:%d, Description:%d)\n", result.GetModule(), result.GetDescription());
            BaseDeviceAccessor::PushErrorLog(true, "A:%X", result.GetInnerValueForDebug());
        }
    }

    m_MmcDevice.Awaken();
}

Result MmcDeviceAccessor::SelectMmcPartition(MmcPartition mmcPartition) NN_NOEXCEPT
{
    NN_DETAIL_SDMMC_DEVICE_LOCK_GUARD;
    NN_RESULT_DO(m_MmcDevice.CheckAccessible());

    CommandSwitch commandSwitch;
    switch (mmcPartition)
    {
    case MmcPartition_UserData:
        commandSwitch = CommandSwitch_WritePartitionAccessDefault;
        break;
    case MmcPartition_BootPartition1:
        commandSwitch = CommandSwitch_WritePartitionAccessRwBootPartition1;
        break;
    case MmcPartition_BootPartition2:
        commandSwitch = CommandSwitch_WritePartitionAccessRwBootPartition2;
        break;
    case MmcPartition_Unknown:
        NN_FALL_THROUGH;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
    m_MmcPartition = MmcPartition_Unknown;
    NN_RESULT_DO(IssueCommandSwitch(commandSwitch));
    NN_RESULT_DO(BaseDeviceAccessor::IssueCommandSendStatus());
    m_MmcPartition = mmcPartition;

    NN_RESULT_SUCCESS;
}

Result MmcDeviceAccessor::EraseMmc() NN_NOEXCEPT
{
    NN_DETAIL_SDMMC_DEVICE_LOCK_GUARD;
    NN_RESULT_DO(m_MmcDevice.CheckAccessible());

    // 容量取得
    uint32_t partitionCapacity;
    if (m_MmcPartition == MmcPartition_UserData)
    {
        partitionCapacity = m_MmcDevice.GetMemoryCapacity();
    }
    else if ((m_MmcPartition == MmcPartition_BootPartition1) || (m_MmcPartition == MmcPartition_BootPartition2))
    {
        NN_RESULT_DO(GetMmcBootPartitionCapacity(&partitionCapacity));
    }
    else
    {
        // SelectMmcPartition() に失敗した後に Erase されることは想定していない
        NN_ABORT("MMC Partition is unknown.\n");
    }

    // 全域 Erase
    NN_RESULT_DO(IssueCommandEraseGroupStart(0));
    NN_RESULT_DO(IssueCommandEraseGroupEnd(partitionCapacity - 1));
    ManualTimer timer(30000);   // 30 秒
    Result result = IssueCommandErase();
    if (!(result.IsSuccess()
        || ResultDataTimeoutError::Includes(result) // これに Busy のタイムアウトが含まれる Controller がある
        || ResultCommandCompleteSwTimeout::Includes(result)
        || ResultBusySwTimeout::Includes(result)))
    {
        // タイムアウト以外のエラーは失敗とする
        return result;
    }
    // Busy 後のデバイスエラーチェック
    while (true)
    {
        result = BaseDeviceAccessor::IssueCommandSendStatus();
        if (result.IsSuccess())
        {
            break;
        }
        if (!ResultUnexpectedDeviceState::Includes(result))
        {
            // state 不正（tran state ではない）以外のエラーは失敗とする
            return result;
        }
        if (!(timer.Update()))
        {
            // tran state にならないまま、タイムアウトを過ぎている
            return ResultMmcEraseSwTimeout();
        }
    }

    if (m_MmcPartition == MmcPartition_UserData)
    {
        uint8_t cid[DeviceCidSize];
        m_MmcDevice.GetCid(cid, sizeof(cid));
        if (IsToshibaMmc(cid))
        {
            // 東芝製 eMMC の User Data を安定した書き込み速度にする
            result = CancelToshibaMmcMode1();
            if (result.IsFailure())
            {
                NN_DETAIL_SDMMC_ERROR_LOG("CancelToshibaMmcMode1() is failure (Module:%d, Description:%d)\n", result.GetModule(), result.GetDescription());
                // エラーにはせず、以降の処理を続ける
            }
        }
    }

    NN_RESULT_SUCCESS;
}

Result MmcDeviceAccessor::GetMmcBootPartitionCapacity(uint32_t* pOutNumSectors) const NN_NOEXCEPT
{
    NN_RESULT_DO(GetMmcExtendedCsd(m_pWorkBuffer, m_WorkBufferSize));
    NN_ABORT_UNLESS_NOT_NULL(pOutNumSectors);
    *pOutNumSectors = GetBootPartitionCapacity(reinterpret_cast<uint8_t*>(m_pWorkBuffer));

    NN_RESULT_SUCCESS;
}

Result MmcDeviceAccessor::GetMmcExtendedCsd(void* pOutExtendedCsdBuffer, size_t extendedCsdBufferSize) const NN_NOEXCEPT
{
    NN_DETAIL_SDMMC_DEVICE_LOCK_GUARD;
    NN_RESULT_DO(m_MmcDevice.CheckAccessible());

    uint8_t csd[DeviceCsdSize];
    m_MmcDevice.GetCsd(csd, sizeof(csd));
    bool isLessThanStandardVersion = IsLessThanStandardVersion(csd);
    if (isLessThanStandardVersion)
    {
        // Extended CSD を保持しない
        return ResultMmcNotSupportExtendedCsd();
    }

    NN_RESULT_DO(IssueCommandSendExtCsd(pOutExtendedCsdBuffer, extendedCsdBufferSize));

    NN_RESULT_SUCCESS;
}

} // namespace detail {
}} // namespace nn { namespace sdmmc {
