﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "sdmmc_GcAsicDeviceAccessor.h"
#include "sdmmc_Timer.h"
#include "sdmmc_Log.h"
#include <nn/sdmmc/sdmmc_GcAsic.h>
#if (defined(NN_DETAIL_SDMMC_THREAD_SAFE_ENABLE))
    #include <mutex>
#endif
#include <nn/nn_Abort.h>

namespace nn { namespace sdmmc {
namespace detail {

#if (defined(NN_DETAIL_SDMMC_THREAD_SAFE_ENABLE))
    #define NN_DETAIL_SDMMC_DEVICE_LOCK_GUARD   std::lock_guard<nn::os::Mutex> lock(m_GcAsicDevice.m_DeviceMutex)
#else
    #define NN_DETAIL_SDMMC_DEVICE_LOCK_GUARD
#endif

#if (defined(NN_DETAIL_SDMMC_USE_NN_OS_FOR_EVENT))
    #define NN_DETAIL_SDMMC_RESULT_THROW_IF_REMOVED     if (m_GcAsicDevice.IsRemoved()) { return ResultDeviceRemoved(); }
#else
    #define NN_DETAIL_SDMMC_RESULT_THROW_IF_REMOVED
#endif

namespace
{
    #if (defined(NN_DETAIL_SDMMC_GC_ASIC_IS_FPGA))
        SpeedMode GcAsicSpeedMode = SpeedMode_GcAsicFpgaSpeed;
    #else
        SpeedMode GcAsicSpeedMode = SpeedMode_GcAsicSpeed;
    #endif
}

Result GcAsicDeviceAccessor::IssueCommandWriteOperation(const void* pOperationBuffer, size_t operationBufferSize) const NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(pOperationBuffer);
    NN_ABORT_UNLESS(operationBufferSize >= GcAsicOperationSize);

    IHostController* pHostController = BaseDeviceAccessor::GetHostController();

    // Argument は 0 固定
    ResponseType responseType = ResponseType_R1;
    Command command(60, 0, responseType, false);
    TransferData transferData(const_cast<void*>(pOperationBuffer), GcAsicOperationSize, 1, TransferDirection_WriteToDevice);
    Result result = pHostController->IssueCommand(&command, &transferData);
    if (result.IsFailure())
    {
        NN_DETAIL_SDMMC_RESULT_THROW_IF_REMOVED;

        // デバイスが検出したエラーを優先して返す
        NN_DETAIL_SDMMC_ERROR_CMD_LOG("CMD60 is failure(Module:%d, Description:%d)\n", result.GetModule(), result.GetDescription());
        Result overridingResult = result;

        // デバイスまで CMD60 が届いていたか不明だが、CMD12 を発行する
        // (GcAsic には ILLEGAL_COMMAND はなく、CMD12 は常時受け付け可能)
        uint32_t response = 0;
        result = pHostController->IssueStopTransmissionCommand(&response);
        if (result.IsSuccess())
        {
            result = m_GcAsicDevice.CheckDeviceStatus(response);
            if (result.IsFailure())
            {
                // デバイスがエラーを検出していた場合は、それを表す Result に置き換える
                overridingResult = result;
            }
        }
        if (result.IsSuccess())
        {
            NN_DETAIL_SDMMC_ERROR_CMD_LOG("CMD12 Resp 0x%08X\n", response);
        }
        else
        {
            NN_DETAIL_SDMMC_ERROR_CMD_LOG("CMD12 Resp 0x%08X(unsure) is failure(Module:%d, Description:%d)\n", response, result.GetModule(), result.GetDescription());
        }

        NN_DETAIL_SDMMC_RESULT_THROW_IF_REMOVED;

        // デバイス内部のエラーを CMD13 発行によりクリアする
        uint32_t deviceStatus = 0;
        result = BaseDeviceAccessor::IssueCommandSendStatus(&deviceStatus, 0);  // 無視すべきエラービットはない
        if ((!ResultDeviceStatusHasError::Includes(overridingResult)) && ResultDeviceStatusHasError::Includes(result))
        {
            // デバイスがエラーを検出していた場合は、それを表す Result に置き換える
            overridingResult = result;
        }
        if (result.IsSuccess())
        {
            NN_DETAIL_SDMMC_ERROR_CMD_LOG("CMD13 Resp 0x%08X\n", deviceStatus);
        }
        else
        {
            NN_DETAIL_SDMMC_ERROR_CMD_LOG("CMD13 Resp 0x%08X(unsure) is failure(Module:%d, Description:%d)\n", deviceStatus, result.GetModule(), result.GetDescription());
        }

        return overridingResult;
    }

    uint32_t response;
    pHostController->GetLastResponse(&response, sizeof(response), responseType);
    NN_RESULT_DO(m_GcAsicDevice.CheckDeviceStatus(response));

    NN_RESULT_SUCCESS;
}

Result GcAsicDeviceAccessor::IssueCommandFinishOperation() const NN_NOEXCEPT
{
    // Argument は 0 固定
    return BaseDeviceAccessor::IssueCommandAndCheckR1(61, 0, true, DeviceState_Tran);
}

Result GcAsicDeviceAccessor::IssueCommandSleep() const NN_NOEXCEPT
{
    // Argument は 0 固定
    return BaseDeviceAccessor::IssueCommandAndCheckR1(62, 0, true, DeviceState_Tran);
}

Result GcAsicDeviceAccessor::IssueCommandUpdateKey() const NN_NOEXCEPT
{
    // Argument は 0 固定
    return BaseDeviceAccessor::IssueCommandAndCheckR1(63, 0, true, DeviceState_Tran);
}

Result GcAsicDeviceAccessor::StartupGcAsicDevice() NN_NOEXCEPT
{
    // HostController 起動
    IHostController* pHostController = BaseDeviceAccessor::GetHostController();
    NN_RESULT_DO(pHostController->Startup(BusPower_1_8V, BusWidth_8Bit, GcAsicSpeedMode, false));

    // 10 clock cycles 待ち
    WaitClocks(10, pHostController->GetDeviceClockFrequencyKHz());

    // いきなり Tuning
    NN_ABORT_UNLESS(pHostController->IsSupportedTuning());
    NN_RESULT_DO(pHostController->Tuning(GcAsicSpeedMode, 21));

    // Argument は 0 固定のため true, false どちらでも構わないが、
    // IssueCommandMultipleBlock() で IsHighCapacity() を使用できるように、いずれか設定する
    m_GcAsicDevice.SetHighCapacity(false);

    // IssueCommandMultipleBlock() で GetMemoryCapacity() を使用した
    // 終端チェックが無効になるよう便宜上 0 を設定する
    m_GcAsicDevice.SetMemoryCapacity(0);

    // 以降、クロック供給を抑える
    pHostController->SetPowerSaving(true);

    NN_RESULT_SUCCESS;
}

Result GcAsicDeviceAccessor::OnActivate() NN_NOEXCEPT
{
    Result result = StartupGcAsicDevice();
    if (result.IsSuccess())
    {
        NN_RESULT_SUCCESS;
    }

    NN_DETAIL_SDMMC_ERROR_LOG("StartupGcAsicDevice() is failure(Module:%d, Description:%d)\n", result.GetModule(), result.GetDescription());

    // HostController を落としておく
    IHostController* pHostController = BaseDeviceAccessor::GetHostController();
    pHostController->Shutdown();

    return result;
}

Result GcAsicDeviceAccessor::OnReadWrite(uint32_t sectorIndex, uint32_t numSectors, void* pBuffer, size_t bufferSize, bool isRead) NN_NOEXCEPT
{
    // 転送セクタ数が 0 より大きいことを確認
    NN_ABORT_UNLESS(numSectors > 0);

    // バッファサイズが十分であることを確認
    NN_ABORT_UNLESS((bufferSize / SectorSize) >= numSectors);

    uint32_t transferredNumBlocks;
    NN_RESULT_DO(BaseDeviceAccessor::ReadWriteSingly(&transferredNumBlocks, sectorIndex, numSectors, pBuffer, isRead));

    // 実際に転送されたセクタ数が指定された転送セクタ数と一致することを確認
    NN_ABORT_UNLESS(transferredNumBlocks == numSectors);

    NN_RESULT_SUCCESS;
}

void GcAsicDeviceAccessor::Initialize() NN_NOEXCEPT
{
    NN_DETAIL_SDMMC_DEVICE_LOCK_GUARD;
    if (m_IsInitialized)
    {
        return;
    }

    BaseDeviceAccessor::SetDevice(&m_GcAsicDevice);

    IHostController* pHostController = BaseDeviceAccessor::GetHostController();
    #if (defined(NN_DETAIL_SDMMC_USE_NN_OS_FOR_EVENT))
        m_GcAsicDevice.InitializeRemovedEvent();
        pHostController->PreSetRemovedEvent(m_GcAsicDevice.GetRemovedEvent());
    #endif
    pHostController->Initialize();

    m_IsInitialized = true;
}

void GcAsicDeviceAccessor::Finalize() NN_NOEXCEPT
{
    NN_DETAIL_SDMMC_DEVICE_LOCK_GUARD;
    if (!m_IsInitialized)
    {
        return;
    }
    m_IsInitialized = false;

    BaseDeviceAccessor::Deactivate();

    IHostController* pHostController = BaseDeviceAccessor::GetHostController();
    pHostController->Finalize();
    #if (defined(NN_DETAIL_SDMMC_USE_NN_OS_FOR_EVENT))
        m_GcAsicDevice.FinalizeRemovedEvent();
    #endif
}

Result GcAsicDeviceAccessor::GetSpeedMode(SpeedMode* pOutSpeedMode) const NN_NOEXCEPT
{
    NN_DETAIL_SDMMC_DEVICE_LOCK_GUARD;
    NN_RESULT_DO(m_GcAsicDevice.CheckAccessible());

    // ゲームカード ASIC 側 Speed Mode は固定で取得機能はない
    NN_ABORT_UNLESS_NOT_NULL(pOutSpeedMode);
    *pOutSpeedMode = GcAsicSpeedMode;

    NN_RESULT_SUCCESS;
}

void GcAsicDeviceAccessor::PutGcAsicToSleep() NN_NOEXCEPT
{
    NN_DETAIL_SDMMC_DEVICE_LOCK_GUARD;
    if (!(m_GcAsicDevice.IsAwake()))
    {
        // Sleep 済みならば何もしない
        return;
    }

    #if (defined(NN_DETAIL_SDMMC_USE_NN_OS_FOR_EVENT))
        if (m_GcAsicDevice.IsActive() && (!(m_GcAsicDevice.IsRemoved())))
    #else
        if (m_GcAsicDevice.IsActive())
    #endif
    {
        IHostController* pHostController = BaseDeviceAccessor::GetHostController();
        pHostController->PutToSleep();
    }

    m_GcAsicDevice.PutToSleep();
}

Result GcAsicDeviceAccessor::AwakenGcAsic() NN_NOEXCEPT
{
    NN_DETAIL_SDMMC_DEVICE_LOCK_GUARD;
    if (m_GcAsicDevice.IsAwake())
    {
        // Awake 済みならば何もしない
        NN_RESULT_SUCCESS;
    }

    m_GcAsicDevice.Awaken();

    #if (defined(NN_DETAIL_SDMMC_USE_NN_OS_FOR_EVENT))
        if (m_GcAsicDevice.IsActive() && (!(m_GcAsicDevice.IsRemoved())))
    #else
        if (m_GcAsicDevice.IsActive())
    #endif
    {
        IHostController* pHostController = BaseDeviceAccessor::GetHostController();
        Result result = pHostController->Awaken();
        if (result.IsFailure())
        {
            NN_DETAIL_SDMMC_ERROR_LOG("Awaken() is failure(Module:%d, Description:%d)\n", result.GetModule(), result.GetDescription());
            return result;  // HostController の起動に失敗してエラーを返す場合も m_GcAsicDevice.Awaken() とする
        }
    }

    NN_RESULT_SUCCESS;
}

Result GcAsicDeviceAccessor::WriteGcAsicOperation(const void* pOperationBuffer, size_t operationBufferSize) NN_NOEXCEPT
{
    NN_DETAIL_SDMMC_DEVICE_LOCK_GUARD;
    NN_RESULT_DO(m_GcAsicDevice.CheckAccessible());

    NN_RESULT_DO(IssueCommandWriteOperation(pOperationBuffer, operationBufferSize));
    NN_RESULT_DO(BaseDeviceAccessor::IssueCommandSendStatus());

    NN_RESULT_SUCCESS;
}

Result GcAsicDeviceAccessor::FinishGcAsicOperation() NN_NOEXCEPT
{
    NN_DETAIL_SDMMC_DEVICE_LOCK_GUARD;
    NN_RESULT_DO(m_GcAsicDevice.CheckAccessible());

    NN_RESULT_DO(IssueCommandFinishOperation());
    NN_RESULT_DO(BaseDeviceAccessor::IssueCommandSendStatus());

    NN_RESULT_SUCCESS;
}

Result GcAsicDeviceAccessor::AbortGcAsicOperation() NN_NOEXCEPT
{
    NN_DETAIL_SDMMC_DEVICE_LOCK_GUARD;
    NN_RESULT_DO(m_GcAsicDevice.CheckAccessible());

    uint32_t response = 0;
    IHostController* pHostController = BaseDeviceAccessor::GetHostController();
    NN_RESULT_DO(pHostController->IssueStopTransmissionCommand(&response));
    NN_RESULT_DO(m_GcAsicDevice.CheckDeviceStatus(response));

    NN_RESULT_SUCCESS;
}

Result GcAsicDeviceAccessor::SleepGcAsic() NN_NOEXCEPT
{
    NN_DETAIL_SDMMC_DEVICE_LOCK_GUARD;
    NN_RESULT_DO(m_GcAsicDevice.CheckAccessible());

    NN_RESULT_DO(IssueCommandSleep());
    NN_RESULT_DO(BaseDeviceAccessor::IssueCommandSendStatus());

    NN_RESULT_SUCCESS;
}

Result GcAsicDeviceAccessor::UpdateGcAsicKey() NN_NOEXCEPT
{
    NN_DETAIL_SDMMC_DEVICE_LOCK_GUARD;
    NN_RESULT_DO(m_GcAsicDevice.CheckAccessible());

    NN_RESULT_DO(IssueCommandUpdateKey());
    NN_RESULT_DO(BaseDeviceAccessor::IssueCommandSendStatus());

    NN_RESULT_SUCCESS;
}

} // namespace detail {
}} // namespace nn { namespace sdmmc {
