﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "sdmmc_ClockResetController.tegra.h"
#include "sdmmc_ClockResetController.tegra.reg.h"
#include "sdmmc_ClockResetController.tegra.pcv.h"
#if (defined(NN_DETAIL_SDMMC_THREAD_SAFE_ENABLE))
    #include <mutex>
    #include <nn/os/os_Mutex.h>
#endif
#include <nn/nn_Abort.h>

namespace nn { namespace sdmmc {
namespace detail {

namespace ClockResetController {

namespace
{
    #if (defined(NN_DETAIL_SDMMC_THREAD_SAFE_ENABLE))
        nn::os::Mutex g_Mutex(false);
        #define NN_DETAIL_SDMMC_INITIALIZE_LOCK_GUARD   std::lock_guard<nn::os::Mutex> lock(g_Mutex)
    #else
        #define NN_DETAIL_SDMMC_INITIALIZE_LOCK_GUARD
    #endif

    bool g_IsInitialized = false;
    bool g_IsModuleInitialized[Module_Num];
    bool g_IsPcvControl = false;

    void InitializeIfNeeded() NN_NOEXCEPT
    {
        if (g_IsInitialized)
        {
            return; // 初期化済み
        }

        for (int i = 0; i < Module_Num; i++)
        {
            g_IsModuleInitialized[i] = false;
        }

        g_IsInitialized = true;
    }
}

void SwitchToPcvControl() NN_NOEXCEPT
{
    NN_DETAIL_SDMMC_INITIALIZE_LOCK_GUARD;

    InitializeIfNeeded();

    if (g_IsPcvControl)
    {
        return; // pcv 制御に切り替え済み
    }

    for (int i = 0; i < Module_Num; i++)
    {
        if (g_IsModuleInitialized[i])
        {
            detail::ClockResetController::reg::Finalize(static_cast<Module>(i));
        }
    }

    g_IsPcvControl = true;

    for (int i = 0; i < Module_Num; i++)
    {
        if (g_IsModuleInitialized[i])
        {
            detail::ClockResetController::pcv::Initialize(static_cast<Module>(i));
        }
    }
}

void Initialize(Module module) NN_NOEXCEPT
{
    NN_DETAIL_SDMMC_INITIALIZE_LOCK_GUARD;

    InitializeIfNeeded();

    g_IsModuleInitialized[module] = true;
    if (g_IsPcvControl)
    {
        detail::ClockResetController::pcv::Initialize(module);
    }
    else
    {
        detail::ClockResetController::reg::Initialize(module);
    }
}

void Finalize(Module module) NN_NOEXCEPT
{
    NN_DETAIL_SDMMC_INITIALIZE_LOCK_GUARD;

    if (!g_IsInitialized)
    {
        return; // まだ初期化されていない
    }

    if (g_IsPcvControl)
    {
        detail::ClockResetController::pcv::Finalize(module);
    }
    else
    {
        detail::ClockResetController::reg::Finalize(module);
    }
    g_IsModuleInitialized[module] = false;
}

bool IsAvailable(Module module) NN_NOEXCEPT
{
    NN_ABORT_UNLESS(g_IsInitialized);

    if (g_IsPcvControl)
    {
        return detail::ClockResetController::pcv::IsAvailable(module);
    }
    else
    {
        return detail::ClockResetController::reg::IsAvailable(module);
    }
}

void SetClockFrequencyKHz(uint32_t* pOutActualClockFrequencyKHz, Module module, uint32_t targetClockFrequencyKHz) NN_NOEXCEPT
{
    NN_ABORT_UNLESS(g_IsInitialized);

    if (g_IsPcvControl)
    {
        detail::ClockResetController::pcv::SetClockFrequencyKHz(pOutActualClockFrequencyKHz, module, targetClockFrequencyKHz);
    }
    else
    {
        detail::ClockResetController::reg::SetClockFrequencyKHz(pOutActualClockFrequencyKHz, module, targetClockFrequencyKHz);
    }
}

void ReleaseReset(Module module, uint32_t targetClockFrequencyKHz) NN_NOEXCEPT
{
    NN_ABORT_UNLESS(g_IsInitialized);

    if (g_IsPcvControl)
    {
        detail::ClockResetController::pcv::ReleaseReset(module, targetClockFrequencyKHz);
    }
    else
    {
        detail::ClockResetController::reg::ReleaseReset(module, targetClockFrequencyKHz);
    }
}

void AssertReset(Module module) NN_NOEXCEPT
{
    NN_ABORT_UNLESS(g_IsInitialized);

    if (g_IsPcvControl)
    {
        detail::ClockResetController::pcv::AssertReset(module);
    }
    else
    {
        detail::ClockResetController::reg::AssertReset(module);
    }
}

} // namespace ClockResetController {

} // namespace detail {
}} // namespace nn { namespace sdmmc {
