﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/psc.h>

namespace nn { namespace pinmux { namespace server {

// Gpio -> Pinmux -> GpioLow の順の依存でないと un-predictable glitches on output lines が発生する (NSBG-4121)
const nn::psc::PmModuleId PinmuxDependencies[]         = { nn::psc::PmModuleId_GpioLow };

enum StateEdge
{
    StateEdge_Unknown,
    StateEdge_FullAwakeToMinimumAwake,
    StateEdge_MinimumAwakeToSleepReady,
    StateEdge_SleepReadyToEssentialServicesSleepReady,
    StateEdge_EssentialServicesSleepReadyToEssentialServicesAwake,
    StateEdge_EssentialServicesAwakeToMinimumAwake,
    StateEdge_MinimumAwakeToFullAwake,
};

inline StateEdge GetStateEdge(nn::psc::PmState previousState, nn::psc::PmState state)
{
    if (previousState == nn::psc::PmState_FullAwake &&
        state == nn::psc::PmState_MinimumAwake)
    {
        return StateEdge_FullAwakeToMinimumAwake;
    }
    else if (previousState == nn::psc::PmState_MinimumAwake &&
             state         == nn::psc::PmState_SleepReady)
    {
        return StateEdge_MinimumAwakeToSleepReady;
    }
    else if (previousState == nn::psc::PmState_SleepReady &&
             state         == nn::psc::PmState_EssentialServicesSleepReady)
    {
        return StateEdge_SleepReadyToEssentialServicesSleepReady;
    }
    else if (previousState == nn::psc::PmState_EssentialServicesSleepReady &&
             state         == nn::psc::PmState_EssentialServicesAwake)
    {
        return StateEdge_EssentialServicesSleepReadyToEssentialServicesAwake;
    }
    else if (previousState == nn::psc::PmState_EssentialServicesAwake &&
             state         == nn::psc::PmState_MinimumAwake)
    {
        return StateEdge_EssentialServicesAwakeToMinimumAwake;
    }
    else if (previousState == nn::psc::PmState_MinimumAwake &&
             state         == nn::psc::PmState_FullAwake)
    {
        return StateEdge_MinimumAwakeToFullAwake;
    }
    else
    {
        return StateEdge_Unknown;
    }
}


/**
* @brief   サーバースレッドで使用するパラメータ設定オブジェクトです。
*
* @details
*   pPmModule          PmModule へのポインタ
*   pServerManager     HipcSimpleAllInOneServerManager を継承したサーバーマネージャーへのポインタ
*   exitIpcLoopState   IpcLoop 関数から抜けるための StateEdge
*/
template <typename ServerManagerT>
struct ServerThreadParam
{
    nn::psc::PmModule*  pPmModule;
    ServerManagerT*     pServerManager;
    StateEdge   exitIpcLoopState;
};

/**
* @brief psc からの特定イベント (エッジ判定) でループを抜ける IPC 用のループ関数です。
*
* @tparam ServerManagerT HipcSimpleAllInOneServerManager を継承したサーバーマネージャーを指定します。
* @tparam ModuleId psc のモジュールを指定します。
*
* @param[in] pPreviousState  1 つ前の State を保存している値へのポインタ
* @param[in] pState          現在の State を保存している値へのポインタ
* @param[in] pFlags          Pm の flag を保存している値へのポインタ
* @param[in] threadParam     サーバースレッドのパラメータを保存している構造体へのポインタ
*
* @details
*   psc からの特定イベント (エッジ判定) でループを抜ける IPC 用のループ関数です。@n
* 	条件に一致するまでは IPC メッセージ処理を行い、内部で psc からの通知に対する Acknowledge も行います。@n
* 	Acknowledge 以外の実処理は行いません。@n
* 	条件に一致すると Acknowledge することなく返ります。
*
*/
template <typename ServerManagerT, nn::psc::PmModuleId ModuleId>
void IpcLoop(nn::psc::PmState* pPreviousState,
    nn::psc::PmState* pState,
    nn::psc::PmFlagSet* pFlags,
    const ServerThreadParam<ServerManagerT>* threadParam) NN_NOEXCEPT
{
    nn::os::MultiWaitHolderType pmEventHolder;
    nn::os::InitializeMultiWaitHolder(
        &pmEventHolder,
        threadParam->pPmModule->GetEventPointer()->GetBase());
    nn::os::SetMultiWaitHolderUserData(
        &pmEventHolder,
        ModuleId); // PmModuleId は SF のタグと衝突しないことを保証している

    threadParam->pServerManager->AddUserWaitHolder(&pmEventHolder);

    while (NN_STATIC_CONDITION(true))
    {
        nn::os::MultiWaitHolderType* p = threadParam->pServerManager->Wait();

        switch (nn::os::GetMultiWaitHolderUserData(p))
        {
        // SF processing
        case ServerManagerT::HipcSimpleAllInOneServerManager::AcceptTag:
        case ServerManagerT::HipcSimpleAllInOneServerManager::InvokeTag:
            threadParam->pServerManager->ProcessAuto(p);
            break;

        case ModuleId:
        {
            bool tryWaitResult = threadParam->pPmModule->GetEventPointer()->TryWait();
            NN_SDK_ASSERT(tryWaitResult);
            NN_UNUSED(tryWaitResult);
            threadParam->pPmModule->GetEventPointer()->Clear();

            NN_ABORT_UNLESS_RESULT_SUCCESS(threadParam->pPmModule->GetRequest(pState, pFlags));
            auto stateEdge = GetStateEdge(*pPreviousState, *pState);

            // IPC 不可状態に入る条件
            if (stateEdge == threadParam->exitIpcLoopState)
            {
                return;
            }
            else
            {
                threadParam->pServerManager->AddUserWaitHolder(&pmEventHolder);
                threadParam->pPmModule->Acknowledge(*pState, nn::ResultSuccess());
                *pPreviousState = *pState;
                break;
            }
        }

        default: NN_UNEXPECTED_DEFAULT;
        }
    }
}

}}} // nn::pinmux::server
