﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/sf/sf_HipcServer.h>
#include <nn/sf/sf_ObjectFactory.h>
#include <nn/psc.h>

#include <nn/pinmux/pinmux_ServiceName.h>
#include <nn/pinmux/pinmux_IManager.sfdl.h>
#include <nn/pinmux/driver/pinmux_Suspend.h>

#include <nn/pinmux/server/pinmux_ManagerImpl.h>
#include <nn/pinmux/server/pinmux_Server.h>

#include "pinmux_ServerUtil.h"

namespace nn { namespace pinmux { namespace server {

namespace {

    // サーバー側から発行する全セッションの Max
    // TORIAEZU : 今予定されているセッションの数(4) + α(10)
    const int PinmuxSessionCountMax = 14;

    // TORIAEZU : 今はマネージャーは 1つだけ
    class PinmuxServerManager
        : public nn::sf::HipcSimpleAllInOneServerManager<PinmuxSessionCountMax, 1>
    {};

    PinmuxServerManager g_ServerManager;

    nn::sf::UnmanagedServiceObject<nn::pinmux::IManager, nn::pinmux::server::ManagerImpl>   g_ServiceObject;

    // スリープ等の通知を受け取るオブジェクト
    nn::psc::PmModule g_PmModule;

    bool g_IsHipcServerRegistered = false;
    bool g_IsHipcServerStarted = false;
}

nn::sf::SharedPointer<nn::pinmux::IManager> GetServiceObject() NN_NOEXCEPT
{
    return g_ServiceObject.GetShared();
}

void RegisterHipcServer() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(!g_IsHipcServerRegistered);

    // サービス名の登録とポートの初期化
    // 現在はどのプロセスからアクセスしても 1つのマネージャーにアクセスされる
    NN_ABORT_UNLESS_RESULT_SUCCESS(g_ServerManager.RegisterObjectForPort(g_ServiceObject.GetShared(), PinmuxSessionCountMax, nn::pinmux::PinmuxServiceName));

    g_IsHipcServerRegistered = true;
}

void StartHipcServer() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_IsHipcServerRegistered);
    NN_SDK_REQUIRES(!g_IsHipcServerStarted);

    // PSC へのモジュール登録
    auto result = g_PmModule.Initialize(
            nn::psc::PmModuleId_Pinmux,
            PinmuxDependencies,
            sizeof(PinmuxDependencies) / sizeof(PinmuxDependencies[0]),
            nn::os::EventClearMode_ManualClear);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    g_ServerManager.Start();

    g_IsHipcServerStarted = true;
}

void LoopHipcServer() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_IsHipcServerRegistered);
    NN_SDK_REQUIRES(g_IsHipcServerStarted);

    nn::psc::PmState    state         = nn::psc::PmState_Unknown;
    nn::psc::PmState    previousState = nn::psc::PmState_Unknown;
    nn::psc::PmFlagSet  flags;
    static const ServerThreadParam<PinmuxServerManager> threadParam =
    {
        &g_PmModule,
        &g_ServerManager,
        StateEdge_SleepReadyToEssentialServicesSleepReady
    };

    bool ipcEnable = true;

    while (NN_STATIC_CONDITION(true))
    {
        if (ipcEnable)
        {
            // IPC 許可状態
            IpcLoop<PinmuxServerManager, nn::psc::PmModuleId_Pinmux>(&previousState, &state, &flags, &threadParam);
        }
        else
        {
            // IPC 禁止状態 (PSC のイベントのみ待ち受ける)
            g_PmModule.GetEventPointer()->Wait();
            g_PmModule.GetEventPointer()->Clear();

            NN_ABORT_UNLESS_RESULT_SUCCESS(g_PmModule.GetRequest(&state, &flags));
        }

        auto stateEdge = GetStateEdge(previousState, state);
        switch (stateEdge)
        {
        case StateEdge_SleepReadyToEssentialServicesSleepReady:
            // Pinmux 設定値を保存する
            nn::pinmux::driver::Suspend();
            ipcEnable = false;
            break;

        case StateEdge_EssentialServicesSleepReadyToEssentialServicesAwake:
            // Pinmux 設定値を書き戻す
            nn::pinmux::driver::Resume();
            ipcEnable = true;
            break;

        default:
            // その他の State 変化時は IPC 可能
            ipcEnable = true;
            break;
        }

        g_PmModule.Acknowledge(state, nn::ResultSuccess());
        previousState = state;
    }
}

void StopHipcServer() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_IsHipcServerRegistered);
    NN_SDK_REQUIRES(g_IsHipcServerStarted);

    g_ServerManager.RequestStop();

    g_IsHipcServerStarted = false;
}

}}}
