﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   PINMUX ドライバの API インタフェース部分。
 */

#include <mutex>

#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_Abort.h>

#include <nn/dd.h>
#include <nn/os.h>

#include <nn/pinmux/driver/pinmux_Lib.h>
#include <nn/pinmux/driver/pinmux_Api.h>
#include <nn/pinmux/driver/pinmux_ApiForHardwareTest.h>
#include <nn/pinmux/pinmux_Type.h>
#include <nn/pinmux/detail/pinmux_Log.h>

#include <nne/pinmux/pinmux.h>

#include "detail/pinmux_GpioRegAccessor-soc.tegra.h"
#include "detail/pinmux_CombinationList-soc.tegra.h"
#include "detail/pinmux_DdUtil.h"
#include "detail/pinmux_SetSdmmc1Mode-soc.tegra-x1.h"

#if defined(NN_BUILD_CONFIG_HARDWARE_NX)
#include <nn/spl/spl_Api.h>
#include "detail/pinmux_InitialConfig-hardware.icosa.h"
#include "detail/pinmux_InitialConfig-hardware.copper.h"
#include "detail/pinmux_InitialConfig-hardware.iowa.h"
#include "detail/pinmux_InitialConfig-hardware.hoag.h"
#include "detail/pinmux_InitialDrivePadConfig-hardware.nx.h"
#include "detail/pinmux_DeepSleepConfig-hardware.icosa.h"
#include "detail/pinmux_DeepSleepConfig-hardware.copper.h"
#include "detail/pinmux_DeepSleepConfig-hardware.iowa.h"
#include "detail/pinmux_DeepSleepConfig-hardware.hoag.h"
#include "detail/pinmux_SetExtConMode-soc.tegra-x1.h"
#endif

namespace {

// Initialize の参照カウント
int g_InitializeCount = 0;

// 参照カウントを守る Mutex
struct StaticMutex
{
    nn::os::MutexType mutex;
    void lock() NN_NOEXCEPT
    {
        nn::os::LockMutex(&mutex);
    }
    void unlock() NN_NOEXCEPT
    {
        nn::os::UnlockMutex(&mutex);
    }
} g_InitializeCountMutex = { NN_OS_MUTEX_INITIALIZER(false) };

// VirtualAddress
uintptr_t g_GpioVirtualAddress;

void OpenSessionImpl(nn::pinmux::driver::PinmuxSession* pOutSession, int name) NN_NOEXCEPT
{
    // 単純に enum の値を int にして入れておくだけ
    pOutSession->_internalNumber = name;
}

void CloseSessionImpl(nn::pinmux::driver::PinmuxSession* pSession) NN_NOEXCEPT
{
    // 今のところ特に何もしない
    NN_UNUSED(pSession);
}

#if defined(NN_BUILD_CONFIG_HARDWARE_NX)
nn::spl::HardwareType GetHardwareType() NN_NOEXCEPT
{
    nn::Bit64 config;
    nn::spl::Initialize();
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::spl::GetConfig(&config, nn::spl::ConfigItem_HardwareType));
    nn::spl::Finalize();
    return static_cast<nn::spl::HardwareType>(config);
}
#endif

void SetSleepConfig() NN_NOEXCEPT
{
#if defined(NN_BUILD_CONFIG_HARDWARE_NX)
    switch(GetHardwareType())
    {
    case nn::spl::HardwareType_Icosa :
        nne::pinmux::tegra::UpdatePinmuxPadList(nne::pinmux::tegra::tk2::deepSleepConfigListForIcosa);
        NN_DETAIL_PINMUX_TRACE("Set Deep Sleep Config\n");
        break;

    case nn::spl::HardwareType_Copper :
        nne::pinmux::tegra::UpdatePinmuxPadList(nne::pinmux::tegra::tk2::deepSleepConfigListForCopper);
        NN_DETAIL_PINMUX_TRACE("Set Deep Sleep Config For Copper\n");
        break;

    case nn::spl::HardwareType_Hoag :
        nne::pinmux::tegra::UpdatePinmuxPadList(nne::pinmux::tegra::tk2::deepSleepConfigListForHoag);
        NN_DETAIL_PINMUX_TRACE("Set Deep Sleep Config For Hoag\n");
        break;

    case nn::spl::HardwareType_Iowa :
        nne::pinmux::tegra::UpdatePinmuxPadList(nne::pinmux::tegra::tk2::deepSleepConfigListForIowa);
        NN_DETAIL_PINMUX_TRACE("Set Deep Sleep Config For Iowa\n");
        break;

    default : NN_UNEXPECTED_DEFAULT;
    }
#endif
}

// TORIAEZU : gcc ビルドで未使用関数で怒られるのでビルド分岐。SoC が tegra なら使用可能。
#if defined(NN_BUILD_CONFIG_SPEC_NX)
void SetGpioMode(int gpioNumber, bool isGpioMode)
{
    // GPIO 番号を アクセスするアドレス・ビットへ変換
    nn::Bit32* accessAddress = GetGpioRegAccessAddress(nn::pinmux::driver::detail::GpioRegisterType_GPIO_CNF, gpioNumber, g_GpioVirtualAddress);
    int bitPosition      = nn::pinmux::driver::detail::ConvertInternalGpioPadNumberToBitPosition(gpioNumber);

    if(isGpioMode)
    {
        nn::pinmux::driver::detail::SetBitForTegraMaskedWrite( nn::pinmux::driver::detail::GpioCnf_Gpio, bitPosition, accessAddress);
    }
    else
    {
        nn::pinmux::driver::detail::SetBitForTegraMaskedWrite( nn::pinmux::driver::detail::GpioCnf_Sfio, bitPosition, accessAddress);
    }
    // Ensure Register Write
    nn::pinmux::driver::detail::DummyRead(accessAddress);
}

bool IsGpioMode(int gpioNumber)
{
    // GPIO 番号を アクセスするアドレス・ビットへ変換
    nn::Bit32* accessAddress = GetGpioRegAccessAddress(nn::pinmux::driver::detail::GpioRegisterType_GPIO_CNF, gpioNumber, g_GpioVirtualAddress);
    int bitPosition      = nn::pinmux::driver::detail::ConvertInternalGpioPadNumberToBitPosition(gpioNumber);

    if(nn::pinmux::driver::detail::GetBit<nn::Bit32>(accessAddress, bitPosition))
    {
        return true;
    }
    else
    {
        return false;
    }
}

#endif

}

namespace nn {
namespace pinmux {
namespace driver {

void Initialize() NN_NOEXCEPT
{
    std::lock_guard<StaticMutex> lock(g_InitializeCountMutex);

    if(g_InitializeCount == 0)
    {
        g_GpioVirtualAddress = nn::dd::QueryIoMappingAddress(detail::GpioPhysicalAddress, detail::GpioAddressSize);
        if (g_GpioVirtualAddress == 0)
        {
            // physicalAddress が指す I/O アドレスがマッピングされていない
            NN_ABORT("I/O registers for 0x%llx are not mapped. Make sure the capability setting is properly set for this process.\n", detail::GpioPhysicalAddress);
        }

        if(!nne::pinmux::tegra::IsInitialized())
        {
            nne::pinmux::tegra::Initialize();
        }

    }

    g_InitializeCount++;
}

void Finalize() NN_NOEXCEPT
{
    std::lock_guard<StaticMutex> lock(g_InitializeCountMutex);

    NN_SDK_ASSERT(g_InitializeCount > 0);

    g_InitializeCount--;
    if(g_InitializeCount == 0)
    {
        nne::pinmux::tegra::Finalize();
    }
}

void OpenSession(PinmuxSession* pOutSession, AssignablePinGroupName name) NN_NOEXCEPT
{
    OpenSessionImpl(pOutSession, static_cast<int>(name));
}

void CloseSession(PinmuxSession* pSession) NN_NOEXCEPT
{
    CloseSessionImpl(pSession);
}

void SetPinAssignment(PinmuxSession* pSession, PinAssignment assignment) NN_NOEXCEPT
{
    // 組み合わせのチェック
    NN_SDK_REQUIRES(detail::IsSupportPinAssignment(pSession->_internalNumber, static_cast<int>(assignment)));
    NN_ABORT_UNLESS(detail::IsSupportPinAssignment(pSession->_internalNumber, static_cast<int>(assignment)));

    // SDMMC1 の特殊な機能切り替えのみの特別対応
    if(assignment == PinAssignment_Sdmmc1OutputHigh
        || assignment == PinAssignment_Sdmmc1ResetState)
    {
        detail::SetSdmmc1Mode(assignment ,g_GpioVirtualAddress);
        return;
    }

    for(int i=0; i<detail::PinmuxGroupList[pSession->_internalNumber].number; i++)
    {
        nne::pinmux::tegra::PinmuxPadConfig pinmuxConfig;

        pinmuxConfig.index   = detail::PinmuxGroupList[pSession->_internalNumber].pCombination[i].pinmuxNumber;
        int gpioNumber       = detail::PinmuxGroupList[pSession->_internalNumber].pCombination[i].gpioNumber;

        // TODO : 設定すべき Pin 設定が増えたら、ここも切り出して関数化する
        switch(assignment)
        {
#if defined(NN_BUILD_CONFIG_HARDWARE_NX)
        case PinAssignment_ExtConTxUart:
            SetGpioMode(gpioNumber, false);
            detail::SetExtConUartMode(pinmuxConfig.index, true);
            break;

        case PinAssignment_ExtConTxGpio:
            SetGpioMode(gpioNumber, true);
            detail::SetExtConUartMode(pinmuxConfig.index, false);
            break;
#endif

#if defined(NN_BUILD_CONFIG_SPEC_NX)
        case PinAssignment_PwmFanGpio:
            SetGpioMode(gpioNumber, true);
            pinmuxConfig.option = nne::pinmux::tegra::PinmuxOpt_Gpio |
                                  nne::pinmux::tegra::PinmuxOpt_Input |
                                  nne::pinmux::tegra::PinmuxOpt_NoPupd;
            pinmuxConfig.optionMask = nne::pinmux::tegra::PinmuxOptBitMask_Pm |
                                      nne::pinmux::tegra::PinmuxOptBitMask_Dir |
                                      nne::pinmux::tegra::PinmuxOptBitMask_Pupd;
            nne::pinmux::tegra::UpdateSinglePinmuxPad(pinmuxConfig);
            break;

        case PinAssignment_PwmFanPwm:
            SetGpioMode(gpioNumber, false);
            pinmuxConfig.option = nne::pinmux::tegra::tx1::LcdGpio2::Pm_Pwm1 |
                                  nne::pinmux::tegra::PinmuxOpt_Output |
                                  nne::pinmux::tegra::PinmuxOpt_NoPupd;
            pinmuxConfig.optionMask = nne::pinmux::tegra::PinmuxOptBitMask_Pm |
                                      nne::pinmux::tegra::PinmuxOptBitMask_Dir |
                                      nne::pinmux::tegra::PinmuxOptBitMask_Pupd;
            nne::pinmux::tegra::UpdateSinglePinmuxPad(pinmuxConfig);
            break;

#endif
        case PinAssignment_Sdmmc1SchmtEnable:
            pinmuxConfig.option = nne::pinmux::tegra::PinmuxOpt_SchmittTrigger;
            pinmuxConfig.optionMask = nne::pinmux::tegra::PinmuxOptBitMask_Schmt;
            nne::pinmux::tegra::UpdateSinglePinmuxPad(pinmuxConfig);
            break;

        case PinAssignment_Sdmmc1SchmtDisable :
            pinmuxConfig.option = nne::pinmux::tegra::PinmuxOpt_CmosMode;
            pinmuxConfig.optionMask = nne::pinmux::tegra::PinmuxOptBitMask_Schmt;
            nne::pinmux::tegra::UpdateSinglePinmuxPad(pinmuxConfig);
            break;

        default: NN_UNEXPECTED_DEFAULT;
        }

#if defined(NN_BUILD_CONFIG_SPEC_GENERIC)
        // TORIAEZU : 現時点で Generic の時には使われないので、UNUSED にしておく
        NN_UNUSED(gpioNumber);
#endif

    }
}

PinAssignment GetPinAssignment(PinmuxSession* pSession) NN_NOEXCEPT
{
    switch(static_cast<AssignablePinGroupName>(pSession->_internalNumber))
    {
#if defined(NN_BUILD_CONFIG_SPEC_NX)
    case AssignablePinGroupName_ExtConUTx:
    case AssignablePinGroupName_ExtConSTx:
        {
            //int pinmuxNumber = detail::PinmuxGroupList[pSession->_internalNumber].pCombination[0].pinmuxNumber;
            int gpioNumber   = detail::PinmuxGroupList[pSession->_internalNumber].pCombination[0].gpioNumber;

            if(IsGpioMode(gpioNumber))
            {
                return PinAssignment_ExtConTxGpio;
            }
            else
            {
                // 念の為、UART になっているかチェック
                // 今は初期設定が下記設定になっていないため、disable にしておく
                //NN_SDK_ASSERT( (nne::pinmux::tegra::tx1::Uart2Tx::Pm_Uartb &
                //    nne::pinmux::tegra::PinmuxOptBitMask_Pm) ==
                //(nne::pinmux::tegra::ReadRegPad(pinmuxNumber) &
                //nne::pinmux::tegra::PinmuxOptBitMask_Pm));

                return PinAssignment_ExtConTxUart;
            }
        }
        break;

    case AssignablePinGroupName_PwmFan:
        {
            int gpioNumber   = detail::PinmuxGroupList[pSession->_internalNumber].pCombination[0].gpioNumber;

            if(IsGpioMode(gpioNumber))
            {
                return PinAssignment_PwmFanGpio;
            }
            else
            {
                return PinAssignment_PwmFanPwm;
            }
        }
        break;

#endif

    case AssignablePinGroupName_Sdmmc1:
        NN_SDK_ASSERT("This AssignablePinGroupName does not support this function\n");
        return static_cast<PinAssignment>(-1);

    default: NN_UNEXPECTED_DEFAULT;
    }
}

void OpenSessionForHardwareTest(PinmuxSession* pOutSession, AssignablePinGroupNameForHardwareTest name) NN_NOEXCEPT
{
    OpenSessionImpl(pOutSession, static_cast<int>(name));
}

void CloseSessionForHardwareTest(PinmuxSession* pSession) NN_NOEXCEPT
{
    CloseSessionImpl(pSession);
}

void SetPinAssignmentForHardwareTest(PinmuxSession* pSession, PinAssignmentForHardwareTest assignment) NN_NOEXCEPT
{
    // 組み合わせのチェック
    NN_SDK_REQUIRES(detail::IsSupportPinAssignment(pSession->_internalNumber, static_cast<int>(assignment)));
    NN_ABORT_UNLESS(detail::IsSupportPinAssignment(pSession->_internalNumber, static_cast<int>(assignment)));

#if defined(NN_BUILD_CONFIG_SPEC_NX)
    // TORIAEZU : 設定値をここにべた書き。工程検査用にピン設定が増えたら、ちゃんとする。
    switch(assignment)
    {
    case PinAssignmentForHardwareTest_UartDHiZ:

        for(int i=0; i<detail::NumberOfUartDPin; i++)
        {
            nne::pinmux::tegra::PinmuxPadConfig pinmuxConfig;

            pinmuxConfig.index   = detail::UartDNumbers[i].pinmuxNumber;

            pinmuxConfig.option =     nne::pinmux::tegra::PinmuxOpt_Gpio  |
                                      nne::pinmux::tegra::PinmuxOpt_Input |
                                      nne::pinmux::tegra::PinmuxOpt_NoPupd;
            pinmuxConfig.optionMask = nne::pinmux::tegra::PinmuxOptBitMask_Pm  |
                                      nne::pinmux::tegra::PinmuxOptBitMask_Dir |
                                      nne::pinmux::tegra::PinmuxOptBitMask_Pupd;
            nne::pinmux::tegra::UpdateSinglePinmuxPad(pinmuxConfig);
        }
        break;

    case PinAssignmentForHardwareTest_UartDEnable:

        for(int i=0; i<detail::NumberOfUartDPin; i++)
        {
            nne::pinmux::tegra::PinmuxPadConfig pinmuxConfig;

            pinmuxConfig.index   = detail::UartDNumbers[i].pinmuxNumber;

            switch(pinmuxConfig.index)
            {
            case nne::pinmux::tegra::tx1::PinmuxPadIndex_Uart4Tx:
                pinmuxConfig.option =     nne::pinmux::tegra::Uart4Tx::Pm_Uartd  |
                                          nne::pinmux::tegra::PinmuxOpt_Output   |
                                          nne::pinmux::tegra::PinmuxOpt_PullDown ;
                pinmuxConfig.optionMask = nne::pinmux::tegra::PinmuxOptBitMask_Pm |
                                          nne::pinmux::tegra::PinmuxOptBitMask_Dir|
                                          nne::pinmux::tegra::PinmuxOptBitMask_Pupd;
                break;

            case nne::pinmux::tegra::tx1::PinmuxPadIndex_Uart4Rx:
                pinmuxConfig.option =     nne::pinmux::tegra::Uart4Rx::Pm_Uartd  |
                                          nne::pinmux::tegra::PinmuxOpt_Input    |
                                          nne::pinmux::tegra::PinmuxOpt_PullDown ;
                pinmuxConfig.optionMask = nne::pinmux::tegra::PinmuxOptBitMask_Pm  |
                                          nne::pinmux::tegra::PinmuxOptBitMask_Dir |
                                          nne::pinmux::tegra::PinmuxOptBitMask_Pupd;
                break;

            case nne::pinmux::tegra::tx1::PinmuxPadIndex_Uart4Rts:
                pinmuxConfig.option =     nne::pinmux::tegra::Uart4Rts::Pm_Uartd  |
                                          nne::pinmux::tegra::PinmuxOpt_Output    |
                                          nne::pinmux::tegra::PinmuxOpt_PullDown  ;
                pinmuxConfig.optionMask = nne::pinmux::tegra::PinmuxOptBitMask_Pm |
                                          nne::pinmux::tegra::PinmuxOptBitMask_Dir|
                                          nne::pinmux::tegra::PinmuxOptBitMask_Pupd;
                break;

            case nne::pinmux::tegra::tx1::PinmuxPadIndex_Uart4Cts:
                pinmuxConfig.option =     nne::pinmux::tegra::Uart4Cts::Pm_Uartd  |
                                          nne::pinmux::tegra::PinmuxOpt_Input     |
                                          nne::pinmux::tegra::PinmuxOpt_PullDown  ;
                pinmuxConfig.optionMask = nne::pinmux::tegra::PinmuxOptBitMask_Pm  |
                                          nne::pinmux::tegra::PinmuxOptBitMask_Dir |
                                          nne::pinmux::tegra::PinmuxOptBitMask_Pupd;
                break;

            default: NN_UNEXPECTED_DEFAULT;

            }

            nne::pinmux::tegra::UpdateSinglePinmuxPad(pinmuxConfig);
        }
        break;

    default: NN_UNEXPECTED_DEFAULT;
    }
#endif
}

//以下は driver にのみ存在する非公開 API
// For Bus Process
void Suspend() NN_NOEXCEPT
{
    NN_DETAIL_PINMUX_TRACE("Sampling all pinmux setting\n");

    // 現在のピン設定をサンプリングする
    nne::pinmux::tegra::Suspend();

    // Suspend で値をサンプリングした後、Sleep 用の Pin 設定に変更する
#if defined(NN_BUILD_CONFIG_HARDWARE_NX)
    // ExtCon はスリープ時の wake event を取れるように UART を GPIO に変更する
    const int ExtConUTxGpioNumber = NN_GPIO_GET_PAD_NUM(nn::pinmux::driver::detail::GpioPadPort_PG, 0);
    const int ExtConSTxGpioNumber = NN_GPIO_GET_PAD_NUM(nn::pinmux::driver::detail::GpioPadPort_PD, 1);

    detail::SetBitForTegraMaskedWrite( detail::GpioCnf_Gpio,
                                       detail::ConvertInternalGpioPadNumberToBitPosition(ExtConUTxGpioNumber),
                                       GetGpioRegAccessAddress(detail::GpioRegisterType_GPIO_CNF, ExtConUTxGpioNumber, g_GpioVirtualAddress)
                                      );
    detail::DummyRead(GetGpioRegAccessAddress(detail::GpioRegisterType_GPIO_CNF, ExtConUTxGpioNumber, g_GpioVirtualAddress));

    detail::SetBitForTegraMaskedWrite( detail::GpioCnf_Gpio,
                                       detail::ConvertInternalGpioPadNumberToBitPosition(ExtConSTxGpioNumber),
                                       GetGpioRegAccessAddress(detail::GpioRegisterType_GPIO_CNF, ExtConSTxGpioNumber, g_GpioVirtualAddress)
                                      );
    detail::DummyRead(GetGpioRegAccessAddress(detail::GpioRegisterType_GPIO_CNF, ExtConSTxGpioNumber, g_GpioVirtualAddress));
#endif

    SetSleepConfig();
}

void Resume() NN_NOEXCEPT
{
    NN_DETAIL_PINMUX_TRACE("Restore all pinmux setting\n");
    nne::pinmux::tegra::Resume();
}

// For boot Process
void SetInitialConfig() NN_NOEXCEPT
{
#if defined(NN_BUILD_CONFIG_HARDWARE_NX)

    switch(GetHardwareType())
    {
    case nn::spl::HardwareType_Icosa :
        nne::pinmux::tegra::UpdatePinmuxPadList(nne::pinmux::tegra::tk2::initConfigListForIcosa);
        NN_DETAIL_PINMUX_TRACE("Set Initial Config\n");
        break;

    case nn::spl::HardwareType_Copper :
        nne::pinmux::tegra::UpdatePinmuxPadList(nne::pinmux::tegra::tk2::initConfigListForCopper);
        NN_DETAIL_PINMUX_TRACE("Set Initial Config For Copper\n");
        break;

    case nn::spl::HardwareType_Hoag :
        nne::pinmux::tegra::UpdatePinmuxPadList(nne::pinmux::tegra::tk2::initConfigListForHoag);
        // Mariko での TORIAEZU E_SCHMT 設定を踏襲
        for (int i = 0; i < detail::NumberOfSdmmc2Pin; i++)
        {
            nne::pinmux::tegra::PinmuxPadConfig pinmuxConfig;

            pinmuxConfig.index = detail::Sdmmc2Numbers[i].pinmuxNumber;
            pinmuxConfig.option = nne::pinmux::tegra::PinmuxOpt_SchmittTrigger;
            pinmuxConfig.optionMask = nne::pinmux::tegra::PinmuxOptBitMask_Schmt;
            nne::pinmux::tegra::UpdateSinglePinmuxPad(pinmuxConfig);
        }
        NN_DETAIL_PINMUX_TRACE("Set Initial Config For Hoag\n");
        break;

    case nn::spl::HardwareType_Iowa :
        nne::pinmux::tegra::UpdatePinmuxPadList(nne::pinmux::tegra::tk2::initConfigListForIowa);
        // TORIAEZU : Odin Mariko では起動時に SDMMC2 の E_SCHMT を enable にする必要がある
        // E_SCHMT を設定するピンが増えたら初期設定へ追加する
        for( int i = 0; i < detail::NumberOfSdmmc2Pin; i++ )
        {
            nne::pinmux::tegra::PinmuxPadConfig pinmuxConfig;

            pinmuxConfig.index   = detail::Sdmmc2Numbers[i].pinmuxNumber;
            pinmuxConfig.option = nne::pinmux::tegra::PinmuxOpt_SchmittTrigger;
            pinmuxConfig.optionMask = nne::pinmux::tegra::PinmuxOptBitMask_Schmt;
            nne::pinmux::tegra::UpdateSinglePinmuxPad(pinmuxConfig);
        }
        NN_DETAIL_PINMUX_TRACE("Set Initial Config For Iowa\n");
        break;

    default : NN_UNEXPECTED_DEFAULT;
    }

#else
    NN_DETAIL_PINMUX_TRACE("Not set Initial Config\n");
#endif

}

void SetInitialDrivePadConfig() NN_NOEXCEPT
{
#if defined(NN_BUILD_CONFIG_HARDWARE_NX)
    nne::pinmux::tegra::UpdateDrivePadList(nne::pinmux::tegra::tx1::initDrivePadConfigList);

    NN_DETAIL_PINMUX_TRACE("Set DrivePad Initial Config\n");
#else
    NN_DETAIL_PINMUX_TRACE("Not set DrivePad Initial Config\n");
#endif
}

} // driver
} // pinmux
} // nn
