﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/ovln/ovln_SenderForOverlay.h>

#include <nn/nn_Common.h>
#include <nn/nn_StaticAssert.h>

#include <atomic>
#include <nn/ovln/ovln_Sender.h>
#include <new>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/nn_Abort.h>
#include <nn/nn_SdkAssert.h>
#include <nn/util/util_IntUtil.h>

#include "ovln_OverlayCommonInternal.h"

#include <nn/nn_SdkLog.h>

namespace nn { namespace ovln {

Result InitializeSenderLibraryForOverlay() NN_NOEXCEPT
{
    return InitializeSenderLibrary();
}

void FinalizeSenderLibraryForOverlay() NN_NOEXCEPT
{
    FinalizeSenderLibrary();
}

namespace {

struct SenderForOverlay
{
    Sender sender;
    SendOption sendOption;
};
NN_STATIC_ASSERT(sizeof(SenderForOverlay) <= sizeof(SenderForOverlayType));

Result InitializeSenderImpl(SenderForOverlayType* pSender, const QueueAttribute& queueAttribute, const SendOption& sendOption) NN_NOEXCEPT
{
    auto p = new (pSender) SenderForOverlay();
    NN_UTIL_SCOPE_EXIT
    {
        if (p)
        {
            p->~SenderForOverlay();
        }
    };

    NN_RESULT_DO(p->sender.Initialize(ovln::detail::GetOverlaySourceName(), queueAttribute));

    p->sendOption = sendOption;
    p = nullptr;
    NN_RESULT_SUCCESS;
}

}

Result InitializeSenderForValue(SenderForOverlayType* pSender) NN_NOEXCEPT
{
    QueueAttribute queueAttribute = {};
    queueAttribute.queueLength = 1;
    SendOption sendOption = {};
    sendOption.enqueuePosition = EnqueuePosition_Front;
    sendOption.overflowOption = OverflowOption_RemoveFront;
    return InitializeSenderImpl(pSender, queueAttribute, sendOption);
}

Result InitializeSenderWithQueue(SenderForOverlayType* pSender, int queueLength) NN_NOEXCEPT
{
    QueueAttribute queueAttribute = {};
    NN_SDK_REQUIRES(util::IsIntValueRepresentable<decltype(queueAttribute.queueLength)>(queueLength));
    queueAttribute.queueLength = static_cast<decltype(queueAttribute.queueLength)>(queueLength);
    SendOption sendOption = {};
    sendOption.enqueuePosition = EnqueuePosition_Back;
    sendOption.overflowOption = OverflowOption_Error;
    return InitializeSenderImpl(pSender, queueAttribute, sendOption);
}

void FinalizeSender(SenderForOverlayType* pSender) NN_NOEXCEPT
{
    auto p = reinterpret_cast<SenderForOverlay*>(pSender);
    p->sender.Finalize();
    p->~SenderForOverlay();
}

bool Send(SenderForOverlayType* pSender, const Message& message) NN_NOEXCEPT
{
    auto p = reinterpret_cast<SenderForOverlay*>(pSender);
    return p->sender.Send(message, p->sendOption);
}

uint32_t GetUnreceivedMessageCount(const SenderForOverlayType* pSender) NN_NOEXCEPT
{
    auto p = reinterpret_cast<const SenderForOverlay*>(pSender);
    return p->sender.GetUnreceivedMessageCount();
}

}}
