﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/ovln/ovln_Sender.h>

#include <nn/ovln/ovln_Services.sfdl.h>
#include <nn/nn_SdkAssert.h>
#include <nn/result/result_HandlingUtility.h>
#include <utility>
#include <nn/ovln/ovln_ResultPrivate.h>
#include <cstring>
#include <nn/sf/sf_ShimLibraryUtility.h>
#include <type_traits>

namespace nn { namespace ovln {

namespace detail {

sf::SharedPointer<ISenderService> GetSenderServicePrepared() NN_NOEXCEPT;

} // detail

namespace {

sf::ShimLibraryObjectHolder<ISenderService> g_SenderService = NN_SF_SHIM_LIBRARY_OBJECT_HOLDER_INITIALIZER;

} // anon

Result InitializeSenderLibrary() NN_NOEXCEPT
{
    #if defined(NN_OVLN_USES_DIRECT_AGENT)
        return g_SenderService.InitializeHolder([] (sf::SharedPointer<ISenderService>* pOut) -> Result
        {
            *pOut = detail::GetSenderServicePrepared();
            NN_RESULT_SUCCESS;
        });
    #elif defined(NN_OVLN_USES_HIPC_AGENT)
        static nn::sf::SimpleAllInOneHipcSubDomainClientManager<10> g_HipcSub = NN_SF_SIMPLE_ALL_IN_ONE_HIPC_SUB_DOMAIN_CLIENT_MANAGER_INITIALIZER;
        return g_HipcSub.InitializeShimLibraryHolder(&g_SenderService, "ovln:snd");
    #else
        #error "not supported"
    #endif
}

void FinalizeSenderLibrary() NN_NOEXCEPT
{
    g_SenderService.FinalizeHolder();
}

sf::ShimLibraryObjectHolder<ISenderService>& GetSenderServiceShimLibraryObjectHolder() NN_NOEXCEPT
{
    return g_SenderService;
}

Sender::Sender() NN_NOEXCEPT
    : m_pSender(nullptr)
{
}

bool Sender::IsValid() const NN_NOEXCEPT
{
    return m_pSender != nullptr;
}

Result Sender::Initialize(const SourceName& sourceName, const QueueAttribute& queueAttribute) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(!IsValid());

    auto service = g_SenderService.GetObject();
    sf::SharedPointer<ISender> sender;
    NN_RESULT_DO(service->OpenSender(&sender, sourceName, queueAttribute));

    this->m_pSender = sender.Detach();

    NN_SDK_ASSERT(IsValid());
    NN_RESULT_SUCCESS;
}

void Sender::Finalize() NN_NOEXCEPT
{
    if (IsValid())
    {
        sf::ReleaseSharedObject(m_pSender);
        this->m_pSender = nullptr;
    }
    NN_SDK_ASSERT(!IsValid());
}

Sender::~Sender() NN_NOEXCEPT
{
    Finalize();
}

void Sender::swap(Sender& other) NN_NOEXCEPT
{
    using std::swap;
    swap(this->m_pSender, other.m_pSender);
}

Sender::Sender(Sender&& other) NN_NOEXCEPT
    : m_pSender(nullptr)
{
    NN_SDK_ASSERT(!IsValid());
    this->swap(other);
}

Sender& Sender::operator=(Sender&& rhs) NN_NOEXCEPT
{
    Sender(std::move(rhs)).swap(*this);
    return *this;
}

bool Sender::Send(Message message, SendOption option)
{
    NN_SDK_REQUIRES(IsValid());
    NN_SDK_REQUIRES_LESS_EQUAL(message.dataSize, sizeof(message.data));

    RawMessage rawMessage;
    rawMessage.tag = message.tag;
    rawMessage.dataSize = message.dataSize;
    NN_SDK_ASSERT(message.dataSize <= sizeof(rawMessage.data));
    std::memcpy(&rawMessage.data, &message.data, message.dataSize);

    NN_RESULT_TRY(m_pSender->Send(rawMessage, option))
        NN_RESULT_CATCH(ovln::ResultQueueIsFull)
        {
            return false;
        }
    NN_RESULT_END_TRY

    return true;
}

uint32_t Sender::GetUnreceivedMessageCount() const NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsValid());
    uint32_t ret;
    NN_ABORT_UNLESS_RESULT_SUCCESS(m_pSender->GetUnreceivedMessageCount(&ret));
    return ret;
}

}}
