﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/ovln/ovln_NotificationAgent.h>

#include <nn/ovln/ovln_Services.sfdl.h>
#include <nn/sf/sf_ObjectFactory.h>
#include <nn/util/util_IntrusiveList.h>
#include <nn/nn_SdkAssert.h>
#include <nn/util/util_Optional.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <utility>
#include <nn/util/util_Exchange.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/sf/sf_Types.h>
#include <nn/nn_Allocator.h>
#include <nn/sf/sf_MemoryResource.h>
#include <new>
#include <algorithm>
#include <nn/os/os_Mutex.h>
#include <mutex>
#include <nn/ovln/ovln_ResultPrivate.h>
#include <nn/os/os_Tick.h>

namespace nn { namespace ovln {

class NotificationAgent
{
};

namespace {

template <typename List>
static void MoveAllToFront(List* pTarget, List* pOther) NN_NOEXCEPT
{
    pTarget->splice(pTarget->begin(), *pOther);
}

class NotificationAgentImpl
{
private:

    typedef sf::ObjectFactory<sf::MemoryResourceAllocationPolicy> SfObjectFactory;
    typedef int64_t TimeStamp;

    class MessageHolder
        : public util::IntrusiveListBaseNode<MessageHolder>
    {
    public:

        RawMessage rawMessage;
        TimeStamp timeStamp;
        os::Tick tick;

    };

    typedef util::IntrusiveList<MessageHolder, util::IntrusiveListBaseNodeTraits<MessageHolder>> MessageHolderList;

    class Receiver;
    class QueueHolder;

    class Source
        : public sf::ISharedObject
        , public util::IntrusiveListBaseNode<Source>
    {
    private:

        util::IntrusiveList<QueueHolder, util::IntrusiveListBaseNodeTraits<QueueHolder>> m_QueueHolders;
        Receiver* m_pReceiver;
        int m_ItemCount;

    public:

        Source() NN_NOEXCEPT
            : m_pReceiver(nullptr)
            , m_ItemCount(0)
        {
        }

        ~Source() NN_NOEXCEPT
        {
            NN_SDK_REQUIRES(m_QueueHolders.empty());
        }

        void AddQueueHolder(QueueHolder* pQueueHolder) NN_NOEXCEPT
        {
            NN_SDK_REQUIRES(!pQueueHolder->CanPop());
            m_QueueHolders.push_back(*pQueueHolder);
        }

        void RemoveQueueHolder(QueueHolder* pQueueHolder) NN_NOEXCEPT
        {
            NN_SDK_REQUIRES(!pQueueHolder->CanPop());
            m_QueueHolders.erase(m_QueueHolders.iterator_to(*pQueueHolder));
        }

        bool AttachReceiver(Receiver* pReceiver) NN_NOEXCEPT
        {
            NN_SDK_REQUIRES(pReceiver);
            if (m_pReceiver)
            {
                return false;
            }
            this->m_pReceiver = pReceiver;
            return true;
        }

        void DetachReceiver(Receiver* pReceiver) NN_NOEXCEPT
        {
            NN_SDK_REQUIRES(m_pReceiver == pReceiver);
            NN_UNUSED(pReceiver);
            this->m_pReceiver = nullptr;
        }

        int GetItemCount() const NN_NOEXCEPT
        {
            return m_ItemCount;
        }

        void IncreaseItemCount(int n) NN_NOEXCEPT
        {
            NN_SDK_REQUIRES(n >= 0);
            NN_SDK_ASSERT(m_ItemCount >= 0);
            if (n == 0)
            {
                return;
            }
            this->m_ItemCount += n;
            if (m_pReceiver)
            {
                m_pReceiver->IncreaseItemCount(n);
            }
        }

        void DecreaseItemCount(int n) NN_NOEXCEPT
        {
            NN_SDK_REQUIRES(n >= 0);
            NN_SDK_REQUIRES(m_ItemCount >= n);
            if (n == 0)
            {
                return;
            }
            this->m_ItemCount -= n;
            if (m_pReceiver)
            {
                m_pReceiver->DecreaseItemCount(n);
            }
        }

        util::IntrusiveList<QueueHolder, util::IntrusiveListBaseNodeTraits<QueueHolder>>& GetQueueHolders() NN_NOEXCEPT
        {
            return m_QueueHolders;
        }

        const util::IntrusiveList<QueueHolder, util::IntrusiveListBaseNodeTraits<QueueHolder>>& GetQueueHolders() const NN_NOEXCEPT
        {
            return m_QueueHolders;
        }

    };

    class Queue
    {
    private:

        MessageHolderList m_Queue;
        MessageHolderList m_FreeList;

        MessageHolder* AllocateImpl(const RawMessage& rawMessage, TimeStamp timeStamp) NN_NOEXCEPT
        {
            NN_SDK_REQUIRES(CanPush());
            auto p = &m_FreeList.front();
            m_FreeList.pop_front();
            p->rawMessage = rawMessage;
            p->timeStamp = timeStamp;
            p->tick = os::GetSystemTick();
            return p;
        }

        void PopFrontImpl() NN_NOEXCEPT
        {
            NN_SDK_REQUIRES(CanPop());
            auto p = &m_Queue.front();
            m_Queue.pop_front();
            m_FreeList.push_front(*p);
        }

        void PopBackImpl() NN_NOEXCEPT
        {
            NN_SDK_REQUIRES(CanPop());
            auto p = &m_Queue.back();
            m_Queue.pop_back();
            m_FreeList.push_front(*p);
        }

        Result ResolveOverflow(TimeStamp* pTimeStamp, OverflowOption overflowOption) NN_NOEXCEPT
        {
            switch (overflowOption)
            {
                case OverflowOption_Error:
                {
                    NN_RESULT_THROW(ovln::ResultQueueIsFull());
                }
                case OverflowOption_RemoveBack:
                {
                    PopBackImpl();
                    NN_RESULT_SUCCESS;
                }
                case OverflowOption_RemoveFront:
                {
                    // 先頭のものでタイムスタンプを上書き
                    *pTimeStamp = m_Queue.front().timeStamp;
                    PopFrontImpl();
                    NN_RESULT_SUCCESS;
                }
                case OverflowOption_Block:
                {
                    NN_RESULT_THROW(ovln::ResultNotImplemented());
                }
                default:
                {
                    NN_RESULT_THROW(ovln::ResultPreconditionViolation());
                }
            }
        }

        Result EnsureCanPush(bool* pOverFlow, TimeStamp* pTimeStamp, OverflowOption overflowOption) NN_NOEXCEPT
        {
            *pOverFlow = false;
            if (!CanPush())
            {
                NN_RESULT_DO(ResolveOverflow(pTimeStamp, overflowOption));
                *pOverFlow = true;
            }
            NN_SDK_ASSERT(CanPush());
            NN_RESULT_SUCCESS;
        }

        Result PushImpl(const RawMessage& rawMessage, TimeStamp timeStamp, EnqueuePosition enqueuePosition) NN_NOEXCEPT
        {
            NN_SDK_ASSERT(CanPush());
            switch (enqueuePosition)
            {
                case EnqueuePosition_Back:
                {
                    m_Queue.push_back(*AllocateImpl(rawMessage, timeStamp));
                    NN_RESULT_SUCCESS;
                }
                case EnqueuePosition_Front:
                {
                    if (!m_Queue.empty())
                    {
                        // 先頭に割り込ませる場合には、先頭のタイムスタンプ値を継承する
                        timeStamp = m_Queue.front().timeStamp;
                    }
                    m_Queue.push_front(*AllocateImpl(rawMessage, timeStamp));
                    NN_RESULT_SUCCESS;
                }
                default:
                {
                    NN_RESULT_THROW(ovln::ResultPreconditionViolation());
                }
            }
        }

    public:

        Queue() NN_NOEXCEPT
        {
        }

        Queue(Queue&& other) NN_NOEXCEPT
        {
            MoveAllToFront(&m_FreeList, &other.m_FreeList);
            MoveAllToFront(&m_Queue, &other.m_Queue);
        }

        ~Queue() NN_NOEXCEPT
        {
            NN_SDK_REQUIRES(m_Queue.empty());
            NN_SDK_REQUIRES(m_FreeList.empty());
        }

        MessageHolderList& GetFreeList() NN_NOEXCEPT
        {
            return m_FreeList;
        }

        const MessageHolderList& GetFreeList() const NN_NOEXCEPT
        {
            return m_FreeList;
        }

        int GetItemCount() const NN_NOEXCEPT
        {
            return m_Queue.size();
        }

        bool CanPop() const NN_NOEXCEPT
        {
            return !m_Queue.empty();
        }

        const MessageHolder& GetFront() const NN_NOEXCEPT
        {
            NN_SDK_REQUIRES(CanPop());
            return m_Queue.front();
        }

        void PopFront() NN_NOEXCEPT
        {
            PopFrontImpl();
        }

        bool CanPush() const NN_NOEXCEPT
        {
            return !m_FreeList.empty();
        }

        Result Push(int* pAdded, const RawMessage& rawMessage, SendOption option, TimeStamp timeStamp) NN_NOEXCEPT
        {
            bool overflow;
            NN_RESULT_DO(EnsureCanPush(&overflow, &timeStamp, static_cast<OverflowOption>(option.overflowOption)));
            NN_RESULT_DO(PushImpl(rawMessage, timeStamp, static_cast<EnqueuePosition>(option.enqueuePosition)));
            *pAdded = overflow ? 0 : 1;
            NN_RESULT_SUCCESS;
        }

        void Clear() NN_NOEXCEPT
        {
            MoveAllToFront(&m_FreeList, &m_Queue);
        }

    };

    MessageHolder* NewMessageHolderImpl() NN_NOEXCEPT
    {
        auto buffer = m_pMemoryResource->allocate(sizeof(MessageHolder), NN_ALIGNOF(MessageHolder));
        return buffer ? new (buffer) MessageHolder() : nullptr;
    }

    void DeleteMessageHolderImpl(MessageHolder* p) NN_NOEXCEPT
    {
        p->~MessageHolder();
        m_pMemoryResource->deallocate(p, sizeof(MessageHolder), NN_ALIGNOF(MessageHolder));
    }

    void DeleteEntriesImpl(MessageHolderList* pList) NN_NOEXCEPT
    {
        auto&& list = *pList;
        while (!list.empty())
        {
            auto p = &list.front();
            list.pop_front();
            DeleteMessageHolderImpl(p);
        }
    }

    bool AcquireQueueEntriesImpl(Queue* pQueue, uint32_t length) NN_NOEXCEPT
    {
        MessageHolderList list;
        auto success = false;
        NN_UTIL_SCOPE_EXIT
        {
            if (!success)
            {
                DeleteEntriesImpl(&list);
            }
        };

        for (auto i = 0u; i < length; ++i)
        {
            auto p = NewMessageHolderImpl();
            if (!p)
            {
                return false;
            }
            list.push_front(*p);
        }

        success = true;
        MoveAllToFront(&pQueue->GetFreeList(), &list);
        return true;
    }

    void ReleaseQueueEntriesImpl(Queue* pQueue) NN_NOEXCEPT
    {
        DeleteEntriesImpl(&pQueue->GetFreeList());
    }

    class QueueHolder
        : public util::IntrusiveListBaseNode<QueueHolder>
    {
    private:

        bool m_Valid;
        NotificationAgentImpl* m_pAgent;
        Queue m_Queue;
        sf::SharedPointer<Source> m_Source;

    public:

        QueueHolder() NN_NOEXCEPT
            : m_Valid(false)
        {
        }

        Result Initialize(NotificationAgentImpl* pAgent, sf::SharedPointer<Source> source, uint32_t queueLength) NN_NOEXCEPT
        {
            NN_SDK_REQUIRES(!m_Valid);
            NN_RESULT_THROW_UNLESS(pAgent->AcquireQueueEntriesImpl(&m_Queue, queueLength), ovln::ResultMemoryAllocationFailed());
            NN_SDK_ASSERT(!m_Queue.CanPop());
            this->m_Valid = true;
            source->AddQueueHolder(this);
            this->m_Source = std::move(source);
            this->m_pAgent = pAgent;
            NN_RESULT_SUCCESS;
        }

        ~QueueHolder() NN_NOEXCEPT
        {
            if (m_Valid)
            {
                m_Source->DecreaseItemCount(m_Queue.GetItemCount());
                m_Queue.Clear();
                m_Source->RemoveQueueHolder(this);
                m_pAgent->ReleaseQueueEntriesImpl(&m_Queue);
            }
        }

        int GetItemCount() const NN_NOEXCEPT
        {
            return m_Queue.GetItemCount();
        }

        bool CanPop() const NN_NOEXCEPT
        {
            NN_SDK_REQUIRES(m_Valid);
            return m_Queue.CanPop();
        }

        const MessageHolder& GetPopItem() const NN_NOEXCEPT
        {
            NN_SDK_REQUIRES(m_Valid);
            NN_SDK_REQUIRES(m_Queue.CanPop());
            return m_Queue.GetFront();
        }

        void Pop() NN_NOEXCEPT
        {
            NN_SDK_REQUIRES(m_Valid);
            NN_SDK_REQUIRES(m_Queue.CanPop());
            m_Queue.PopFront();
            m_Source->DecreaseItemCount(1);
        }

        Result Push(const ovln::RawMessage& message, ovln::SendOption option) NN_NOEXCEPT
        {
            NN_SDK_REQUIRES(m_Valid);
            int added;
            NN_RESULT_DO(m_Queue.Push(&added, message, option, m_pAgent->MakeTimeStamp()));
            NN_SDK_ASSERT(m_Queue.CanPop());
            m_Source->IncreaseItemCount(added);
            NN_RESULT_SUCCESS;
        }

        int GetCurrentItemCount() const NN_NOEXCEPT
        {
            NN_SDK_REQUIRES(m_Valid);
            return m_Queue.GetItemCount();
        }

    };

    class Receiver
    {
    private:

        bool m_Valid;
        NotificationAgentImpl* m_pAgent;
        util::IntrusiveList<Source, util::IntrusiveListBaseNodeTraits<Source>> m_Sources;
        os::SystemEventType m_Event;
        int m_ItemCount;

        Result AddSourceImpl(sf::SharedPointer<Source> source) NN_NOEXCEPT
        {
            NN_RESULT_THROW_UNLESS(source->AttachReceiver(this), ovln::ResultSourceIsBusy());
            IncreaseItemCount(source->GetItemCount());
            m_Sources.push_back(*source.Detach());
            NN_RESULT_SUCCESS;
        }

        void RemoveSourceImpl(Source* pSource) NN_NOEXCEPT
        {
            DecreaseItemCount(pSource->GetItemCount());
            m_Sources.erase(m_Sources.iterator_to(*pSource));
            pSource->DetachReceiver(this);
            sf::ReleaseSharedObject(pSource);
        }

    public:

        Receiver() NN_NOEXCEPT
            : m_Valid(false)
        {
        }

        Result Initialize(NotificationAgentImpl* pAgent) NN_NOEXCEPT
        {
            NN_RESULT_DO(os::CreateSystemEvent(&m_Event, os::EventClearMode_ManualClear, true));
            this->m_pAgent = pAgent;
            this->m_ItemCount = 0;
            this->m_Valid = true;
            NN_RESULT_SUCCESS;
        }

        ~Receiver() NN_NOEXCEPT
        {
            if (m_Valid)
            {
                while (!m_Sources.empty())
                {
                    RemoveSourceImpl(&m_Sources.front());
                }
                os::DestroySystemEvent(&m_Event);
                NN_SDK_ASSERT(m_ItemCount == 0);
            }
        }

        Result AddSource(const ovln::SourceName& sourceName) NN_NOEXCEPT
        {
            NN_SDK_REQUIRES(m_Valid);
            sf::SharedPointer<Source> source;
            NN_RESULT_DO(m_pAgent->GetSource(&source, sourceName));
            return AddSourceImpl(std::move(source));
        }

        Result RemoveSource(const ovln::SourceName& sourceName) NN_NOEXCEPT
        {
            NN_SDK_REQUIRES(m_Valid);
            sf::SharedPointer<Source> source;
            NN_RESULT_DO(m_pAgent->GetSource(&source, sourceName));
            RemoveSourceImpl(source.Get());
            NN_RESULT_SUCCESS;
        }

        Result GetReceiveEventHandle(sf::Out<sf::NativeHandle> pOut) NN_NOEXCEPT
        {
            NN_SDK_REQUIRES(m_Valid);
            *pOut = sf::NativeHandle(os::GetReadableHandleOfSystemEvent(&m_Event), false);
            NN_RESULT_SUCCESS;
        }

        Result Receive(sf::Out<ovln::RawMessage> pMessage, sf::Out<int64_t> pOutTick) NN_NOEXCEPT
        {
            NN_SDK_REQUIRES(m_Valid);
            auto pTarget = static_cast<QueueHolder*>(nullptr);
            auto targetTimeStamp = TimeStamp();
            for (auto&& source : m_Sources)
            {
                for (auto&& holder : source.GetQueueHolders())
                {
                    if (holder.CanPop())
                    {
                        auto&& e = holder.GetPopItem();
                        if (!(pTarget && targetTimeStamp <= e.timeStamp))
                        {
                            pTarget = &holder;
                            targetTimeStamp = e.timeStamp;
                        }
                    }
                }
            }
            NN_RESULT_THROW_UNLESS(pTarget, ovln::ResultNoMessage());
            *pMessage = pTarget->GetPopItem().rawMessage;
            *pOutTick = pTarget->GetPopItem().tick.GetInt64Value();
            pTarget->Pop();
            NN_RESULT_SUCCESS;
        }

        void IncreaseItemCount(int n) NN_NOEXCEPT
        {
            NN_SDK_REQUIRES(m_Valid);
            NN_SDK_REQUIRES(n >= 0);
            NN_SDK_ASSERT(m_ItemCount >= 0);
            if (n == 0)
            {
                return;
            }
            auto previousItemCount = m_ItemCount;
            this->m_ItemCount += n;
            if (previousItemCount == 0)
            {
                os::SignalSystemEvent(&m_Event);
            }
        }

        void DecreaseItemCount(int n) NN_NOEXCEPT
        {
            NN_SDK_REQUIRES(m_Valid);
            NN_SDK_REQUIRES(n >= 0);
            if (!(m_ItemCount >= n))
            {
            NN_SDK_REQUIRES(m_ItemCount >= n);
            }
            if (n == 0)
            {
                return;
            }
            this->m_ItemCount -= n;
            if (m_ItemCount == 0)
            {
                os::ClearSystemEvent(&m_Event);
            }
        }

    };

    class SourceManager
    {
    private:

        class SourceHolder
            : public util::IntrusiveListBaseNode<SourceHolder>
            , public Source
        {
        private:

            SourceManager* m_pParent;
            SourceName m_Name;

        public:

            SourceHolder(SourceManager* pParent, const SourceName& name) NN_NOEXCEPT
                : m_pParent(pParent)
                , m_Name(name)
            {
                m_pParent->OnConstructSourceImpl(this);
            }

            ~SourceHolder() NN_NOEXCEPT
            {
                m_pParent->OnDestructSourceImpl(this);
            }

            const SourceName& GetName() const NN_NOEXCEPT
            {
                return m_Name;
            }

        };

        MemoryResource* m_pMemoryResource;
        util::IntrusiveList<SourceHolder, util::IntrusiveListBaseNodeTraits<SourceHolder>> m_Sources;

        void OnConstructSourceImpl(SourceHolder* p) NN_NOEXCEPT
        {
            m_Sources.push_front(*p);
        }

        void OnDestructSourceImpl(SourceHolder* p) NN_NOEXCEPT
        {
            m_Sources.erase(m_Sources.iterator_to(*p));
        }

    public:

        explicit SourceManager(MemoryResource* pMemoryResource) NN_NOEXCEPT
            : m_pMemoryResource(pMemoryResource)
        {
        }

        Result GetSource(sf::SharedPointer<Source>* pOut, const SourceName& name) NN_NOEXCEPT
        {
            auto it = std::find_if(m_Sources.begin(), m_Sources.end(), [&name](const SourceHolder& source)
            {
                return source.GetName() == name;
            });
            if (it != m_Sources.end())
            {
                *pOut = sf::SharedPointer<Source>(&*it, true);
                NN_RESULT_SUCCESS;
            }
            else
            {
                auto ret = SfObjectFactory::CreateUserSharedObject<SourceHolder>(m_pMemoryResource, this, name);
                NN_RESULT_THROW_UNLESS(ret, ovln::ResultMemoryAllocationFailed());
                *pOut = std::move(ret);
                NN_RESULT_SUCCESS;
            }
        }

    };

    Result GetSource(sf::SharedPointer<Source>* pOut, const SourceName& name) NN_NOEXCEPT
    {
        return m_SourceManager.GetSource(pOut, name);
    }

    TimeStamp MakeTimeStamp() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(m_AgentLock.IsLockedByCurrent());
        return ++m_CurrentTimeStamp;
    }

    class AgentLock
        : public nn::os::Mutex
    {
    public:

        AgentLock() NN_NOEXCEPT
            : Mutex(false)
        {
        }

        bool IsLockedByCurrent() const NN_NOEXCEPT
        {
            return this->IsLockedByCurrentThread();
        }

    };

    class SenderObject
    {
    private:

        AgentLock* m_pAgentLock;
        util::optional<QueueHolder> m_QueueHolder;

    public:

        explicit SenderObject(AgentLock* pAgentLock) NN_NOEXCEPT
            : m_pAgentLock(pAgentLock)
            , m_QueueHolder(util::in_place)
        {
        }

        ~SenderObject() NN_NOEXCEPT
        {
            std::lock_guard<decltype(*m_pAgentLock)> lk(*m_pAgentLock);
            m_QueueHolder = util::nullopt;
        }

        Result Initialize(NotificationAgentImpl* pAgent, sf::SharedPointer<Source> source, uint32_t queueLength) NN_NOEXCEPT
        {
            NN_SDK_ASSERT(m_pAgentLock->IsLockedByCurrent());
            return m_QueueHolder->Initialize(pAgent, std::move(source), queueLength);
        }

        Result Send(const ovln::RawMessage& message, ovln::SendOption option) NN_NOEXCEPT
        {
            std::lock_guard<decltype(*m_pAgentLock)> lk(*m_pAgentLock);
            return m_QueueHolder->Push(message, option);
        }

        Result GetUnreceivedMessageCount(sf::Out<uint32_t> pOut) NN_NOEXCEPT
        {
            std::lock_guard<decltype(*m_pAgentLock)> lk(*m_pAgentLock);
            *pOut = static_cast<uint32_t>(m_QueueHolder->GetItemCount());
            NN_RESULT_SUCCESS;
        }

    };

    class ReceiverObject
    {
    private:

        AgentLock* m_pAgentLock;
        util::optional<Receiver> m_Receiver;

    public:

        explicit ReceiverObject(AgentLock* pAgentLock) NN_NOEXCEPT
            : m_pAgentLock(pAgentLock)
            , m_Receiver(util::in_place)
        {
        }

        ~ReceiverObject() NN_NOEXCEPT
        {
            std::lock_guard<decltype(*m_pAgentLock)> lk(*m_pAgentLock);
            m_Receiver = util::nullopt;
        }

        Result Initialize(NotificationAgentImpl* pAgent) NN_NOEXCEPT
        {
            NN_SDK_ASSERT(m_pAgentLock->IsLockedByCurrent());
            return m_Receiver->Initialize(pAgent);
        }

        Result AddSource(const ovln::SourceName& sourceName) NN_NOEXCEPT
        {
            std::lock_guard<decltype(*m_pAgentLock)> lk(*m_pAgentLock);
            return m_Receiver->AddSource(sourceName);
        }

        Result RemoveSource(const ovln::SourceName& sourceName) NN_NOEXCEPT
        {
            std::lock_guard<decltype(*m_pAgentLock)> lk(*m_pAgentLock);
            return m_Receiver->RemoveSource(sourceName);
        }

        Result GetReceiveEventHandle(sf::Out<sf::NativeHandle> pOut) NN_NOEXCEPT
        {
            std::lock_guard<decltype(*m_pAgentLock)> lk(*m_pAgentLock);
            return m_Receiver->GetReceiveEventHandle(pOut);
        }

        Result Receive(sf::Out<ovln::RawMessage> pMessage) NN_NOEXCEPT
        {
            int64_t dummy;
            return ReceiveWithTick(pMessage, &dummy);
        }

        Result ReceiveWithTick(sf::Out<ovln::RawMessage> pMessage, sf::Out<int64_t> pOutTick) NN_NOEXCEPT
        {
            std::lock_guard<decltype(*m_pAgentLock)> lk(*m_pAgentLock);
            return m_Receiver->Receive(pMessage, pOutTick);
        }

    };

    AgentLock m_AgentLock;
    MemoryResource* m_pMemoryResource;
    SourceManager m_SourceManager;
    TimeStamp m_CurrentTimeStamp;

public:

    explicit NotificationAgentImpl(MemoryResource* pMemoryResource) NN_NOEXCEPT
        : m_pMemoryResource(pMemoryResource)
        , m_SourceManager(pMemoryResource)
        , m_CurrentTimeStamp(0)
    {
    }

    Result OpenSender(sf::Out<sf::SharedPointer<ovln::ISender>> pOut, const ovln::SourceName& sourceName, ovln::QueueAttribute queueAttribute) NN_NOEXCEPT
    {
        std::lock_guard<decltype(m_AgentLock)> lk(m_AgentLock);

        sf::SharedPointer<Source> source;
        NN_RESULT_DO(GetSource(&source, sourceName));

        auto ret = SfObjectFactory::CreateSharedEmplaced<ovln::ISender, SenderObject>(m_pMemoryResource, &this->m_AgentLock);
        NN_RESULT_THROW_UNLESS(ret, ovln::ResultMemoryAllocationFailed());
        NN_RESULT_DO(ret.GetImpl().Initialize(this, std::move(source), queueAttribute.queueLength));

        *pOut = std::move(ret);
        NN_RESULT_SUCCESS;
    }

    Result OpenReceiver(sf::Out<sf::SharedPointer<ovln::IReceiver>> pOut) NN_NOEXCEPT
    {
        std::lock_guard<decltype(m_AgentLock)> lk(m_AgentLock);

        auto ret = SfObjectFactory::CreateSharedEmplaced<ovln::IReceiver, ReceiverObject>(m_pMemoryResource, &this->m_AgentLock);
        NN_RESULT_THROW_UNLESS(ret, ovln::ResultMemoryAllocationFailed());
        NN_RESULT_DO(ret.GetImpl().Initialize(this));

        *pOut = std::move(ret);
        NN_RESULT_SUCCESS;
    }

};

class NotificationAgentObject
    : public NotificationAgent
{
private:

    NotificationAgentImpl m_Impl;
    sf::UnmanagedServiceObjectByPointer<ovln::IReceiverService, NotificationAgentImpl> m_ReceiverService;
    sf::UnmanagedServiceObjectByPointer<ovln::ISenderService, NotificationAgentImpl> m_SenderService;

public:

    explicit NotificationAgentObject(MemoryResource* pMemoryResource) NN_NOEXCEPT
        : m_Impl(pMemoryResource)
        , m_ReceiverService(&m_Impl)
        , m_SenderService(&m_Impl)
    {
    }

    sf::SharedPointer<ovln::IReceiverService> GetReceiverService() NN_NOEXCEPT
    {
        return m_ReceiverService.GetShared();
    }

    sf::SharedPointer<ovln::ISenderService> GetSenderService() NN_NOEXCEPT
    {
        return m_SenderService.GetShared();
    }

    MemoryResource* pMemoryResource;

};

}

NotificationAgent* CreateNotificationAgent(MemoryResource* pMemoryResource) NN_NOEXCEPT
{
    auto buffer = pMemoryResource->allocate(sizeof(NotificationAgentObject), NN_ALIGNOF(NotificationAgentObject));
    if (!buffer)
    {
        return nullptr;
    }
    auto p = new (buffer) NotificationAgentObject(pMemoryResource);
    p->pMemoryResource = pMemoryResource;
    return p;
}

void DestroyNotificationAgent(NotificationAgent* p_) NN_NOEXCEPT
{
    auto p = static_cast<NotificationAgentObject*>(p_);
    auto pMemoryResource = p->pMemoryResource;
    p->~NotificationAgentObject();
    pMemoryResource->deallocate(p, sizeof(NotificationAgentObject), NN_ALIGNOF(NotificationAgentObject));
}

sf::SharedPointer<ovln::IReceiverService> GetReceiverService(NotificationAgent* p_) NN_NOEXCEPT
{
    auto p = static_cast<NotificationAgentObject*>(p_);
    return p->GetReceiverService();
}

sf::SharedPointer<ovln::ISenderService> GetSenderService(NotificationAgent* p_) NN_NOEXCEPT
{
    auto p = static_cast<NotificationAgentObject*>(p_);
    return p->GetSenderService();
}

}}
