﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_BitTypes.h>
#include <nn/nn_SdkLog.h>

#include <nn/svc/svc_ThreadLocalRegion.h>

#include "osdbg_ThreadInfo-os.horizon.h"
#include "osdbg_ThreadLocalRegion-os.horizon.h"

namespace nn { namespace osdbg {
namespace detail {

//-----------------------------------------------------------------------------
//  デバッグ対象スレッドの ThreadType の先頭アドレスを返す。
//  失敗した場合は NULL を返す。
//
ThreadTypeCommon* GetTargetThreadType(ThreadInfo* info) NN_NOEXCEPT
{
    bool isLp64 = IsLp64(info);

    // まず、デバッグ対象スレッドの TLS 領域にある Thread ポインタ情報を取得
    auto* threadLocalRegion    = GetTargetThreadLocalRegion(info);
    auto  threadPointerAddress = isLp64 ? reinterpret_cast<uintptr_t>(&threadLocalRegion->lp64.pThreadType)
                                        : reinterpret_cast<uintptr_t>(&threadLocalRegion->ilp32.pThreadType);

    // ここの 0 代入は上位 32bit を 0 初期化するため
    uintptr64_t pThreadType = 0;
    Result result = svc::ReadDebugProcessMemory(
                        reinterpret_cast<uintptr_t>( &pThreadType ),
                        info->_debugHandle,
                        threadPointerAddress,
                        isLp64 ? sizeof(uintptr64_t) : sizeof(uintptr32_t));
    if (!result.IsSuccess())
    {
        NN_SDK_LOG("[osdbg] svc::ReadDebugProcessMemory(tid=%lld, tp=0x%p) %s pThreadType=0x%p result=0x%08x\n", info->_debugInfoCreateThread.id, threadPointerAddress, isLp64 ? "lp64" : "ilp32", pThreadType, result.GetInnerValueForDebug());
        return NULL;
    }
    if (pThreadType)
    {
        return reinterpret_cast<ThreadTypeCommon*>(pThreadType);
    }

    // pThreadType == NULL の場合は、対象スレッドが生成された直後で、
    // まだ ThreadType* ポインタ情報が TLS 領域にセットされてない状態である。
    // この場合、当該スレッドコンテキストの r0(x0) レジスタの内容を取得し、
    // その値を ThreadType* ポインタ情報として返す（libnn_os の設計と連動）。

    svc::ThreadContext  context;
    result = svc::GetDebugThreadContext( &context,
                                         info->_debugHandle,
                                         info->_debugInfoCreateThread.id,
                                         svc::ContextFlag_General);
    if (!result.IsSuccess())
    {
        NN_SDK_LOG("[osdbg] svc::GetDebugThreadContext(tid=%lld) r0=0x%p result=0x%08x\n", info->_debugInfoCreateThread.id, context.r[0], result.GetInnerValueForDebug());
        return NULL;
    }

    return reinterpret_cast<ThreadTypeCommon*>(context.r[0]);
}

//-----------------------------------------------------------------------------

}   // namespace detail
}}  // namespace nn::osdbg

