﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <new>
#include <mutex>

#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/diag/text/diag_SdkTextOs.h>
#include <nn/nn_Result.h>
#include <nn/os/os_Result.h>
#include <nn/os/os_SdkResult.h>
#include <nn/os/os_Types.h>

#include "detail/os_UnsafeMemoryImpl.h"
#include "detail/os_MemoryHeapManager.h"

namespace nn { namespace os {

//--------------------------------------------------------------------------
Result AllocateUnsafeMemory(uintptr_t* pOutAddress, size_t size) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( size > 0, NN_TEXT_OS("size=0x%zx"), size );
    NN_SDK_REQUIRES( detail::IsAligned( size, MemoryPageSize ), NN_TEXT_OS("size=0x%zx"), size );

    // TORIAEZU: 使用中の領域にマップしようとした場合、64 回までリトライする
    for (int loopCount = 0; loopCount < 64; ++loopCount)
    {
        uintptr_t mappedAddress;

        auto result = detail::UnsafeMemoryManagerImpl::AllocateUnsafeMemoryImpl(&mappedAddress, size);
        if (result.IsSuccess())
        {
            *pOutAddress = mappedAddress;
            NN_RESULT_SUCCESS;
        }

        if (!ResultInvalidCurrentMemoryState::Includes(result) && !ResultNoGuardSpace::Includes(result))
        {
            return result;
        }
    }

    return os::ResultOutOfMemory();
}

//--------------------------------------------------------------------------
Result FreeUnsafeMemory(uintptr_t address, size_t size) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( detail::IsAligned( address, MemoryPageSize ), NN_TEXT_OS("address=0x%zx"), address );
    NN_SDK_REQUIRES( detail::IsAligned( size, MemoryPageSize ), NN_TEXT_OS("size=0x%zx"), size );

    return detail::UnsafeMemoryManagerImpl::FreeUnsafeMemoryImpl(address, size);
}

}} // namespace nn::os
