﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <new>
#include <mutex>

#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/diag/text/diag_SdkTextOs.h>
#include <nn/nn_Result.h>
#include <nn/os/os_Types.h>
#include <nn/os/os_NativeHandleTypes.h>
#include <nn/os/os_TransferMemoryApi.h>
#include <nn/os/os_TransferMemoryTypes.h>
#include <nn/os/detail/os_InternalCriticalSection.h>

#include "detail/os_Diag.h"
#include "detail/os_Common.h"
#include "detail/os_ThreadManager.h"
#include "detail/os_MemoryHeapManager.h"
#include "detail/os_TransferMemoryImpl.h"
#include "detail/os_AslrSpaceManagerTypes.h"
#include "detail/os_AslrSpaceManager.h"


namespace nn { namespace os {

namespace {

//--------------------------------------------------------------------------
//  Transfer memory を指定したアドレスにマップ（ロックしない内部実装）
nn::Result MapTransferMemoryWithAddressUnsafe(nn::os::TransferMemoryType* transferMemory, void* address, nn::os::MemoryPermission ownerPermission) NN_NOEXCEPT
{
    void* mappedAddress = NULL;
    nn::Result result = nn::os::detail::TransferMemoryImpl::Map(&mappedAddress,
                                                        transferMemory->_handle,
                                                        address,
                                                        transferMemory->_size,
                                                        ownerPermission);
    if (result.IsFailure())
    {
        return result;
    }

    transferMemory->_address  = mappedAddress;
    transferMemory->_state    = nn::os::TransferMemoryType::State_Mapped;

    return nn::ResultSuccess();
}

}

//--------------------------------------------------------------------------

inline void SetupTransferMemoryType(TransferMemoryType* transferMemory, size_t size, NativeHandle handle, bool managed) NN_NOEXCEPT
{
    // メンバの初期化
    transferMemory->_handle       = handle;
    transferMemory->_size         = size;
    transferMemory->_address      = NULL;
    transferMemory->_allocated    = false;

    // このメンバが true の場合、_handle メンバの管理責任も同時に保有している。
    // そのため、DestroyTransferMemory() 時にクローズする必要がある。
    transferMemory->_isHandleManaged    = managed;

    // コンストラクタを呼ぶ
    new( &transferMemory->_csTransferMemory ) detail::InternalCriticalSection;

    // Transfer memory を初期化状態へ
    transferMemory->_state    = TransferMemoryType::State_Created;
}

//--------------------------------------------------------------------------
//  Transfer memory の作成（ハンドルの生成）
nn::Result CreateTransferMemory(TransferMemoryType* pOutTransferMemory, void* addr, size_t size, MemoryPermission permission) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( size > 0, NN_TEXT_OS("nn::os::CreateTransferMemory(): size > 0 ではありません（size=0x%zx）"), size );
    NN_SDK_REQUIRES( detail::IsAligned( size, MemoryPageSize ), NN_TEXT_OS("nn::os::CreateTransferMemory(): size が MemoryPageSize の整数倍ではありません（size=0x%zx）"), size );
    NN_SDK_REQUIRES( addr != NULL, NN_TEXT_OS("nn::os::CreateTransferMemory(): addr が NULL です。"));
    NN_SDK_REQUIRES( detail::IsAligned( reinterpret_cast<uintptr_t>(addr), MemoryPageSize ), NN_TEXT_OS("nn::os::CreateTransferMemory(): addr が MemoryPageSize の整数倍ではありません（addr=0x%p）"), addr );

    NativeHandle  handle;
    auto result = detail::TransferMemoryImpl::Create(&handle, addr, size, permission);
    if (!result.IsSuccess())
    {
        return result;
    }

    // メンバの初期化
    SetupTransferMemoryType(pOutTransferMemory, size, handle, true);

    return nn::ResultSuccess();
}


//--------------------------------------------------------------------------
//  TransferMemoryTypeオブジェクトのアタッチ（既存ハンドルを関連付ける）
void AttachTransferMemory(TransferMemoryType* pOutTransferMemory, size_t size, NativeHandle handle, bool managed) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( size > 0, NN_TEXT_OS("nn::os::AttachTransferMemory(): size > 0 ではありません（size=0x%zx）"), size );
    NN_SDK_REQUIRES( detail::IsAligned( size, MemoryPageSize ), NN_TEXT_OS("nn::os::AttachTransferMemory(): size が MemoryPageSize の整数倍ではありません（size=0x%zx）"), size );
    NN_SDK_REQUIRES( handle != nn::os::InvalidNativeHandle, NN_TEXT_OS("nn::os::AttachTransferMemory(): handle が有効なハンドルではありません。") );

    // メンバの初期化
    SetupTransferMemoryType(pOutTransferMemory, size, handle, managed);
}


//--------------------------------------------------------------------------
//  TransferMemoryTypeオブジェクトのデタッチ（既存ハンドルを切り離す）
NativeHandle DetachTransferMemory(TransferMemoryType* transferMemory) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( transferMemory->_state == TransferMemoryType::State_Created, NN_TEXT_OS("nn::os::DetachTransferMemory(): Transfer memory が生成済み状態ではありません。") );

    // 状態を変更
    transferMemory->_state = TransferMemoryType::State_Detached;

    // ハンドルを切り離す
    NativeHandle handle = transferMemory->_handle;
    transferMemory->_handle          = 0U;
    transferMemory->_isHandleManaged = false;

    return handle;
}


//--------------------------------------------------------------------------
//  Transfer memory の削除（ハンドルも削除）
void DestroyTransferMemory(TransferMemoryType* transferMemory) NN_NOEXCEPT
{
    if ( transferMemory->_state == TransferMemoryType::State_Mapped )
    {
        UnmapTransferMemory(transferMemory);
    }

    auto state = transferMemory->_state;
    NN_SDK_REQUIRES( state == TransferMemoryType::State_Created || state == TransferMemoryType::State_Detached, NN_TEXT_OS("nn::os::DestroyTransferMemory(): Transfer memory が生成済み状態もしくはデタッチ状態ではありません。") );
    NN_UNUSED(state);

    // 状態を変更
    transferMemory->_state    = TransferMemoryType::State_Uninitialized;

    if (transferMemory->_isHandleManaged)
    {
        detail::TransferMemoryImpl::Close(transferMemory->_handle);
    }
    transferMemory->_isHandleManaged = false;

    transferMemory->_address  = NULL;
    transferMemory->_size     = 0;
    transferMemory->_handle   = 0;

    // デストラクタを呼ぶ
    Get(transferMemory->_csTransferMemory).~InternalCriticalSection();
}


//--------------------------------------------------------------------------
//  Transfer memory をマップ
nn::Result MapTransferMemory(void** pOutAddress, TransferMemoryType* transferMemory, MemoryPermission ownerPermission) NN_NOEXCEPT
{
    // クリティカルセクション
    // ロック区間が長いが、通常使用では全く問題にならない。
    // そもそも、複数スレッドから同時に呼ばれるような使い方は使用方法が
    // 間違っている可能性が高い。ここでのロックは安全に abort するためである。
    std::lock_guard<detail::InternalCriticalSection> lockForSuspend( Get(detail::GetCurrentThread()->_csThread) );
    std::lock_guard<detail::InternalCriticalSection> lock( Get(transferMemory->_csTransferMemory) );

    NN_SDK_REQUIRES( transferMemory->_state == TransferMemoryType::State_Created, NN_TEXT_OS("nn::os::MapTransferMemory(): Transfer memory が生成済み状態ではありません。") );

    // TORIAEZU: 使用中の領域にマップしようとした場合、64 回までリトライする
    for (int loopCount = 0; loopCount < 64; ++loopCount)
    {
        // フリーマップ領域からマップ先を確保
        void* mappingAddress = detail::GetAslrSpaceManagerInstance()->AllocateSpace(transferMemory->_size);
        if (!mappingAddress)
        {
            return ResultOutOfAddressSpace();
        }

        transferMemory->_allocated = true;

        // アドレス付きで実際にマップ
        nn::Result result = MapTransferMemoryWithAddressUnsafe(transferMemory, mappingAddress, ownerPermission);

        // 成功した場合
        if (result.IsSuccess())
        {
            // 前後にガード空間があるかチェック
            if (!detail::GetAslrSpaceManagerInstance()->CheckGuardSpace(reinterpret_cast<uintptr_t>(transferMemory->_address), transferMemory->_size))
            {
                // なかった場合、アンマップしてリトライ
                detail::TransferMemoryImpl::Unmap(transferMemory->_handle, transferMemory->_address, transferMemory->_size);
            }
            else
            {
                // mappingAddress にマップされるとは限らない（現時点では Win32 の場合、別の場所にマップされる）
                // 実際にマップされたアドレスは transferMemory->_address に入っている
                *pOutAddress = transferMemory->_address;
                return nn::ResultSuccess();
            }
        }

        // マップに失敗した場合、アドレス空間を返却
        transferMemory->_allocated = false;

        // ResultInvalidCurrentMemoryState 以外のエラーの場合、すぐに返る
        if (!ResultInvalidCurrentMemoryState::Includes(result))
        {
            return result;
        }
    }

    // マップに必要な仮想アドレス空間が足りなかった
    return ResultOutOfAddressSpace();
}


//--------------------------------------------------------------------------
//  Transfer memory をアンマップ
void UnmapTransferMemory(TransferMemoryType* transferMemory) NN_NOEXCEPT
{
    // クリティカルセクション
    // ロック区間が長いが、通常使用では全く問題にならない。
    // そもそも、複数スレッドから同時に呼ばれるような使い方は使用方法が
    // 間違っている可能性が高い。ここでのロックは安全に abort するためである。
    std::lock_guard<detail::InternalCriticalSection> lock( Get(transferMemory->_csTransferMemory) );

    if ( transferMemory->_state != TransferMemoryType::State_Mapped )
    {
        return;
    }

    detail::TransferMemoryImpl::Unmap(transferMemory->_handle, transferMemory->_address, transferMemory->_size);

    // 空間の返却
    if (transferMemory->_allocated)
    {
        transferMemory->_allocated = false;
    }

    // メンバの初期化
    transferMemory->_address  = NULL;
    transferMemory->_state    = TransferMemoryType::State_Created;
}

//--------------------------------------------------------------------------

}} // namespace nn::os

