﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_TimeSpan.h>

#include <nn/os/os_Macro.h>
#include <nn/os/os_ThreadCommon.h>
#include <nn/os/os_SdkThreadCommon.h>
#include <nn/os/os_ThreadTypes.h>
#include <nn/os/os_ThreadApi.h>

#include <nnc/result/result_CConverter.h>
#include <nnc/os/os_Macro.h>
#include <nnc/os/os_ThreadCommon.h>
#include <nnc/os/os_SdkThreadCommon.h>
#include <nnc/os/os_ThreadTypes.h>
#include <nnc/os/os_ThreadApi.h>

#include "detail/os_Diag.h"
#include "detail/os_Common.h"

namespace nn { namespace os {
    NN_NORETURN void QuickExit() NN_NOEXCEPT;
}}


//---------------------------------------------------------------------------
//  C++ 用の型とサイズやアライメント、定数値等が一致しているのかをチェック
//---------------------------------------------------------------------------

static_assert(sizeof(nnosThreadType) == sizeof(nn::os::ThreadType), "");
static_assert(NN_ALIGNOF(nnosThreadType) == NN_ALIGNOF(nn::os::ThreadType), "");

static_assert(NN_OS_HIGHEST_SYSTEM_THREAD_PRIORITY == nn::os::HighestSystemThreadPriority, "");
static_assert(NN_OS_LOWEST_SYSTEM_THREAD_PRIORITY  == nn::os::LowestSystemThreadPriority,  "");

static_assert(NN_OS_THREAD_NAME_LENGTH_MAX == nn::os::ThreadNameLengthMax, "");

static_assert(NN_OS_THREAD_PRIORITY_RANGE_SIZE == nn::os::ThreadPriorityRangeSize, "");
static_assert(NN_OS_HIGHEST_THREAD_PRIORITY == nn::os::HighestThreadPriority, "");
static_assert(NN_OS_DEFAULT_THREAD_PRIORITY == nn::os::DefaultThreadPriority, "");
static_assert(NN_OS_LOWEST_THREAD_PRIORITY  == nn::os::LowestThreadPriority,  "");

static_assert(NN_OS_GUARDED_STACK_ALIGNMENT == nn::os::GuardedStackAlignment, "");
static_assert(NN_OS_THREAD_STACK_ALIGNMENT  == nn::os::ThreadStackAlignment, "");
static_assert(NN_OS_STACK_REGION_ALIGNMENT  == nn::os::StackRegionAlignment, "");


//---------------------------------------------------------------------------
//  C 関数の定義
//---------------------------------------------------------------------------

NN_OS_EXTERN_C nnResult nnosCreateThread(nnosThreadType* thread, nnosThreadFunction function, void* argument, void* stack, size_t stackSize, int priority)
{
    nn::Result result = nn::os::CreateThread(reinterpret_cast<nn::os::ThreadType*>(thread), function, argument, stack, stackSize, priority);

    return nn::result::ConvertToC(result);
}

NN_OS_EXTERN_C nnResult nnosCreateThreadWithCoreNumber(nnosThreadType* thread, nnosThreadFunction function, void* argument, void* stack, size_t stackSize, int priority, int coreNumber)
{
    nn::Result result = nn::os::CreateThread(reinterpret_cast<nn::os::ThreadType*>(thread), function, argument, stack, stackSize, priority, coreNumber);

    return nn::result::ConvertToC(result);
}

NN_OS_EXTERN_C void nnosDestroyThread(nnosThreadType* thread)
{
    nn::os::DestroyThread(reinterpret_cast<nn::os::ThreadType*>(thread));
}

NN_OS_EXTERN_C void nnosStartThread(nnosThreadType* thread)
{
    nn::os::StartThread(reinterpret_cast<nn::os::ThreadType*>(thread));
}

NN_OS_EXTERN_C nnosThreadType* nnosGetCurrentThread(void)   // NOLINT(readability/function)
{
    auto thread = nn::os::GetCurrentThread();
    return reinterpret_cast<nnosThreadType*>(thread);
}

NN_OS_EXTERN_C void nnosWaitThread(nnosThreadType* thread)
{
    nn::os::WaitThread(reinterpret_cast<nn::os::ThreadType*>(thread));
}

NN_OS_EXTERN_C void nnosYieldThread(void)   // NOLINT(readability/function)
{
    nn::os::YieldThread();
}

NN_OS_EXTERN_C void nnosSleepThread(int64_t nanoseconds)
{
    nn::os::SleepThread(nn::TimeSpan::FromNanoSeconds(nanoseconds));
}

NN_OS_EXTERN_C int nnosChangeThreadPriority(nnosThreadType* thread, int priority)
{
    return nn::os::ChangeThreadPriority(reinterpret_cast<nn::os::ThreadType*>(thread), priority);
}

NN_OS_EXTERN_C int nnosGetThreadPriority(const nnosThreadType* thread)
{
    return nn::os::GetThreadPriority(reinterpret_cast<const nn::os::ThreadType*>(thread));
}

NN_OS_EXTERN_C int nnosGetThreadCurrentPriority(const nnosThreadType* thread)
{
    return nn::os::GetThreadCurrentPriority(reinterpret_cast<const nn::os::ThreadType*>(thread));
}

NN_OS_EXTERN_C void nnosSetThreadName(nnosThreadType* thread, const char* threadName)
{
    nn::os::SetThreadName(reinterpret_cast<nn::os::ThreadType*>(thread), threadName);
}

NN_OS_EXTERN_C void nnosSetThreadNamePointer(nnosThreadType* thread, const char* threadName)
{
    nn::os::SetThreadNamePointer(reinterpret_cast<nn::os::ThreadType*>(thread), threadName);
}

NN_OS_EXTERN_C const char* nnosGetThreadNamePointer(const nnosThreadType* thread)
{
    return nn::os::GetThreadNamePointer(reinterpret_cast<const nn::os::ThreadType*>(thread));
}

NN_OS_EXTERN_C int nnosGetCurrentProcessorNumber(void)  // NOLINT(readability/function)
{
    return nn::os::GetCurrentCoreNumber();
}

NN_OS_EXTERN_C int nnosGetCurrentCoreNumber(void)  // NOLINT(readability/function)
{
    return nn::os::GetCurrentCoreNumber();
}

NN_OS_EXTERN_C void nnosQuickExit() NN_NOEXCEPT
{
    return nn::os::QuickExit();
}

NN_OS_EXTERN_C void nnosInitializeMultiWaitHolderForThread(nnosMultiWaitHolderType* multiWaitHolder, nnosThreadType* thread)
{
    nn::os::InitializeMultiWaitHolder(reinterpret_cast<nn::os::MultiWaitHolderType*>(multiWaitHolder), reinterpret_cast<nn::os::ThreadType*>(thread));
}
