﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>

#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/diag/text/diag_SdkTextOs.h>
#include <nn/os/os_MultipleWaitTypes.h>
#include <nn/os/os_MultipleWaitApi.h>
#include <nn/os/os_MemoryFenceApi.h>

#include "detail/os_Diag.h"
#include "detail/os_Common.h"
#include "detail/os_MultipleWaitHelper.h"
#include "detail/os_MultipleWaitHolderBase.h"
#include "detail/os_MultipleWaitHolderImpl.h"


//---------------------------------------------------------------------------
//  C++ 関数の定義
//---------------------------------------------------------------------------

namespace nn { namespace os {

namespace {

    NN_FORCEINLINE detail::MultiWaitImpl& GetMultiWaitImpl(MultiWaitType* p) NN_NOEXCEPT
    {
        return Get( p->_multiWaitImpl );
    }

    NN_FORCEINLINE MultiWaitHolderType* CastToMultiWaitHolder(detail::MultiWaitHolderBase* p) NN_NOEXCEPT
    {
        return reinterpret_cast<MultiWaitHolderType*>( p );
    }

#if !defined(NN_SDK_BUILD_RELEASE)
    #define NN_OS_ASSERT_IF_WAITING(p) \
                do \
                { \
                    NN_SDK_ASSERT_EQUAL(false, (p)->_isWaiting); \
                    os::FenceMemoryLoadAny(); \
                } while(NN_STATIC_CONDITION(0))
    #define NN_OS_PRE_PROCESS_TO_WAITANY(p) \
                do \
                { \
                    p->_isWaiting = true; \
                    os::FenceMemoryStoreAny(); \
                } while(NN_STATIC_CONDITION(0))
    #define NN_OS_POST_PROCESS_TO_WAITANY(p) \
                do \
                { \
                    os::FenceMemoryAnyStore(); \
                    p->_isWaiting = false; \
                } while(NN_STATIC_CONDITION(0))
#else
    #define NN_OS_ASSERT_IF_WAITING(p) \
                do { } while(NN_STATIC_CONDITION(0))
    #define NN_OS_PRE_PROCESS_TO_WAITANY(p) \
                do { } while(NN_STATIC_CONDITION(0))
    #define NN_OS_POST_PROCESS_TO_WAITANY(p) \
                do { } while(NN_STATIC_CONDITION(0))
#endif

}

//---------------------------------------------------------------------------
//  MultiWaitType オブジェクトの初期化
void InitializeMultiWait( MultiWaitType* pMultiWait ) NN_NOEXCEPT
{
    auto&   multiWait = Get( pMultiWait->_multiWaitImpl );

    // 初期化処理（コンストラクタを呼び出す）
    new( &multiWait ) detail::MultiWaitImpl();

#if !defined(NN_SDK_BUILD_RELEASE)
    pMultiWait->_isWaiting = false;
    os::FenceMemoryStoreStore();
#endif

    // MultiWaitType を Initialized 状態へ（最後に行なう）
    pMultiWait->_state = MultiWaitType::State_Initialized;
}


//---------------------------------------------------------------------------
//  MultiWaitType オブジェクトのファイナライズ
void FinalizeMultiWait( MultiWaitType* pMultiWait ) NN_NOEXCEPT
{
    auto&   multiWait = Get( pMultiWait->_multiWaitImpl );

    // 事前条件
    NN_SDK_REQUIRES( pMultiWait->_state == MultiWaitType::State_Initialized, NN_TEXT_OS("nn::os::FinalizeMultiWait(): 指定された多重待ちオブジェクトが初期化されていません。") );
    NN_SDK_REQUIRES( multiWait.IsListEmpty(), NN_TEXT_OS("nn::os::FinalizeMultiWait(): 指定された多重待ちオブジェクトには多重待ちホルダーがリンクされています。") );

    // MultiWaitType を NotInitialized 状態へ（最初に行なう）
    pMultiWait->_state = MultiWaitType::State_NotInitialized;

    // 終了処理（デストラクタを呼び出す）
    multiWait.~MultiWaitImpl();
}


//---------------------------------------------------------------------------
//  対象 multiWait オブジェクトのいずれかの条件が揃うまで待機
MultiWaitHolderType* WaitAny( MultiWaitType* pMultiWait ) NN_NOEXCEPT
{
    auto&   multiWait = GetMultiWaitImpl( pMultiWait );

    // 事前条件チェック
    NN_SDK_REQUIRES( pMultiWait->_state == MultiWaitType::State_Initialized, NN_TEXT_OS("nn::os::WaitAny(): 指定された多重待ちオブジェクトが初期化されていません。") );
    NN_SDK_REQUIRES( multiWait.IsListNotEmpty(), NN_TEXT_OS("nn::os::WaitAny(): 指定された多重待ちオブジェクトに多重待ちホルダーがリンクされていません。") );

    NN_OS_PRE_PROCESS_TO_WAITANY(pMultiWait);
    auto* holder = CastToMultiWaitHolder( multiWait.WaitAny() );
    NN_OS_POST_PROCESS_TO_WAITANY(pMultiWait);
    NN_SDK_ASSERT( holder != NULL );
    return holder;
}


//---------------------------------------------------------------------------
//  対象 multiWait オブジェクトのいずれかの条件が揃っているか確認（ポーリング）
MultiWaitHolderType* TryWaitAny( MultiWaitType* pMultiWait ) NN_NOEXCEPT
{
    auto&   multiWait = GetMultiWaitImpl( pMultiWait );

    // 事前条件チェック
    NN_SDK_REQUIRES( pMultiWait->_state == MultiWaitType::State_Initialized, NN_TEXT_OS("nn::os::TryWaitAny(): 指定された多重待ちオブジェクトが初期化されていません。") );
    NN_SDK_REQUIRES( multiWait.IsListNotEmpty(), NN_TEXT_OS("nn::os::TryWaitAny(): 指定された多重待ちオブジェクトに多重待ちホルダーがリンクされていません。") );

    NN_OS_PRE_PROCESS_TO_WAITANY(pMultiWait);
    auto pHolder = CastToMultiWaitHolder( multiWait.TryWaitAny() );
    NN_OS_POST_PROCESS_TO_WAITANY(pMultiWait);
    return pHolder;
}


//---------------------------------------------------------------------------
//  対象 multiWait オブジェクトのいずれかの条件が揃うまで待機（タイムアウト付き）
MultiWaitHolderType* TimedWaitAny( MultiWaitType* pMultiWait, TimeSpan timeout) NN_NOEXCEPT
{
    auto&   multiWait = GetMultiWaitImpl( pMultiWait );

    // 事前条件チェック
    NN_SDK_REQUIRES( pMultiWait->_state == MultiWaitType::State_Initialized, NN_TEXT_OS("nn::os::TimedWaitAny(): 指定された多重待ちオブジェクトが初期化されていません。") );
    NN_SDK_REQUIRES( multiWait.IsListNotEmpty(), NN_TEXT_OS("nn::os::TimedWaitAny(): 指定された多重待ちオブジェクトに多重待ちホルダーがリンクされていません。") );
    NN_SDK_REQUIRES( timeout.GetNanoSeconds() >= 0, NN_TEXT_OS("nn::os::TimedWaitAny(): timeout 値が不正です。") );

    NN_OS_PRE_PROCESS_TO_WAITANY(pMultiWait);
    auto pHolder = CastToMultiWaitHolder( multiWait.TimedWaitAny( timeout) );
    NN_OS_POST_PROCESS_TO_WAITANY(pMultiWait);
    return pHolder;
}


//---------------------------------------------------------------------------
//  MultiWaitHolderType オブジェクトのファイナライズ
void FinalizeMultiWaitHolder( MultiWaitHolderType* pHolder ) NN_NOEXCEPT
{
    auto holder = reinterpret_cast<detail::MultiWaitHolderBase*>( &Get(pHolder->_holderImpl) );
    NN_UNUSED(holder);  // Warning C4189 対策

    // 事前条件
    NN_SDK_REQUIRES( holder->IsNotLinked(), NN_TEXT_OS("nn::os::FinalizeMultiWaitHolder(): 指定された多重待ちホルダーが多重待ちオブジェクトにリンクされたままです。") );

    // MultiWaitHolderBase のデストラクタを呼出す
    holder->~MultiWaitHolderBase();
}


//---------------------------------------------------------------------------
//  multiWaitHolder を multiWait へ繋ぐ
void LinkMultiWaitHolder( MultiWaitType* pMultiWait, MultiWaitHolderType* pHolder) NN_NOEXCEPT
{
    auto& multiWait = Get( pMultiWait->_multiWaitImpl );
    auto  holder    = reinterpret_cast<detail::MultiWaitHolderBase*>( &Get(pHolder->_holderImpl) );

    // 事前条件
    NN_SDK_REQUIRES( pMultiWait->_state == MultiWaitType::State_Initialized, NN_TEXT_OS("nn::os::LinkMultiWaitHolder(): 指定された多重待ちオブジェクトが初期化されていません。") );
    NN_SDK_REQUIRES( holder->IsNotLinked(), NN_TEXT_OS("nn::os::LinkMultiWaitHolder(): 指定された多重待ちホルダーは既に多重待ちオブジェクトにリンクされています。") );
    NN_OS_ASSERT_IF_WAITING(pMultiWait);

    // リストへ繋ぐ
    multiWait.PushBackToList( *holder );
    holder->SetMultiWait( &multiWait );
}


//---------------------------------------------------------------------------
//  multiWaitHolder を multiWait から外す
void UnlinkMultiWaitHolder( MultiWaitHolderType* pHolder) NN_NOEXCEPT
{
    auto holder = reinterpret_cast<detail::MultiWaitHolderBase*>( &Get(pHolder->_holderImpl) );

    // 事前条件
    NN_SDK_REQUIRES( holder->IsLinked(), NN_TEXT_OS("nn::os::UnlinkMultiWaitHolder(): 指定された多重待ちホルダーはどこにもリンクされていません。") );
#if !defined(NN_SDK_BUILD_RELEASE)
    {
        auto pImpl = reinterpret_cast<char*>(holder->GetMultiWait());
        auto pMultiWait = reinterpret_cast<MultiWaitType*>(pImpl - offsetof(MultiWaitType, _multiWaitImpl));
        NN_OS_ASSERT_IF_WAITING(pMultiWait);
    }
#endif

    // リストから外す
    holder->GetMultiWait()->EraseFromList( *holder );
    holder->SetMultiWait( NULL );
}


//---------------------------------------------------------------------------
//  multiWait からすべて外す
void UnlinkAllMultiWaitHolder( MultiWaitType* pMultiWait) NN_NOEXCEPT
{
    auto& multiWait = Get( pMultiWait->_multiWaitImpl );

    // 事前条件
    NN_SDK_REQUIRES( pMultiWait->_state == MultiWaitType::State_Initialized, NN_TEXT_OS("nn::os::UnlinkAllMultiWaitHolder(): 指定された多重待ちオブジェクトが初期化されていません。") );
    NN_OS_ASSERT_IF_WAITING(pMultiWait);

    multiWait.EraseAllFromList();
}


//---------------------------------------------------------------------------
//  pDst から pSrc に全て移動する
void MoveAllMultiWaitHolder( MultiWaitType* pDst, MultiWaitType* pSrc) NN_NOEXCEPT
{
    auto& dst = Get( pDst->_multiWaitImpl );
    auto& src = Get( pSrc->_multiWaitImpl );

    // 事前条件
    NN_SDK_REQUIRES( pDst->_state == MultiWaitType::State_Initialized, NN_TEXT_OS("nn::os::MoveAllMultiWaitHolder(): 第一引数に指定された多重待ちオブジェクトが初期化されていません。") );
    NN_SDK_REQUIRES( pSrc->_state == MultiWaitType::State_Initialized, NN_TEXT_OS("nn::os::MoveAllMultiWaitHolder(): 第二引数に指定された多重待ちオブジェクトが初期化されていません。") );
    NN_OS_ASSERT_IF_WAITING(pDst);
    NN_OS_ASSERT_IF_WAITING(pSrc);

    dst.MoveAllFromOther(&src);
}


//---------------------------------------------------------------------------
//  multiWaitHolder のユーザ変数に値を設定
void SetMultiWaitHolderUserData( MultiWaitHolderType* pHolder, uintptr_t userData) NN_NOEXCEPT
{
    pHolder->userData = userData;
}


//---------------------------------------------------------------------------
//  multiWaitHolder のユーザ変数から値を取得
uintptr_t GetMultiWaitHolderUserData( const MultiWaitHolderType* pHolder) NN_NOEXCEPT
{
    return pHolder->userData;
}


//---------------------------------------------------------------------------
//  MultiWaitHolderType の初期化（nn::os::NativeHandle を関連付ける）
void InitializeMultiWaitHolder(MultiWaitHolderType* pHolder, NativeHandle handle) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( handle != nn::os::InvalidNativeHandle, NN_TEXT_OS("nn::os::InitializeMultiWaitHolder(): handle が有効なハンドルではありません。") );

    // 初期化処理（コンストラクタを呼び出す）
    // Get() がなくてもポインタは得られるが、TypedStorage のサイズと
    // アライメントチェックのため、Get() を使って実装しておく。
    new(&Get(pHolder->_holderImpl)) detail::MultiWaitHolderOfNativeHandle(handle);

    //  構造体メンバの初期化
    pHolder->userData = 0;
}


}} // namespace nn::os

