﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_Abort.h>
#include <nn/diag/text/diag_SdkTextOs.h>
#include <nn/os/os_MemoryHeapApi.h>

#include "detail/os_Diag.h"
#include "detail/os_Common.h"
#include "detail/os_MemoryHeapManager.h"


//--------------------------------------------------------------------------

namespace nn { namespace os {

//--------------------------------------------------------------------------
//  メモリヒープサイズの変更
Result SetMemoryHeapSize(size_t size) NN_NOEXCEPT
{
    // 事前条件
    NN_SDK_REQUIRES( detail::IsAligned( size, MemoryHeapUnitSize ), NN_TEXT_OS("nn::os::SetMemoryHeapSize(): size が MemoryHeapUnitSize の整数倍ではありません（size=0x%zx）"), size );

    // メモリヒープ領域のサイズを変更する（size=0 も可）
    return  detail::GetMemoryHeapManagerInstance()->SetHeapSize( size );
}


//--------------------------------------------------------------------------
//  メモリヒープアドレスの取得
uintptr_t GetMemoryHeapAddress() NN_NOEXCEPT
{
    // メモリヒープ領域の先頭アドレスを取得
    return  detail::GetMemoryHeapManagerInstance()->GetHeapAddress();
}


//--------------------------------------------------------------------------
//  メモリヒープサイズの取得
size_t GetMemoryHeapSize() NN_NOEXCEPT
{
    // メモリヒープ領域のサイズを取得
    return  detail::GetMemoryHeapManagerInstance()->GetHeapSize();
}


//--------------------------------------------------------------------------
//  メモリヒープ領域からメモリブロックを獲得
Result AllocateMemoryBlock(uintptr_t* address, size_t size) NN_NOEXCEPT
{
    // 事前条件
    NN_SDK_REQUIRES( size > 0, NN_TEXT_OS("nn::os::AllocateMemoryBlock(): size に 0 もしくはマイナス値が指定されました。") );
    NN_SDK_REQUIRES( detail::IsAligned( size, MemoryBlockUnitSize ), NN_TEXT_OS("nn::os::AllocateMemoryBlock(): size が MemoryBlockUnitSize の整数倍ではありません（size=0x%zx）"), size );

    // メモリヒープ領域からメモリブロックを取得
    return detail::GetMemoryHeapManagerInstance()->AllocateFromHeap( address, size );
}


//--------------------------------------------------------------------------
//  メモリヒープ領域へメモリブロックを返却
void FreeMemoryBlock(uintptr_t address, size_t size) NN_NOEXCEPT
{
    detail::MemoryHeapManager* manager = detail::GetMemoryHeapManagerInstance();

    // 事前条件
    NN_SDK_REQUIRES( detail::IsAligned( address, MemoryBlockUnitSize ), NN_TEXT_OS("nn::os::FreeMemoryBlock(): address が MemoryBlockUnitSize の整数倍ではありません（address=0x%p）"), address );

    NN_SDK_REQUIRES( size > 0, NN_TEXT_OS("nn::os::FreeMemoryBlock(): size > 0 ではありません（size=0x%zx）"), size );

    NN_SDK_REQUIRES( detail::IsAligned( size, MemoryBlockUnitSize ), NN_TEXT_OS("nn::os::FreeMemoryBlock(): size が MemoryBlockUnitSize の整数倍ではありません（size=0x%zx）"), size );

    // 指定されたメモリブロックが、ヒープ領域内かどうかをチェック
    NN_ABORT_UNLESS( manager->IsRegionInMemoryHeap( address, size ), NN_TEXT_OS("nn::os::FreeMemoryBlock(): 指定されたメモリ領域がメモリヒープ内ではありません。(address=0x%p, size=0x%zx)"), address, size);

    // メモリヒープ領域へメモリブロックを返却
    manager->ReleaseToHeap( address, size );
}


}} // namespace nn::os


