﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <new>
#include <mutex>

#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/diag/text/diag_SdkTextOs.h>
#include <nn/os/os_EventCommon.h>
#include <nn/os/os_InterruptEventCommon.h>
#include <nn/os/os_InterruptEventTypes.h>
#include <nn/os/os_InterruptEventApi.h>

#include "detail/os_Diag.h"
#include "detail/os_Common.h"
#include "detail/os_TimeoutHelper.h"
#include "detail/os_InterruptEventImpl.h"
#include "detail/os_MultipleWaitHelper.h"
#include "detail/os_MultipleWaitHolderOfInterruptEvent.h"
#include "detail/os_MultipleWaitHolderImpl.h"


//---------------------------------------------------------------------------
//  C++ 関数の定義
//---------------------------------------------------------------------------

namespace nn { namespace os {

//---------------------------------------------------------------------------
//  InterruptEventType オブジェクトのイニシャライズ
void InitializeInterruptEvent(InterruptEventType* event, InterruptName name, EventClearMode clearMode) NN_NOEXCEPT
{
    // メンバの初期化
    event->_clearMode    = static_cast<uint8_t>( clearMode );

    // ターゲット依存の初期化（コンストラクタを呼ぶ）
    new( &event->_impl ) detail::InterruptEventImpl( name, clearMode );

    // 割込みイベントを Initialized 状態へ（最後に行なう）
    event->_state        = InterruptEventType::State_Initialized;
}


//---------------------------------------------------------------------------
//  InterruptEventType オブジェクトのファイナライズ
void FinalizeInterruptEvent(InterruptEventType* event) NN_NOEXCEPT
{
    // InterruptEventType オブジェクトが Initialized 状態でなければ実行停止
    NN_SDK_REQUIRES( event->_state == InterruptEventType::State_Initialized, NN_TEXT_OS("nn::os::FinalizeInterruptEvent(): 指定された割込みイベントが初期化されていません。") );

    // 割込みイベントを NotInitialized 状態へ（最初に行なう）
    event->_state  = InterruptEventType::State_NotInitialized;

    // ターゲット依存の終了処理（デストラクタを呼ぶ）
    Get(event->_impl).~InterruptEventImpl();
}


//---------------------------------------------------------------------------
//  InterruptEventType をクリア状態へ
void ClearInterruptEvent(InterruptEventType* event) NN_NOEXCEPT
{
    // InterruptEventType オブジェクトが Initialized 状態でなければ実行停止
    NN_SDK_REQUIRES( event->_state == InterruptEventType::State_Initialized, NN_TEXT_OS("nn::os::ClearInterruptEvent(): 指定された割込みイベントは初期化されていません。") );

    // ターゲット依存のクリア処理
    Get( event->_impl ).Clear();
}


//---------------------------------------------------------------------------
//  InterruptEventType がシグナル状態になるまで待機
void WaitInterruptEvent(InterruptEventType* event) NN_NOEXCEPT
{
    // 事前条件
    NN_SDK_REQUIRES( event->_state == InterruptEventType::State_Initialized, NN_TEXT_OS("nn::os::WaitInterruptEvent(): 指定された割込みイベントが初期化されていません。") );

    // ターゲット依存の待機処理
    Get( event->_impl ).Wait();
}


//---------------------------------------------------------------------------
//  InterruptEventType がシグナル状態になるまで待機（ポーリング）
bool TryWaitInterruptEvent(InterruptEventType* event) NN_NOEXCEPT
{
    // 事前条件
    NN_SDK_REQUIRES( event->_state == InterruptEventType::State_Initialized, NN_TEXT_OS("nn::os::TryWaitInterruptEvent(): 指定された割込みイベントが初期化されていません。") );

    // ターゲット依存のポーリング
    return Get( event->_impl ).TryWait();
}


//---------------------------------------------------------------------------
//  InterruptEventType がシグナル状態になるまで待機（タイムアウト付き）
bool TimedWaitInterruptEvent(InterruptEventType* event, TimeSpan timeout) NN_NOEXCEPT
{
    // 事前条件
    NN_SDK_REQUIRES( event->_state == InterruptEventType::State_Initialized, NN_TEXT_OS("nn::os::TimedWaitInterruptEvent(): 指定された割込みイベントが初期化されていません。") );
    NN_SDK_REQUIRES( timeout.GetNanoSeconds() >= 0, NN_TEXT_OS("nn::os::TimedWaitInterruptEvent(): timeout 値が不正です。") );

    // ターゲット依存のタイムアウト付き待機処理
    return Get( event->_impl ).TimedWait( timeout );
}


//---------------------------------------------------------------------------
//  MultiWaitHolderType の初期化（InterruptEventType を関連付ける）
void InitializeMultiWaitHolder(MultiWaitHolderType* pHolder, InterruptEventType* event) NN_NOEXCEPT
{
    // 事前条件
    NN_SDK_REQUIRES( event->_state == InterruptEventType::State_Initialized, NN_TEXT_OS("nn::os::InitializeMultiWaitHolder(): 指定された割込みイベントが初期化されていません。") );

    // 初期化処理（コンストラクタを呼び出す）
    // Get() がなくてもポインタは得られるが、TypedStorage のサイズと
    // アライメントチェックのため、Get() を使って実装しておく。
    new( &Get(pHolder->_holderImpl) ) detail::MultiWaitHolderOfInterruptEvent( event );

    //  構造体メンバの初期化
    pHolder->userData = 0;
}


}} // namespace nn::os

