﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Abort.h>
#include <nn/diag/text/diag_SdkTextOs.h>
#include <nn/util/util_BitUtil.h>
#include <nn/os/os_ThreadTypes.h>
#include <nn/os/os_FiberCommon.h>
#include <nn/os/os_FiberTypes.h>
#include <nn/os/os_FiberApi.h>

#include "detail/os_Diag.h"
#include "detail/os_Common.h"
#include "detail/os_FiberImpl.h"
#include "detail/os_ThreadManager.h"
#include "detail/os_StackGuardManager.h"

#include <new>
#include <mutex>
#include <cstring>

namespace nn { namespace os {

//-----------------------------------------------------------------------------

namespace {

    //-------------------------------------------------------------------------
    //  nextFiber の指している遷移先の実ファイバオブジェクトを返す。
    //
    inline FiberType* GetRealNextFiber(FiberType* nextFiber) NN_NOEXCEPT
    {
        if (nextFiber == NULL)
        {
            //  NULL の場合には起動元スレッドを返すが、カレントスレッド自身が
            //  一度もファイバ起動を行なっていない場合には結局 NULL が返る。
            return detail::GetCurrentThread()->_initialFiber;
        }

        // 切り替え先ファイバ状態のチェック
        auto state = nextFiber->_state;
        if ( state == FiberType::State_Ready )
        {
            return nextFiber;
        }

        if (state == FiberType::State_Completed)
        {
            NN_ABORT(NN_TEXT_OS("終了済みのファイバへは遷移できません。"));
        }
        if (state == FiberType::State_Running)
        {
            // カレントファイバを指定した場合もこれに引っかかる
            NN_ABORT(NN_TEXT_OS("動作中のファイバへは遷移できません。"));
        }

        NN_ABORT(NN_TEXT_OS("指定されたファイバの状態が異常です。"));
    }

    //-------------------------------------------------------------------------
    //  ファイバ関数を起動／終了するためのラッパー関数
    //
    void FiberWrapper(FiberType* currentFiber) NN_NOEXCEPT
    {
        // ユーザが定義したファイバ関数の呼び出し
        FiberType* nextFiber = detail::CallFiberFunction(currentFiber->_function, currentFiber->_argument);

        // 切り替え先ファイバの取得と状態チェック
        // ラッパー関数からの呼び出しなので NULL は返らない。
        nextFiber = GetRealNextFiber( nextFiber );
        NN_SDK_ASSERT(nextFiber != NULL);

        // ファイバからファイバへ遷移
        currentFiber->_state = FiberType::State_Completed;
        nextFiber->_state    = FiberType::State_Running;

        detail::GetCurrentThread()->_currentFiber = nextFiber;

        // この関数の中で実際の切換えが起こる
        detail::FiberImpl::Switch(currentFiber, nextFiber);

        NN_ABORT(NN_TEXT_OS("nn::os::InitializeFiber(): 終了したファイバへは遷移できません。"));
    }

    //-------------------------------------------------------------------------
    //  FiberType オブジェクトの初期化
    //  ただし、Impl 部分の初期化は行なわない
    //
    inline void SetupFiberType(FiberType* fiber, FiberFunction function, void* argument, void* stack, size_t stackSize) NN_NOEXCEPT
    {
        fiber->_stackIsAliased  = false;

        fiber->_function        = function;
        fiber->_argument        = argument;
        fiber->_originalStack   = stack;
        fiber->_stack           = stack;
        fiber->_stackSize       = stackSize;

        fiber->_state           = FiberType::State_Ready;
    }

    //-------------------------------------------------------------------------
    //  エイリアススタックを構築する。
    //  エイリアスされたスタックの前後 4KB にはガード用のページが設定される。
    //
    bool AliasFiberStack( FiberType* fiber ) NN_NOEXCEPT
    {
        void*  originalStack = fiber->_stack;
        size_t stackSize     = fiber->_stackSize;

        // TORIAEZU: 64 回までリトライする
        for (int i=0; i<64; ++i)
        {
            // スタックガード用の空間から空き空間を獲得する。
            // 第１引数にはガード領域を含まないスタックサイズを渡し、
            // 返値にはスタックをマップすべきガードを含まない先頭アドレスが返る。
            void* newStack = detail::GetStackGuardManagerInstance()->AllocateStackGuardSpace( stackSize );
            if (!newStack)
            {
                // ここで取れない場合は即失敗
                return false;
            }

            // スタックのエイリアスを構築する
            bool isMapSuccess = detail::FiberImpl::MapAliasStack(newStack, originalStack, stackSize);
            if (isMapSuccess)
            {
                // 前後にガード空間があるかチェック
                if (!detail::GetStackGuardManagerInstance()->CheckGuardSpace(reinterpret_cast<uintptr_t>(newStack), stackSize))
                {
                    // なかった場合、アンマップしてリトライ
                    bool ret = detail::FiberImpl::UnmapAliasStack(newStack, originalStack, stackSize);
                    NN_SDK_ASSERT( ret, NN_TEXT_OS("nn::os::InitializeFiber(): スタック領域のエイリアス解除に失敗しました。"));
                    NN_UNUSED(ret);
                }
                else
                {
                    // エイリアスされたスタック情報を設定
                    fiber->_stackIsAliased  = true;
                    fiber->_stack           = newStack;
                    return true;
                }
            }
            else
            {
                continue;
            }
        }
        return false;
    }

    //-------------------------------------------------------------------------
    //  エイリアススタックを解除する。
    //
    void UnaliasFiberStack( FiberType* fiber ) NN_NOEXCEPT
    {
        // SVC を発行して、スタックのエイリアスを解除する
        bool ret = detail::FiberImpl::UnmapAliasStack( fiber->_stack,
                                                          fiber->_originalStack,
                                                          fiber->_stackSize );
        NN_SDK_ASSERT( ret, NN_TEXT_OS("nn::os::FinalizeFiber(): スタック領域のエイリアス解除に失敗しました。"));
        NN_UNUSED(ret);

        // エイリアス解除されたスタック情報を設定
        fiber->_stackIsAliased  = false;
        fiber->_stack           = fiber->_originalStack;
    }

}   // namespace


//-----------------------------------------------------------------------------
//  ファイバの初期化
void InitializeFiber(FiberType* fiber, FiberFunction function, void* argument, void* stack, size_t stackSize, int fiberFlag) NN_NOEXCEPT
{
    bool enableStackGuard = !(fiberFlag & FiberFlag_NoStackGuard);

    const size_t align = enableStackGuard ? GuardedStackAlignment
                                          : FiberStackAlignment;
    NN_SDK_ASSERT( stack != NULL && util::is_aligned(reinterpret_cast<uintptr_t>(stack), align), "A stack(=0x%p) is not aligned with 0x%zx.", stack, align);

    NN_SDK_ASSERT( stackSize > 0 && util::is_aligned(stackSize, align), "A stackSize(=0x%zx) is not aligned with 0x%zx.", stackSize, align);

    NN_UNUSED( align );


    // FiberType オブジェクトの初期化
    SetupFiberType(fiber, function, argument, stack, stackSize);

    if (enableStackGuard)
    {
        // ファイバのスタック領域をエイリアスする
        bool ret = AliasFiberStack( fiber );
        NN_SDK_ASSERT(ret, NN_TEXT_OS("nn::os::InitializeFiber(): スタック領域のエイリアス構築に失敗しました。"));
        NN_UNUSED(ret);
    }

    // Impl 部分の初期化
    detail::FiberImpl::Initialize( fiber, FiberWrapper );
}


//-----------------------------------------------------------------------------
//  ファイバの破棄
void FinalizeFiber(FiberType* fiber) NN_NOEXCEPT
{
    // 事前条件
    NN_SDK_REQUIRES( fiber->_state == FiberType::State_Ready ||
                     fiber->_state == FiberType::State_Completed,
        NN_TEXT_OS("nn::os::FinalizeFiber(): ファイバが破棄可能な状態ではありません。") );

    // Impl 部分の破棄
    detail::FiberImpl::Finalize( fiber );

    // スタック領域がエイリアスされていれば解除する
    if ( fiber->_stackIsAliased )
    {
        // エイリアスの解除
        UnaliasFiberStack( fiber );
    }

    fiber->_state = FiberType::State_NotInitialized;
}


//-----------------------------------------------------------------------------
//  対象 Fiber の実行を開始
//
//  以下の 4 パターンに対応する。
//  1) NULL  → Fiber: スレッドからファイバへ遷移（元スレッドからファイバ起動）
//  2) Fiber → Fiber: ファイバからファイバへ遷移（ファイバ間切替え）
//  3) Fiber → NULL : ファイバからスレッドへ遷移（元スレッドへ復帰）
//  4) NULL  → NULL : スレッドからスレッドへ遷移（事前条件違反）
//
void SwitchToFiber(FiberType* nextFiber) NN_NOEXCEPT
{
    // 切り替え先ファイバの取得と状態チェック
    nextFiber = GetRealNextFiber( nextFiber );

    // スレッドからスレッドへの遷移は事前条件違反
    NN_SDK_REQUIRES(nextFiber != NULL);

    ThreadType* currentThread = detail::GetCurrentThread();
    FiberType*  currentFiber  = currentThread->_currentFiber;
    if (currentFiber == NULL)
    {
        FiberType   initialFiber;

        // カレントスレッド自身のファイバオブジェクトを構築
        SetupFiberType(&initialFiber,
                       reinterpret_cast<FiberFunction>(currentThread->_threadFunction),
                       currentThread->_argument,
                       currentThread->_stack,
                       currentThread->_stackSize);

        // Impl 部分の初期化
        detail::FiberImpl::ConvertThreadToFiber(&initialFiber);

        // スレッドからファイバへ遷移
        initialFiber._state = FiberType::State_Ready;
        nextFiber->_state   = FiberType::State_Running;

        currentThread->_initialFiber = &initialFiber;
        currentThread->_currentFiber = nextFiber;

        // この関数の中で実際の切換えが起こる
        detail::FiberImpl::Switch(&initialFiber, nextFiber);

        // ファイバからスレッドへ復帰
        currentThread->_initialFiber = NULL;
        currentThread->_currentFiber = NULL;

        // Impl 部分の破棄
        detail::FiberImpl::ConvertFiberToThread(&initialFiber);
        return;
    }

    // ファイバからファイバへ遷移
    currentFiber->_state = FiberType::State_Ready;
    nextFiber->_state    = FiberType::State_Running;

    currentThread->_currentFiber = nextFiber;

    // この関数の中で実際の切換えが起こる
    detail::FiberImpl::Switch(currentFiber, nextFiber);
}

//-----------------------------------------------------------------------------
//  カレントファイバの取得
FiberType* GetCurrentFiber() NN_NOEXCEPT
{
    return detail::GetCurrentThread()->_currentFiber;
}


//-----------------------------------------------------------------------------

}} // namespace nn::os

