﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/os.h>
#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_Abort.h>
#include <nn/os/os_Config.h>
#include <nn/svc/svc_Server.h>
#include <nn/svc/svc_Result.h>
#include <nn/result/result_HandlingUtility.h>

#include "os_AslrSpaceManagerTypes.h"
#include "os_AslrSpaceManager.h"

namespace nn { namespace os {
namespace detail {

//--------------------------------------------------------------------------

class UnsafeMemoryManagerImplByHorizon
{
public:
    static Result AllocateUnsafeMemoryImpl(uintptr_t* pOutAddress, size_t size)
    {
        // ASLR 領域からマップ先を確保
        uintptr_t mappingAddress = reinterpret_cast<uintptr_t>(detail::GetAslrSpaceManagerInstance()->AllocateSpace(size));
        if (mappingAddress == 0)
        {
            // 候補となる空き空間すらない
            return os::ResultOutOfVirtualAddressSpace();
        }

        NN_SDK_ASSERT( mappingAddress % MemoryPageSize == 0);

        // マップ
        auto result = svc::MapPhysicalMemoryUnsafe(mappingAddress, size);
        NN_RESULT_TRY(result)
            NN_RESULT_CATCH( svc::ResultInvalidCurrentMemory )
            {
                NN_RESULT_THROW( os::ResultInvalidCurrentMemoryState() );
            }
            NN_RESULT_CATCH( svc::ResultInvalidRegion )
            {
                NN_RESULT_THROW( os::ResultInvalidCurrentMemoryState() );
            }
            NN_RESULT_CATCH( svc::ResultOutOfResource )
            {
                NN_RESULT_THROW( os::ResultOutOfResource() );
            }
            NN_RESULT_CATCH( svc::ResultOutOfMemory )
            {
                NN_RESULT_THROW( os::ResultOutOfMemory() );
            }
            NN_RESULT_CATCH( svc::ResultLimit )
            {
                NN_RESULT_THROW( os::ResultOutOfMemory() );
            }
            NN_RESULT_CATCH_ALL
            {
                NN_ABORT_UNLESS_RESULT_SUCCESS( result );
            }
        NN_RESULT_END_TRY

        // 前後にガード空間があるかチェック
        if (!detail::GetAslrSpaceManagerInstance()->CheckGuardSpace(mappingAddress, size))
        {
            // なかった場合、アンマップしてリトライ
            result = svc::UnmapPhysicalMemoryUnsafe(mappingAddress, size);

            // ここのアンマップに失敗した場合はどうしようもないのでアボート
            NN_ABORT_UNLESS_RESULT_SUCCESS( result );

            return os::ResultNoGuardSpace();
        }

        // マップしたアドレスを返す
        *pOutAddress = mappingAddress;
        NN_RESULT_SUCCESS;
    }

    static Result FreeUnsafeMemoryImpl(uintptr_t address, size_t size)
    {
        auto result = svc::UnmapPhysicalMemoryUnsafe(address, size);
        NN_RESULT_TRY(result)
            NN_RESULT_CATCH( svc::ResultInvalidRegion )
            {
                NN_RESULT_THROW( os::ResultInvalidParameter() );
            }
            NN_RESULT_CATCH( svc::ResultInvalidCurrentMemory )
            {
                NN_RESULT_THROW( os::ResultBusy() );
            }
            NN_RESULT_CATCH( svc::ResultOutOfResource )
            {
                NN_RESULT_THROW( os::ResultOutOfResource() );
            }
            NN_RESULT_CATCH( svc::ResultOutOfMemory )
            {
                NN_RESULT_THROW( os::ResultOutOfResource() );
            }
            NN_RESULT_CATCH_ALL
            {
                NN_ABORT_UNLESS_RESULT_SUCCESS( result );
            }
        NN_RESULT_END_TRY

        NN_RESULT_SUCCESS;
    }
};

typedef UnsafeMemoryManagerImplByHorizon UnsafeMemoryManagerImpl;

//--------------------------------------------------------------------------

}  // namespace detail
}} // namespace nn::os

