﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "os_Common.h"

#include <nn/nn_Common.h>
#include <nn/diag/text/diag_SdkTextOs.h>
#include <nn/nn_Windows.h>

#include <nn/nn_SdkAssert.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkLog.h>
#include <nn/os/os_Result.h>
#include <nn/os/os_Types.h>
#include <nn/os/os_NativeHandleTypes.h>
#include <nn/os/os_TransferMemoryApi.h>
#include <nn/os/os_TransferMemoryTypes.h>

#include "os_Diag.h"
#include "os_TransferMemoryImpl.h"

namespace nn { namespace os {
namespace detail {

//--------------------------------------------------------------------------
//  Transfer memory の作成（ハンドルの生成）
nn::Result TransferMemoryImplByWin32::Create(NativeHandle* outHandle, void* address, size_t size, os::MemoryPermission permission) NN_NOEXCEPT
{
    NN_UNUSED(permission);
    NN_UNUSED(address);

    DWORD sizeHigh = static_cast<uint64_t>(size) >> 32;
    DWORD sizeLow  = static_cast<DWORD>(size & 0xffffffff);
    HANDLE handle = CreateFileMappingA(INVALID_HANDLE_VALUE, NULL, PAGE_READWRITE, sizeHigh, sizeLow, NULL);

    if (handle == NULL)
    {
        return nn::os::ResultInvalidParameter();
    }

    *outHandle = reinterpret_cast<NativeHandle>( handle );
    return ResultSuccess();
}

//--------------------------------------------------------------------------
//  Transfer memory の削除（ハンドルを削除）
void TransferMemoryImplByWin32::Close(NativeHandle handle) NN_NOEXCEPT
{
    HANDLE winHandle = reinterpret_cast<HANDLE>(handle);

    BOOL ret = CloseHandle(winHandle);
    if (ret == 0)
    {
        // TORIAEZU:
        // 単一プロセス内で複数回クローズが呼ばれた場合の警告。
        // API が Attach/Detach 体系になった時に修正する。
        NN_SDK_LOG(NN_TEXT_OS("nn::os::DestroyTransferMemory(): ハンドルは既にクローズされています。"));
    }
}

//--------------------------------------------------------------------------
//  Transfer memory をマップ
nn::Result TransferMemoryImplByWin32::Map(void** pOutAddress, NativeHandle handle, void* address, size_t size, os::MemoryPermission ownerPermission) NN_NOEXCEPT
{
    NN_UNUSED(address);
    NN_UNUSED(ownerPermission);

    HANDLE winHandle = reinterpret_cast<HANDLE>(handle);

    LPVOID mappedAddress = MapViewOfFileEx(winHandle, FILE_MAP_ALL_ACCESS, 0, 0, size, NULL);
    if (mappedAddress)
    {
        *pOutAddress = mappedAddress;
        return nn::ResultSuccess();
    }

    NN_ABORT(NN_TEXT_OS("nn::os::MapTransferMemory(): マップに失敗しました（err = 0x%08x）"), GetLastError());
}

//--------------------------------------------------------------------------
//  Transfer memory をアンマップ
void TransferMemoryImplByWin32::Unmap(NativeHandle handle, void* address, size_t size) NN_NOEXCEPT
{
    NN_UNUSED(handle);
    NN_UNUSED(size);

    BOOL ret = UnmapViewOfFile(address);
    if (ret)
    {
        return;
    }

    NN_ABORT(NN_TEXT_OS("nn::os::UnmapTransferMemory(): アンマップに失敗しました（err = 0x%08x）"), GetLastError());
}

//--------------------------------------------------------------------------

}  // namespace detail
}} // namespace nn::os

