﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/os/os_Config.h>
#include <nn/nn_TimeSpan.h>
#include <nn/os/os_Tick.h>

#include <algorithm>
#include "os_Common.h"
#include "os_TickManager.h"
#include "os_TimeoutHelper-os.win32.h"

#if !defined(NN_BUILD_CONFIG_OS_SUPPORTS_WIN32)
    #error "OS 種別として Win32 が指定されていません。"
#endif


namespace nn { namespace os {
namespace detail {

//--------------------------------------------------------------------------
// 指定された時間だけスリープする
void TimeoutHelperImplByWin32::Sleep(TimeSpan time) NN_NOEXCEPT
{
    if (time == 0)
    {
        return;
    }

    // タイムアウトが成立する絶対時間を計算しておく
    // 指定時間よりも早くタイムアウトすることを防ぐために +1 tick しておく
    auto*    tickManager     = GetTickManagerInstance();
    uint64_t currentTickVal  = tickManager->GetTick().GetInt64Value();
    uint64_t timeoutTickVal  = tickManager->ConvertToTick(time).GetInt64Value();
    uint64_t absoluteEndTick = currentTickVal + timeoutTickVal + 1;

    // オーバーフローしないよう調整
    auto endTick = Tick(std::min<uint64_t>((std::numeric_limits<int64_t>::max)(), absoluteEndTick));

    for (;;)
    {
        // 現在時間を取得し、目標までまだ時間があるならスリープ
        Tick currTick = tickManager->GetTick();
        if (currTick >= endTick)
        {
            return;
        }

        // Win32 の時間に変換してスリープ
        ::Sleep( ConvertToImplTime(endTick - currTick) );
    }
}

}   // namespace detail
}}  // namespace nn::os

