﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "os_Common.h"
#include <nn/os/os_Config.h>
#include <nn/diag/text/diag_SdkTextOs.h>

#if !defined(NN_BUILD_CONFIG_OS_SUPPORTS_WIN32)
    #error "OS 種別として Win32 が指定されていません。"
#endif

#include <cstdio>
#include <cstdlib>
#include <mutex>
#include <process.h>

#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_Abort.h>
#include <nn/util/util_IntrusiveList.h>
#include <nn/os/os_ThreadTypes.h>
#include <nn/os/os_ThreadApi.h>
#include <nn/os/detail/os_InternalCriticalSection.h>
#include <nn/util/util_BitUtil.h>

#include "os_Diag.h"
#include "os_ThreadManager.h"

#include <nn/nn_Windows.h>
#pragma warning( push )
#pragma warning ( disable : 4668 )
#include <winnt.h>
#pragma warning( pop )

namespace nn { namespace os {
namespace detail {

namespace {

    struct Arglist
    {
        void    (*function)(ThreadType *);
    };

    //-----------------------------------------------------------------------
    // スレッド関数の入口／出口
    unsigned __stdcall StdcallInvokeThread(void* p) NN_NOEXCEPT
    {
        ThreadType* thread = reinterpret_cast<ThreadType*>(p);
        Arglist*    args   = reinterpret_cast<Arglist*>(thread->_originalStack);

        // ここの function はユーザ関数エントリではなく、
        // ThreadManager が用意したスレッド関数ラッパーエントリを指す。
        args->function( thread );

        // 正常終了
        return 0;
    }

    //-----------------------------------------------------------------------
    // ThreadType オブジェクトを動的に遅延生成して管理対象にする
    ThreadType* DynamicAllocateAndRegisterThreadType() NN_NOEXCEPT
    {
        auto* threadManager = GetThreadManagerInstance();
        auto  thread = threadManager->AllocateThreadType();
        NN_ABORT_UNLESS( thread != NULL, NN_TEXT_OS("内部エラー：ThreadType オブジェクトの遅延確保に失敗しました。"));

        // スレッドオブジェクトの初期化
        SetupThreadObjectUnsafe(thread, NULL, NULL, NULL, 0, DefaultThreadPriority);
        thread->_state          = ThreadType::State_Started;
        thread->_autoRegistered = true;

        // カレントスレッドの Win32 HANDLE を取得
        thread->_handle._win32Handle = ::GetCurrentThread();

        // アフィニティ初期状態を設定
        thread->_handle._idealCore    = ::GetCurrentProcessorNumber();
        thread->_handle._affinityMask = nn::os::GetThreadAvailableCoreMask();

        // スレッドを管理対象へ配置。_win32Handle 設定後に呼ぶ必要がある
        threadManager->PlaceThreadObjectUnderThreadManagerSafe( thread );

        return thread;
    }

    //-----------------------------------------------------------------------
    // 遅延登録された Thread の最終後処理
    void CleanupDelayedRegisteredThread(ThreadType* thread) NN_NOEXCEPT
    {
        // 未知のスレッドは、std::thread::join() もしくは std::thread::detach()
        // のいずれかを行なわないとデストラクトできない仕様である。
        // 言い換えると、この２つのいずれかによって、スレッドコンテキストは
        // std::thread のインスタンスと切り離されており、実スレッドの
        // コンテキストの後処理を自スレッド以外から行なうことは出来ない。
        //
        // 結果として、本関数のように Win32 スレッド終了時のデストラクタ機構を
        // 利用して std::thread によって生成されたスレッドの破棄を行なう。

        // スレッド終了時の共通の後処理を行なう
        GetThreadManagerInstance()->CleanupThread();

        // 全スレッドを繋ぐリストから削除
        GetThreadManagerInstance()->EraseFromAllThreadsListSafe( thread );

        // 未知のスレッドは、GetCurrentThread() の中で ThreadType オブジェクト
        // を動的確保＆初期化して遅延登録されているため、最後にそれを開放する。
        GetThreadManagerInstance()->FreeThreadType( thread );
    }

    extern "C"
    void NTAPI ThreadTlsCallbackFunction(PVOID hinstDLL, DWORD fdwReason, PVOID lpvReserved)
    {
        NN_UNUSED(hinstDLL);
        NN_UNUSED(lpvReserved);

        switch(fdwReason)
        {
        case DLL_PROCESS_ATTACH:
                // プロセスがアタッチされると呼ばれる
                break;

        case DLL_PROCESS_DETACH:
                // プロセスがデタッチされると呼ばれる
                break;

        case DLL_THREAD_ATTACH:
                // スレッドがアタッチされると呼ばれる
                break;

        case DLL_THREAD_DETACH:
                // スレッドがデタッチされると呼ばれる
                // GetCurrentThread() によって遅延登録されたスレッドであれば、
                // ThreadType オブジェクトの後始末を行なう。
                {
                    ThreadType* thread = detail::GetCurrentThread();
                    if (thread->_autoRegistered)
                    {
                        CleanupDelayedRegisteredThread( thread );
                    }
                }
                break;

        default:
                break;
        }
    }

// Win32 で TLS callback を動かすためのおまじない
#if !defined(NN_OS_FOR_NONE_CLI)
    #ifdef _WIN64
        #pragma comment(linker, "/INCLUDE:_tls_used")
    #else
        #pragma comment(linker, "/INCLUDE:__tls_used")
    #endif

    // TLS callback が呼ばれるようにコールバック関数のアドレスを配置しておく
    #pragma section(".CRT$XLY",long,read)
    extern "C" __declspec(allocate(".CRT$XLY"))
        PIMAGE_TLS_CALLBACK _xl_y = ThreadTlsCallbackFunction;
#endif  // !defined(NN_OS_FOR_NONE_CLI)


    //-----------------------------------------------------------------------
    // スレッド名をデバッガに通知する。実装については以下を参照のこと。
    //  http://msdn.microsoft.com/ja-jp/library/xcb2z8hs.aspx
    //

    const DWORD MS_VC_EXCEPTION = 0x406D1388;

    #pragma pack(push,8)
    typedef struct tagTHREADNAME_INFO
    {
        DWORD   dwType;     // Must be 0x1000.
        LPCSTR  szName;     // Pointer to name (in user addr space).
        DWORD   dwThreadID; // Thread ID (-1=caller thread).
        DWORD   dwFlags;    // Reserved for future use, must be zero.
    } THREADNAME_INFO;
    #pragma pack(pop)

    void NotifyThreadName(DWORD dwThreadId, const char* threadName) NN_NOEXCEPT
    {
        if (IsDebuggerPresent() == 0)
        {
            // デバッガ非接続時はスレッド名通知の例外を投げない
            return;
        }

        // デバッガにスレッド名を通知する
        THREADNAME_INFO info;
        info.dwType     = 0x1000;
        info.szName     = threadName;
        info.dwThreadID = dwThreadId;
        info.dwFlags    = 0;

        __try
        {
            RaiseException(MS_VC_EXCEPTION, 0,
                           sizeof(info) / sizeof(ULONG_PTR), (ULONG_PTR*)&info);
        }
        __except(EXCEPTION_EXECUTE_HANDLER)
        {
        }
    }

}

//---------------------------------------------------------------------------
//  C++ 関数の定義
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
// ThreadManagerImplByWin32 のコンストラクタ
ThreadManagerImplByWin32::ThreadManagerImplByWin32(ThreadType* mainThread) NN_NOEXCEPT
{
    // CurrentThread オブジェクトへのポインタを格納しておく TLS を確保
    DWORD   tlsIndex = TlsAlloc();
    NN_SDK_ASSERT( tlsIndex != 0xffffffff, NN_TEXT_OS("内部エラー：Win32 TLS スロットを確保できません。"));
    m_currentThreadTlsIndex = tlsIndex;

    // 現在動作中のメインスレッドの ThreadType オブジェクトを初期化
    SetupThreadObjectUnsafe(mainThread, NULL, NULL, NULL, DefaultReservedStackSize, DefaultThreadPriority);

    // 現在動作中のメインスレッドの Win32 HANDLE を取得
    mainThread->_handle._win32Handle = ::GetCurrentThread();

    // アフィニティ初期状態を設定
    mainThread->_handle._idealCore    = ::GetCurrentProcessorNumber();
    mainThread->_handle._affinityMask = GetThreadAvailableCoreMask();
}


//---------------------------------------------------------------------------
// Thread 生成（coreNumber あり）
Result ThreadManagerImplByWin32::CreateThread(ThreadType* thread, void (*function)(ThreadType *), int coreNumber) NN_NOEXCEPT
{
    // スレッド生成用 引数リストを構築
    // スタックの先頭を一時的に使用。Win32 環境では実スタックは別途用意される
    Arglist*   args = reinterpret_cast<Arglist*>( thread->_originalStack );
    args->function  = function;

    // Win32 スレッドの生成（まだ実行開始しない）
    void* handle = reinterpret_cast<void*>( _beginthreadex(NULL,
                                            static_cast<unsigned>(thread->_stackSize),
                                            &StdcallInvokeThread,
                                            thread,
                                            CREATE_SUSPENDED, 0) );

    NN_SDK_ASSERT( handle != NULL, NN_TEXT_OS("nn::os::CreateThread(): Win32 スレッドの生成に失敗しました。"));

    // 生成したスレッドの Win32 HANDLE を設定
    thread->_handle._win32Handle      = handle;

    // アフィニティ初期状態を設定
    thread->_handle._idealCore    = coreNumber;
    thread->_handle._affinityMask = GetThreadAvailableCoreMask();

    return ResultSuccess();
}


//---------------------------------------------------------------------------
// Thread 開始
void ThreadManagerImplByWin32::StartThread(const ThreadType* thread) NN_NOEXCEPT
{
    // スレッドの実行開始（生成時に SUSPENDED 状態だったものを再開）
    // 再開前のサスペンドカウントは 1 でなければならない
    DWORD  count = ::ResumeThread( thread->_handle._win32Handle );
    NN_SDK_ASSERT( count == 1, NN_TEXT_OS("nn::os::StartThread(): Win32 スレッド開始時のサスペンドカウントが異常です（count=%d）。"), count);
    NN_UNUSED(count);
}


//---------------------------------------------------------------------------
// Thread 削除
void ThreadManagerImplByWin32::DestroyThreadUnsafe(ThreadType* thread) NN_NOEXCEPT
{
    // Win32 スレッドハンドルを閉じる
    HANDLE  handle = thread->_handle._win32Handle;
    BOOL ret = ::CloseHandle( handle );
    NN_SDK_ASSERT( ret != 0, NN_TEXT_OS("nn::os::DestroyThread(): Win32 スレッドの削除に失敗しました。") );
    NN_UNUSED(ret);

    thread->_handle._win32Handle  = 0;
}


//---------------------------------------------------------------------------
// Thread が終了するのを待機
void ThreadManagerImplByWin32::WaitForExitThread(ThreadType *thread) NN_NOEXCEPT
{
    // スレッドが本当に終了するのを待つ
    ::WaitForSingleObject( thread->_handle._win32Handle, INFINITE );
}


//---------------------------------------------------------------------------
// Thread の Yield
void ThreadManagerImplByWin32::YieldThread() NN_NOEXCEPT
{
    // 時間を 0 にして Sleep() を呼ぶことで再現する
    ::Sleep(0);
}


//---------------------------------------------------------------------------
// Thread の優先度変更
bool ThreadManagerImplByWin32::ChangePriority(ThreadType *thread, int priority) NN_NOEXCEPT
{
    // Win32 では何もしない
    NN_UNUSED( thread );
    NN_UNUSED( priority );
    return true;
}


//---------------------------------------------------------------------------
// Thread の現在優先度を返す
int ThreadManagerImplByWin32::GetCurrentPriority(const ThreadType *thread) const NN_NOEXCEPT
{
    // Win32 ではベース優先度をそのまま返す
    return thread->_basePriority;
}


//---------------------------------------------------------------------------
// Thread オブジェクトを TLS に設定する
void ThreadManagerImplByWin32::SetCurrentThread(const ThreadType* thread) const NN_NOEXCEPT
{
    TlsSetValue( m_currentThreadTlsIndex,
                    reinterpret_cast<LPVOID>(const_cast<ThreadType*>(thread)) );
}


//---------------------------------------------------------------------------
// Thread オブジェクトを TLS から取得して返す
ThreadType* ThreadManagerImplByWin32::GetCurrentThread() const NN_NOEXCEPT
{
    ThreadType* thread = reinterpret_cast<ThreadType*>( TlsGetValue(m_currentThreadTlsIndex) );

    // OS ライブラリが関与しないところで生成された Win32 スレッドの場合は、
    // NULL を受取ることになる（以下のケース）ため、遅延登録する。
    //
    //  - std::thread でスレッドが生成された場合
    //  - Win32 アプリが強制終了され、静的オブジェクトのデストラクタが動く場合
    //
    if (thread == NULL)
    {
        std::lock_guard<detail::InternalCriticalSection> lock(m_criticalSectionWin32);
        if (thread == NULL)
        {
            thread = DynamicAllocateAndRegisterThreadType();

            // カレントスレッドとして TLS に設定しておく
            this->SetCurrentThread(thread);
        }
        else
        {
            thread = reinterpret_cast<ThreadType*>( TlsGetValue(m_currentThreadTlsIndex) );
        }
    }

    return thread;
}

//---------------------------------------------------------------------------
// 現在の Thread 識別子を取得して返す
uint64_t ThreadManagerImplByWin32::GetThreadId(const ThreadType* thread) const NN_NOEXCEPT
{
    DWORD win32ThreadId = ::GetThreadId( thread->_handle._win32Handle );

    NN_SDK_ASSERT(win32ThreadId != 0, NN_TEXT_OS("内部エラー：Win32 スレッド ID の取得に失敗しました"));

    return  win32ThreadId;
}

//---------------------------------------------------------------------------
// スレッドの中断
void ThreadManagerImplByWin32::SuspendThreadUnsafe(ThreadType* thread) NN_NOEXCEPT
{
    auto prevCount = ::SuspendThread( thread->_handle._win32Handle );
    NN_SDK_ASSERT(prevCount == 0);
    NN_UNUSED(prevCount);
}

//---------------------------------------------------------------------------
// スレッドを再開
void ThreadManagerImplByWin32::ResumeThreadUnsafe(ThreadType* thread) NN_NOEXCEPT
{
    auto prevCount = ::ResumeThread( thread->_handle._win32Handle );
    NN_SDK_ASSERT(prevCount == 1);
    NN_UNUSED(prevCount);
}

//---------------------------------------------------------------------------
// スレッドのコンテキスト情報の取得
void ThreadManagerImplByWin32::GetThreadContextUnsafe(ThreadContextInfo* context, const ThreadType* thread) NN_NOEXCEPT
{
    context->ContextFlags = CONTEXT_FULL;
    BOOL ret = ::GetThreadContext( thread->_handle._win32Handle, context );
    NN_SDK_ASSERT(ret != 0);
    NN_UNUSED(ret);
}


//---------------------------------------------------------------------------
// スレッド名をデバッガ等に登録する
void ThreadManagerImplByWin32::NotifyThreadNameChangedImpl(const ThreadType* thread) const NN_NOEXCEPT
{
    DWORD win32ThreadId = ::GetThreadId( thread->_handle._win32Handle );
    NN_SDK_ASSERT(win32ThreadId != 0, NN_TEXT_OS("内部エラー：Win32 スレッド ID の取得に失敗しました"));

    if (thread == GetCurrentThread())
    {
        win32ThreadId = static_cast<DWORD>(-1);
    }
    NotifyThreadName( win32ThreadId, thread->_namePointer );
}

//---------------------------------------------------------------------------
// 現在動作中のプロセッサ番号の取得
int ThreadManagerImplByWin32::GetCurrentCoreNumber() const NN_NOEXCEPT
{
    return ::GetCurrentProcessorNumber();
}


//---------------------------------------------------------------------------
// コアアフィニティの設定
void ThreadManagerImplByWin32::SetThreadCoreMask(ThreadType* thread, int idealCore, Bit64 affinityMask) const NN_NOEXCEPT
{
    if (idealCore == nn::os::IdealCoreUseDefaultValue)
    {
        // win32 環境では、使用可能なコアのうち、もっとも値の小さいコアのみを使うように設定する
        affinityMask = GetThreadAvailableCoreMask();
        idealCore = nn::util::cntt0(affinityMask);
        affinityMask = 1ULL << idealCore;
    }

    // クリティカルセクション
    detail::InternalCriticalSection&    cs = Get(thread->_csThread);
    std::lock_guard<detail::InternalCriticalSection>    lock( cs );

    auto oldMask = ::SetThreadAffinityMask(thread->_handle._win32Handle, static_cast<DWORD_PTR>(affinityMask));
    NN_ABORT_UNLESS(oldMask != 0);

    // Win32 ではスレッドのアフィニティを取得する API がないため、覚えておく
    if (idealCore != nn::os::IdealCoreNoUpdate)
    {
        thread->_handle._idealCore    = idealCore;
    }
    thread->_handle._affinityMask = affinityMask;
}

//---------------------------------------------------------------------------
// コアアフィニティの取得
void ThreadManagerImplByWin32::GetThreadCoreMask(int* pOutIdealCore, Bit64* pOutAffinityMask, const ThreadType* thread) const NN_NOEXCEPT
{
    // クリティカルセクション
    detail::InternalCriticalSection&    cs = Get(thread->_csThread);
    std::lock_guard<detail::InternalCriticalSection>    lock( cs );

    if (pOutIdealCore)
    {
        *pOutIdealCore    = thread->_handle._idealCore;
    }
    if (pOutAffinityMask)
    {
        *pOutAffinityMask = thread->_handle._affinityMask;
    }
}

//---------------------------------------------------------------------------
// 割り当て可能なコアの取得
nn::Bit64 ThreadManagerImplByWin32::GetThreadAvailableCoreMask() const NN_NOEXCEPT
{
    DWORD_PTR processAffinityMask;
    DWORD_PTR systemAffinityMask;
    auto result = ::GetProcessAffinityMask(::GetCurrentProcess(), &processAffinityMask, &systemAffinityMask);
    NN_ABORT_UNLESS(result != 0);

    return static_cast<nn::Bit64>(processAffinityMask);
}

//---------------------------------------------------------------------------
// 自プロセスをクイック終了する。
// std::quick_exit() は VS2015 以降でサポートされている。
//
NN_NORETURN void ThreadManagerImplByWin32::QuickExit() NN_NOEXCEPT
{
#if defined(NN_BUILD_CONFIG_TOOLCHAIN_VC_VS2012) || defined(NN_BUILD_CONFIG_TOOLCHAIN_VC_VS2013)
    _exit(0);
#else
    std::quick_exit(0);
#endif
}

//---------------------------------------------------------------------------
// 自プロセスを終了する。
//
NN_NORETURN void ThreadManagerImplByWin32::ExitProcessImpl() NN_NOEXCEPT
{
    _exit(0);
}

//---------------------------------------------------------------------------

}  // namespace detail
}} // namespace nn::os


//  TORIAEZU: Win32 環境では自動的に呼ぶようにする
//  TORIAEZU: Win32 環境用のスタートアップが出来るまでの暫定対応
extern "C" void nnosInitialize();
NN_OS_DETAIL_ATTACH_EARLY_INITIALIZER( nnosInitialize );

