﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "os_Common.h"
#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/diag/text/diag_SdkTextOs.h>
#include <nn/nn_Windows.h>

#include <nn/nn_Result.h>
#include <nn/nn_SdkLog.h>
#include <nn/os/os_Result.h>
#include <nn/os/os_Types.h>
#include <nn/os/os_NativeHandleTypes.h>
#include <nn/os/os_SharedMemoryApi.h>
#include <nn/os/os_SharedMemoryTypes.h>

#include "os_Diag.h"
#include "os_SharedMemoryImpl.h"

namespace nn { namespace os {
namespace detail {

//--------------------------------------------------------------------------

namespace {

DWORD ConvertToWin32MemoryPermission(os::MemoryPermission permission) NN_NOEXCEPT
{
    switch (permission)
    {
    case os::MemoryPermission_ReadOnly:
            return PAGE_READONLY;

    case os::MemoryPermission_ReadWrite:
            return PAGE_READWRITE;

    default:
            break;
    }

    NN_ABORT(NN_TEXT_OS("nn::os::CreateSharedMemory(): 指定された permission=%d が不正です。"), permission);
}

}   // namespace

//--------------------------------------------------------------------------
//  共有メモリの作成（ハンドルの生成）
nn::Result SharedMemoryImplByWin32::Create(NativeHandle* outHandle, size_t size, os::MemoryPermission myPermission, os::MemoryPermission otherPermission) NN_NOEXCEPT
{
    auto winMyPermission    = ConvertToWin32MemoryPermission(myPermission);
    NN_UNUSED(otherPermission);

    DWORD sizeHigh = static_cast<uint64_t>(size) >> 32;
    DWORD sizeLow  = static_cast<DWORD>(size & 0xffffffff);
    HANDLE handle = CreateFileMappingA(INVALID_HANDLE_VALUE, NULL, winMyPermission, sizeHigh, sizeLow, NULL);

    if (handle == NULL)
    {
        return nn::os::ResultInvalidParameter();
    }

    *outHandle = reinterpret_cast<NativeHandle>( handle );
    return ResultSuccess();
}

//--------------------------------------------------------------------------
//  共有メモリの削除（ハンドルを削除）
void SharedMemoryImplByWin32::Close(NativeHandle handle) NN_NOEXCEPT
{
    HANDLE winHandle = reinterpret_cast<HANDLE>(handle);

    BOOL ret = CloseHandle(winHandle);
    if (ret == 0)
    {
        // TORIAEZU:
        // 単一プロセス内で複数回クローズが呼ばれた場合の警告。
        // API が Attach/Detach 体系になった時に修正する。
        NN_SDK_LOG(NN_TEXT_OS("nn::os::DestroySharedMemory(): 共有メモリハンドルは既にクローズされています。"));
    }
}

//--------------------------------------------------------------------------
//  共有メモリをマップ
Result SharedMemoryImplByWin32::Map(void** pOutAddress, NativeHandle handle, void* address, size_t size, os::MemoryPermission myPermission) NN_NOEXCEPT
{
    NN_UNUSED(address);

    HANDLE winHandle = reinterpret_cast<HANDLE>(handle);
    DWORD  access    = (myPermission == os::MemoryPermission_ReadOnly)
                       ? FILE_MAP_READ : FILE_MAP_WRITE;

    LPVOID mappedAddress = MapViewOfFileEx(winHandle, access, 0, 0, size, NULL);
    if (mappedAddress)
    {
        *pOutAddress = mappedAddress;
        return ResultSuccess();
    }

    NN_ABORT(NN_TEXT_OS("nn::os::MapSharedMemory(): 共有メモリのマップに失敗しました（err = 0x%08x）"), GetLastError());
}

//--------------------------------------------------------------------------
//  共有メモリをアンマップ
void SharedMemoryImplByWin32::Unmap(NativeHandle handle, void* address, size_t size) NN_NOEXCEPT
{
    NN_UNUSED(handle);
    NN_UNUSED(size);

    BOOL ret = UnmapViewOfFile(address);
    if (ret)
    {
        return;
    }

    NN_ABORT(NN_TEXT_OS("nn::os::UnmapSharedMemory(): 共有メモリのアンマップに失敗しました（err = 0x%08x）"), GetLastError());
}

//--------------------------------------------------------------------------

}  // namespace detail
}} // namespace nn::os

