﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>

#include <nn/nn_Macro.h>
#include <nn/nn_TimeSpan.h>
#include <nn/util/util_TypedStorage.h>
#include <nn/os/os_TimerEventTypes.h>

#include "os_Common.h"
#include "os_MultipleWaitHelper.h"
#include "os_MultipleWaitHolderOfTimerEvent.h"

//---------------------------------------------------------------------------
//  C++ 関数の定義
//---------------------------------------------------------------------------

namespace nn { namespace os {
namespace detail {

//--------------------------------------------------------------------------
// TimerEvent オブジェクトの状態をポーリング
TriBool MultiWaitHolderOfTimerEvent::IsSignaled() const NN_NOEXCEPT
{
    // クリティカルセクション
    std::lock_guard<InternalCriticalSection> lock( Get(m_TimerEvent->_csTimerEvent) );

    return this->IsSignaledUnsafe();
}


//--------------------------------------------------------------------------
// 多重待ちオブジェクトを、TimerEventType の多重待ちリストにリンク
TriBool MultiWaitHolderOfTimerEvent::AddToObjectList() NN_NOEXCEPT
{
    // クリティカルセクション
    std::lock_guard<InternalCriticalSection> lock( Get(m_TimerEvent->_csTimerEvent) );

    Get( m_TimerEvent->_multiWaitObjectList ).PushBackToList( *this );

    return this->IsSignaledUnsafe();
}


//--------------------------------------------------------------------------
// 多重待ちオブジェクトを、TimerEventType の多重待ちリストから外す
void MultiWaitHolderOfTimerEvent::RemoveFromObjectList() NN_NOEXCEPT
{
    // クリティカルセクション
    std::lock_guard<InternalCriticalSection> lock( Get(m_TimerEvent->_csTimerEvent) );

    Get( m_TimerEvent->_multiWaitObjectList ).EraseFromList( *this );
}

//--------------------------------------------------------------------------
// TimerEvent オブジェクトのタイマーによるシグナル化時間を返す
TimeSpan MultiWaitHolderOfTimerEvent::GetAbsoluteTimeToWakeup() const NN_NOEXCEPT
{
    // クリティカルセクション
    std::lock_guard<InternalCriticalSection> lock( Get(m_TimerEvent->_csTimerEvent) );

    if (m_TimerEvent->_timerState == TimerEventType::TimerState_Stop)
    {
        return TimeSpan::FromNanoSeconds(INT64_MAX);
    }

    return Get(m_TimerEvent->_nextTimeToWakeup);
}

//--------------------------------------------------------------------------

}   // namespace detail
}}  // namespace nn::os

