﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>

#include <nn/nn_Macro.h>
#include <nn/util/util_TypedStorage.h>
#include <nn/os/os_MessageQueueTypes.h>

#include "os_Common.h"
#include "os_MultipleWaitHelper.h"
#include "os_MultipleWaitHolderOfMessageQueue.h"

//---------------------------------------------------------------------------
//  C++ 関数の定義
//---------------------------------------------------------------------------

namespace nn { namespace os {
namespace detail {

//--------------------------------------------------------------------------
//  メッセージキュー（NotFull）用の多重待ちホルダーオブジェクト

// MessageQueue オブジェクトの状態をポーリング
TriBool MultiWaitHolderOfMessageQueueNotFull::IsSignaled() const NN_NOEXCEPT
{
    // クリティカルセクション
    std::lock_guard<InternalCriticalSection> lock( Get(m_MessageQueue->_csQueue) );

    return this->IsSignaledUnsafe();
}

// 多重待ちオブジェクトを、MessageQueueType の多重待ちリストにリンク
TriBool MultiWaitHolderOfMessageQueueNotFull::AddToObjectList() NN_NOEXCEPT
{
    // クリティカルセクション
    std::lock_guard<InternalCriticalSection> lock( Get(m_MessageQueue->_csQueue) );

    Get( m_MessageQueue->_multiWaitObjectListNotFull ).PushBackToList( *this );
    return this->IsSignaledUnsafe();
}

// 多重待ちオブジェクトを、MessageQueueType の多重待ちリストから外す
void MultiWaitHolderOfMessageQueueNotFull::RemoveFromObjectList() NN_NOEXCEPT
{
    // クリティカルセクション
    std::lock_guard<InternalCriticalSection> lock( Get(m_MessageQueue->_csQueue) );

    Get( m_MessageQueue->_multiWaitObjectListNotFull ).EraseFromList( *this );
}

//--------------------------------------------------------------------------
//  メッセージキュー（NotEmpty）用の多重待ちホルダーオブジェクト

// MessageQueue オブジェクトの状態をポーリング
TriBool MultiWaitHolderOfMessageQueueNotEmpty::IsSignaled() const NN_NOEXCEPT
{
    // クリティカルセクション
    std::lock_guard<InternalCriticalSection> lock( Get(m_MessageQueue->_csQueue) );

    return this->IsSignaledUnsafe();
}

// 多重待ちオブジェクトを、MessageQueueType の多重待ちリストにリンク
TriBool MultiWaitHolderOfMessageQueueNotEmpty::AddToObjectList() NN_NOEXCEPT
{
    // クリティカルセクション
    std::lock_guard<InternalCriticalSection> lock( Get(m_MessageQueue->_csQueue) );

    Get( m_MessageQueue->_multiWaitObjectListNotEmpty ).PushBackToList( *this );
    return this->IsSignaledUnsafe();
}

// 多重待ちオブジェクトを、MessageQueueType の多重待ちリストから外す
void MultiWaitHolderOfMessageQueueNotEmpty::RemoveFromObjectList() NN_NOEXCEPT
{
    // クリティカルセクション
    std::lock_guard<InternalCriticalSection> lock( Get(m_MessageQueue->_csQueue) );

    Get( m_MessageQueue->_multiWaitObjectListNotEmpty ).EraseFromList( *this );
}

}   // namespace detail
}}  // namespace nn::os

