﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Macro.h>
#include <nn/nn_TimeSpan.h>

#include <nn/util/util_IntrusiveList.h>
#include <nn/os/os_NativeHandleTypes.h>
#include <nn/os/os_ThreadTypes.h>
#include <nn/os/os_MultipleWaitTypes.h>
#include <nn/os/detail/os_InternalCriticalSection.h>
#include <nn/os/detail/os_InternalConditionVariable.h>

#include "os_MultipleWaitHolderBase.h"

#if defined(NN_BUILD_CONFIG_OS_WIN32)
    #include "os_MultipleWaitHelper-os.win32.h"
#elif defined(NN_BUILD_CONFIG_OS_HORIZON)
    #include "os_MultipleWaitHelper-os.horizon.h"
#else
    #error   "未サポートの OS 種別が指定されています。"
#endif


//--------------------------------------------------------------------------

namespace nn { namespace os {
namespace detail {

//--------------------------------------------------------------------------
//  MultiWaitType 構造体の実装用クラス
//

class MultiWaitImpl
{
    typedef nn::util::IntrusiveList<MultiWaitHolderBase, nn::util::IntrusiveListMemberNodeTraits<MultiWaitHolderBase, &MultiWaitHolderBase::m_MultiWaitNode>>    MultiWaitList;

public:
    static const int   WaitIsCanceled = -2;
    static const int   WaitIsTimedout = -1;
    static const int   WaitAnyArrayCountMax = MultiWaitImplByTarget::NativeHandleArrayCountMax;

private:
    MultiWaitHolderBase* InternalWaitAnyImpl(bool infinite, TimeSpan) NN_NOEXCEPT;
    MultiWaitHolderBase* WaitAnyImpl(bool infinite, TimeSpan) NN_NOEXCEPT;

public:
    // メンバ関数
    MultiWaitHolderBase* WaitAny() NN_NOEXCEPT
    {
        return WaitAnyImpl( true, nn::TimeSpan::FromNanoSeconds(INT64_MAX) );
    }

    MultiWaitHolderBase* TryWaitAny() NN_NOEXCEPT
    {
        return WaitAnyImpl( false, nn::TimeSpan(0) );
    }

    MultiWaitHolderBase* TimedWaitAny(TimeSpan timeSpan) NN_NOEXCEPT
    {
        return WaitAnyImpl( false, timeSpan );
    }

    MultiWaitHolderBase* RecalcMultiWaitTimeout(TimeSpan* timeoutMin, TimeSpan endTime) NN_NOEXCEPT;

    int  ConstructObjectsArray(NativeHandle array[], MultiWaitHolderBase* arrayToHolder[], int num) NN_NOEXCEPT;

    void NotifyAndWakeupThread( MultiWaitHolderBase* holder) NN_NOEXCEPT;

    bool IsListEmpty() const NN_NOEXCEPT
    {
        return m_MultiWaitList.empty();
    }
    bool IsListNotEmpty() const NN_NOEXCEPT
    {
        return !m_MultiWaitList.empty();
    }
    void PushBackToList( MultiWaitHolderBase& holderBase ) NN_NOEXCEPT
    {
        m_MultiWaitList.push_back( holderBase );
    }
    void EraseFromList( MultiWaitHolderBase& holderBase ) NN_NOEXCEPT
    {
        m_MultiWaitList.erase( m_MultiWaitList.iterator_to( holderBase ) );
    }
    void EraseAllFromList() NN_NOEXCEPT
    {
        while (!m_MultiWaitList.empty())
        {
            m_MultiWaitList.front().m_pMultiWait = NULL;
            m_MultiWaitList.pop_front();
        }
    }
    void MoveAllFromOther(MultiWaitImpl* pOther) NN_NOEXCEPT
    {
        for (auto& p : pOther->m_MultiWaitList)
        {
            p.SetMultiWait(this);
        }
        m_MultiWaitList.splice(m_MultiWaitList.end(), pOther->m_MultiWaitList);
    }
    TimeSpan GetCurrTime() const NN_NOEXCEPT
    {
        return m_CurrTime;
    }

private:
    MultiWaitHolderBase* AddToEachObjectListAndCheckObjectState() NN_NOEXCEPT;
    void                 RemoveFromEachObjectList()               NN_NOEXCEPT;

private:
    // メンバ変数
    MultiWaitList           m_MultiWaitList;    // 多重待ちオブジェクトのリスト
    MultiWaitHolderBase*    m_SignaledHolder;   // シグナル化されたホルダー
    TimeSpan                m_CurrTime;         // 現在チックの TimeSpan 値

    // 排他制御用
    InternalCriticalSection     m_csMultiWait;
    MultiWaitImplByTarget       m_Impl;
};


//--------------------------------------------------------------------------
//  ThreadType, SemaphoreType, EventType, MessageQueueType などで使用。
//  待機中の多重待ちホルダーをリストするためのリストヘッダクラス。
//
typedef nn::util::IntrusiveList<MultiWaitHolderBase, nn::util::IntrusiveListMemberNodeTraits<MultiWaitHolderBase, &MultiWaitHolderBase::m_WaitObjectNode>>  WaitObjectList;

class MultiWaitObjectList
{
public:
    //-----------------------------------------------------------------------
    //  本オブジェクトを対象に多重待ちを行なっている全ての多重待ちスレッドに
    //  対して、条件変数シグナルを通知しスレッドを起床させる。
    //  このメンバ関数は、対象の同期オブジェクトのクリティカルセクションが
    //  ロックされた状態で呼び出さなければならない。
    void WakeupAllMultiWaitThreadsUnsafe() NN_NOEXCEPT
    {
        for ( MultiWaitHolderBase& holder : m_WaitObjectList )
        {
            holder.m_pMultiWait->NotifyAndWakeupThread( &holder );
        }
    }

    //-----------------------------------------------------------------------
    //  本オブジェクトを対象に多重待ちを行なっている全ての多重待ちスレッドに
    //  対して、条件変数シグナルを通知しスレッドを起床させる。
    //  このメンバ関数は、対象の同期オブジェクトのクリティカルセクションが
    //  ロックされた状態で呼び出さなければならない。
    //
    //  起床されたスレッドは多重待ちのタイムアウト時間を再計算する必要がある。
    void BroadcastToUpdateObjectStateUnsafe() NN_NOEXCEPT
    {
        for ( MultiWaitHolderBase& holder : m_WaitObjectList )
        {
            // オブジェクトがシグナル化されたわけではないので NULL を渡す
            holder.m_pMultiWait->NotifyAndWakeupThread(NULL);
        }
    }

    void PushBackToList( MultiWaitHolderBase& holderBase ) NN_NOEXCEPT
    {
        m_WaitObjectList.push_back( holderBase );
    }

    void EraseFromList( MultiWaitHolderBase& holderBase ) NN_NOEXCEPT
    {
        m_WaitObjectList.erase( m_WaitObjectList.iterator_to( holderBase ) );
    }

    bool IsEmpty() const NN_NOEXCEPT
    {
        return m_WaitObjectList.empty();
    }

private:
    WaitObjectList  m_WaitObjectList;
};


}   // namespace detail
}}  // namespace nn::os

