﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>
#include <algorithm>

#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/diag/text/diag_SdkTextOs.h>
#include <nn/nn_SdkLog.h>
#include <nn/os/os_NativeHandleTypes.h>

#include "os_Common.h"
#include "os_Diag.h"
#include "os_ThreadManager.h"
#include "os_TickManager.h"
#include "os_TimeoutHelper.h"
#include "os_MultipleWaitHelper.h"
#include "os_MultipleWaitHolderBase.h"

//---------------------------------------------------------------------------
//  C++ 関数の定義
//---------------------------------------------------------------------------

namespace nn { namespace os {
namespace detail {

//----------------------------------------------------------------------------
// MultiWait リストの各 Holder のうち、最も近い将来のタイマーシグナル化時間
// に該当する TimerEvent オブジェクトを特定し、そのタイムアウト時間を算出する。
// 返値には対象となる TimerEvent の MultiWaitHolderBase* を返すが、
// 第２引数の endTime がリストの中で最小時間だった場合には NULL を返す。
MultiWaitHolderBase* MultiWaitImpl::RecalcMultiWaitTimeout(TimeSpan* timeoutMin, TimeSpan endTime) NN_NOEXCEPT
{
    MultiWaitHolderBase* holderOfTimeoutMin = NULL;
    TimeSpan             endTimeMin         = endTime;

    // MultiWaitHolder リストの中から最も近い将来の TimerEvent を抽出
    for ( MultiWaitHolderBase& holder : m_MultiWaitList )
    {
        TimeSpan wakeupTime = holder.GetAbsoluteTimeToWakeup();

        if (wakeupTime < endTimeMin)
        {
            endTimeMin          = wakeupTime;
            holderOfTimeoutMin  = &holder;
        }
    }

    // 最小値を返す
    *timeoutMin = (endTimeMin < m_CurrTime) ? 0 : (endTimeMin - m_CurrTime);
    return holderOfTimeoutMin;
}


//------------------------------------------------------------------------
// MultiWait リストの各 Holder のうち、カーネルオブジェクトを対象とした
// ものを抽出し、指定された array に詰めていく。
// 最終的に、抽出したカーネルオブジェクトの総数を返す。
int MultiWaitImpl::ConstructObjectsArray(NativeHandle array[], MultiWaitHolderBase* arrayToHolder[], int num) NN_NOEXCEPT
{
    int count = 0;

    // MultiWaitHolderType の中にカーネルオブジェクトがあれば配列に設定
    for ( MultiWaitHolderBase& holder : m_MultiWaitList )
    {
        NativeHandle    handle;

        if ( holder.GetNativeHandle( &handle ) )
        {
            NN_SDK_ASSERT( count < num, NN_TEXT_OS("多重待ち対象のカーネルオブジェクトが多すぎます。"));
            NN_UNUSED(num);

            array[ count ]         = handle;
            arrayToHolder[ count ] = &holder;
            ++count;
        }
    }

    return count;
}

//------------------------------------------------------------------------
// MultiWaitType リストの各 Holder を対象同期オブジェクトにリンクする。
// 内部で各 Holder が紐付いている対象オブジェクトのロックを取得し、
// 既にシグナル状態のオブジェクトがあれば、その Holder を返す。
// なお、multiWait のクリティカルセクションを事前にロックする必要はない。
// （multiWait リスト操作は自スレッド以外では行なわないため）
MultiWaitHolderBase* MultiWaitImpl::AddToEachObjectListAndCheckObjectState() NN_NOEXCEPT
{
    MultiWaitHolderBase*    signaledHolder = NULL;

    for ( MultiWaitHolderBase& holder : m_MultiWaitList )
    {
        // ホルダーを同期オブジェクトの多重待ちリストに繋ぐ
        auto isSignaled = holder.AddToObjectList();

        // シグナル状態のオブジェクトが見つかれば登録しておく
        if ((signaledHolder == NULL) && (isSignaled == TriBool::True))
        {
            signaledHolder = &holder;
        }
    }

    // 全てのホルダーをリストに繋いでから結果を返す
    return signaledHolder;
}

//-----------------------------------------------------------------------
// 全ての multiWaitHolder を、multiWait 多重待ちリストから外す。
// 内部で各 Holder が紐付いている対象オブジェクトのロックを取得します。
// なお、multiWait のクリティカルセクションを事前にロックする必要はない。
// （multiWait リスト操作は自スレッド以外では行なわないため）
void MultiWaitImpl::RemoveFromEachObjectList() NN_NOEXCEPT
{
    for ( MultiWaitHolderBase& holder : m_MultiWaitList )
    {
        // ホルダーを同期オブジェクトの多重待ちリストから外す
        holder.RemoveFromObjectList();
    }
}

//-----------------------------------------------------------------------
// WaitAny(), TryWaitAny(), TimedWaitAny() の共通実装
//
MultiWaitHolderBase* MultiWaitImpl::InternalWaitAnyImpl(bool infinite, TimeSpan timeout) NN_NOEXCEPT
{
    NativeHandle         objectsArray[ WaitAnyArrayCountMax ];
    MultiWaitHolderBase* objectsArrayToHolder[ WaitAnyArrayCountMax ];

    // カーネルオブジェクトを抽出して配列にセットする
    int objectCount = ConstructObjectsArray( objectsArray,
                                             objectsArrayToHolder,
                                             WaitAnyArrayCountMax );

    // タイムアウトが成立する絶対時間
    TimeSpan absoluteEndTime = infinite
                               ? TimeSpan::FromNanoSeconds(INT64_MAX)
                               : GetCurrentTick().ToTimeSpan() + timeout;

    // TimerEvent と spurious に対応するためのループ
    // 多重待ちリストの中に TimerEvent がある場合、TimerEvent が再キックされて
    // シグナル化時間が変化することがあるため、毎ループごとに TimerEvent の中
    // から最小の待ち時間を抽出し直す必要がある。
    for (;;)
    {
        // 現在の時間を格納。後々 RecalcMultiWaitTimeout の中で参照される。
        m_CurrTime = GetCurrentTick().ToTimeSpan();

        // 多重待ちリスト内の各種 TimerEvent と timeout のタイムアウト値 で、
        // 最も近い未来を指す時間までの timeoutMin と timeoutMinHolder を取得。
        TimeSpan timeoutMin;
        auto* timeoutMinHolder = RecalcMultiWaitTimeout( &timeoutMin,
                                                         absoluteEndTime );

        // いずれかのオブジェクトから通知が来るまで待機
#if 1
        // objectCount ==0 かつ timeoutMin == 0 の場合に
        // svc::WaitSynchronization() の呼出しを行なわない実装。
        int index;
        if (infinite && (timeoutMinHolder == NULL))
        {
            index = m_Impl.WaitAny(objectsArray, WaitAnyArrayCountMax, objectCount);
        }
        else
        {
            if (objectCount == 0 && timeoutMin == 0)
            {
                index = WaitIsTimedout;
            }
            else
            {
                index = m_Impl.TimedWaitAny(objectsArray, WaitAnyArrayCountMax, objectCount, timeoutMin);
            }
        }
#else
        // 元の実装。念のため残しておく。
        int index = (infinite && (timeoutMinHolder == NULL))
                    ? m_Impl.WaitAny(objectsArray, WaitAnyArrayCountMax, objectCount)
                    : m_Impl.TimedWaitAny(objectsArray, WaitAnyArrayCountMax, objectCount, timeoutMin);
#endif

        switch (index)
        {
        case WaitIsTimedout:
                // WaitIsTimedout になるケースは以下の２つのケースがある。
                // 1) TimedWaitAny のタイムアウトが成立
                // 2) いずれかの TimerEvent がタイマー動作によってシグナル化
                //
                // 上記の判別は、timeoutMinHolder で特定可能で、
                // これが NULL だった場合は 1) で、非 NULL なら 2) である。
                if (timeoutMinHolder)
                {
                    m_CurrTime = GetCurrentTick().ToTimeSpan();
                    if (timeoutMinHolder->IsSignaled() == TriBool::True)
                    {   // クリティカルセクション
                        std::lock_guard<InternalCriticalSection> lock(m_csMultiWait);

                        m_SignaledHolder = timeoutMinHolder;
                        return timeoutMinHolder;
                    }

                    // Windows7 環境の場合、タイムアウト分解能が 1msec と粗く、
                    // かつ、タイムアウト指定時間より早く返ってくることがあり、
                    // ここに到達してしまう可能性がある（spurious の一種？）。
                    // Tick ベースで判定している正しいタイムアウト時刻に
                    // まだ到達していないため、もう一度ループを回す。
                    continue;
                }
                return NULL;

        case WaitIsCanceled:
                // WaitIsCanceld は以下の２つのケースがある。
                // 1) NotifyAndWakeupThread() により明示的に起床された。
                // 2) 多重待ちに入る前から以前の WaitCancel 要求が残っていた。
                //
                // 1) の場合、m_SignaledHolder に対象 holder が格納されているが
                // m_SignaledHolder == NULL だった場合は、TimerEvent の
                // StartOneShot/StartPeriodic/Stop 等により、次のタイマー成立
                // 時間に変更があり、一時的に WaitCancel された場合である。
                // その場合にはタイムアウト時間を再計算して再び多重待ちへ。
                //
                // 2) の場合は NULL が格納されたままになっている。
                // これは一種の spurious であり、再び多重待ちに入る必要がある。
                if (m_SignaledHolder)
                {
                    return m_SignaledHolder;
                }
                continue;

        default:
                NN_SDK_ASSERT( (index >= 0) && (index < objectCount), NN_TEXT_OS("内部エラー：カーネルの多重待ちで異常を検出しました。"));

                {   // クリティカルセクション
                    std::lock_guard<InternalCriticalSection> lock(m_csMultiWait);

                    // 待ちが解除されてから、ここに到達するまでの間に、
                    // NotifyAndWakeupThread() が呼ばれて m_SignaledHolder が
                    // 先にセットされる可能性がある。しかし、実際にスレッドが
                    // 起床した要因は本件の index であるため上書きして返す。
                    // もし、NotifyAndWakeupThread() によって起床されていた
                    // 場合は、WaitIsCanceled を受取っているはずである。
                    m_SignaledHolder = objectsArrayToHolder[ index ];
                    return m_SignaledHolder;
                }
        }
    }
}

MultiWaitHolderBase* MultiWaitImpl::WaitAnyImpl(bool infinite, TimeSpan timeout) NN_NOEXCEPT
{
    // シグナル状態にある Holder がないことを示す。
    // 最初にシグナル化されると同時にその holder をセットする。
    // それ以降の上書き（つまり、非 NULL 時の上書き）は一部の例外を除き
    // 禁止とする（詳細は、InternalWaitAnyImpl 関数内のコメントを参照）。
    m_SignaledHolder = NULL;

    // WaitAny 待機中のカレントスレッドハンドルを設定しておく
    m_Impl.SetCurrentThreadHandleForCancelWait();

    // MultiWaitType リストの各 Holder を待ち対象の各同期オブジェクトにリンク。
    // MultiWaitType のロックは不要（MultiWaitType リスト操作は自スレッドのみのため）
    MultiWaitHolderBase* holder = AddToEachObjectListAndCheckObjectState();

    // MultiWaitImpl::NotifyAndWakeupThread() との排他
    {
        std::lock_guard<InternalCriticalSection> lock(m_csMultiWait);
        holder = m_SignaledHolder ? m_SignaledHolder : holder;
    }

    if (holder == NULL)
    {
        // 実際に待ちに入る
        holder = InternalWaitAnyImpl(infinite, timeout);
    }

    // 各 multiWaitHolder を待ち対象だった各同期オブジェクトからアンリンク
    // MultiWaitType のロックは不要（MultiWaitType リスト操作は自スレッドのみのため）
    RemoveFromEachObjectList();

    // WaitAny 待機中のカレントスレッドハンドルをクリアしておく
    m_Impl.ClearCurrentThreadHandleForCancelWait();

    // シグナル化された holder を返す（タイムアウトの場合は NULL）
    return holder;
}

void MultiWaitImpl::NotifyAndWakeupThread(MultiWaitHolderBase* holder) NN_NOEXCEPT
{
    // クリティカルセクション
    std::lock_guard<InternalCriticalSection> lock( m_csMultiWait );

    // シグナル化した Holder 情報を記録（既に非 NULL なら上書きはしない）
    if (m_SignaledHolder == NULL)
    {
        m_SignaledHolder = holder;

        // 多重待ち中のスレッドを起床
        // m_SignaledHolder が NULL だった場合のみ CS 区間中で実施する。
        // 多重待ちスレッドには、WaitIsCanceled が返る。
        m_Impl.CancelWait();
    }
}


}   // namespace detail
}}  // namespace nn::os

