﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "os_Common.h"
#include <nn/os/os_Config.h>
#include <nn/diag/text/diag_SdkTextOs.h>

#include <nn/nn_Windows.h>

#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_TimeSpan.h>
#include <nn/os/os_NativeHandleTypes.h>
#include "os_Diag.h"
#include "os_MultipleWaitHelper.h"
#include "os_MultipleWaitHolderBase.h"


namespace nn { namespace os {
namespace detail {

//---------------------------------------------------------------------------
//  C++ 関数の定義
//---------------------------------------------------------------------------

int MultiWaitImplByWin32::WaitForAnyObjects(int num, NativeHandle array[], int arraySize, DWORD dwMilliSeconds) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( num < arraySize, NN_TEXT_OS("内部エラー：多重待ち対象のオブジェクト配列が小さすぎます。"));
    NN_UNUSED(arraySize);

    array[ num ] = m_WakeupEvent;

    // Win32 の多重待ちを発行
    DWORD result = ::WaitForMultipleObjects(num + 1, array, FALSE, dwMilliSeconds);
    if (result == WAIT_TIMEOUT)
    {
        return MultiWaitImpl::WaitIsTimedout;
    }

    int index = result - WAIT_OBJECT_0;
    NN_SDK_ASSERT( ((index >= 0) && (index <= num)), NN_TEXT_OS("内部エラー：Win32 多重待ちで予期しないエラーが発生しました。"));

    if (index == num)
    {
        return MultiWaitImpl::WaitIsCanceled;
    }

    return index;
}

//--------------------------------------------------------------------------
// コンストラクタ
MultiWaitImplByWin32::MultiWaitImplByWin32() NN_NOEXCEPT
{
    // 自動クリアタイプのイベントを生成（スレッド起床用）
    HANDLE handle = ::CreateEvent( NULL, FALSE, FALSE, NULL );
    NN_SDK_ASSERT( handle != NULL, NN_TEXT_OS("内部エラー：多重待ち用の Win32 イベント生成に失敗しました。"));

    m_WakeupEvent = handle;
}

//--------------------------------------------------------------------------
// デストラクタ
MultiWaitImplByWin32:: ~MultiWaitImplByWin32() NN_NOEXCEPT
{
    ::CloseHandle( m_WakeupEvent );
}

//--------------------------------------------------------------------------
// Wait を Cancel し、スレッドを起床させる
void MultiWaitImplByWin32::CancelWait() NN_NOEXCEPT
{
    ::SetEvent( m_WakeupEvent );
}

//--------------------------------------------------------------------------
int MultiWaitImplByWin32::TimedWaitAny(NativeHandle array[], int arraySize, int num, TimeSpan timeSpan) NN_NOEXCEPT
{
    detail::TimeoutHelper   tmout( timeSpan );

    // １回は WaitForAnyObjects() を呼ぶ必要があるので while() にはしない。
    do
    {
        int index = WaitForAnyObjects( num, array, arraySize,
                                       tmout.GetLeftTimeOnTarget() );

        // Spurious 対応のため、Win32 タイムアウトの時は別途判定する
        if (index != MultiWaitImpl::WaitIsTimedout)
        {
            return index;
        }

    } while( !tmout.IsTimedout() );

    return MultiWaitImpl::WaitIsTimedout;
}

}  // namespace detail
}} // namespace nn::os

