﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "os_Common.h"
#include <nn/os/os_Config.h>
#include <nn/diag/text/diag_SdkTextOs.h>

#if !NN_BUILD_CONFIG_OS_SUPPORTS_WIN32
    #error "OS 種別として Win32 が指定されていません。"
#endif

#include <nn/nn_Common.h>
#include <nn/nn_Windows.h>

#include <nn/nn_SdkAssert.h>
#include <nn/os/os_MemoryPermission.h>
#include "os_Diag.h"
#include "os_MemoryPermissionImpl.h"

#include <algorithm>

//---------------------------------------------------------------------------
// const 定数同士の比較に対する Warnng C4127 の抑制
#pragma warning( disable : 4127 )

//--------------------------------------------------------------------------

namespace nn { namespace os {
namespace detail {


//-----------------------------------------------------------------------------
// メモリアクセス権の変更
void SetMemoryPermissionImpl(uintptr_t address, size_t size, MemoryPermission permission) NN_NOEXCEPT
{
    DWORD   flOldProtect;

    // VirtualProtect が DWORD の値域でしか使えないため、
    // size が 2GB を超える場合に備えてループで対応する
    uintptr_t ptr = address;
    size_t    remainingSize = size;
    while (remainingSize > 0)
    {
        size_t tmpSize = std::min<size_t>(remainingSize, 2U * 1024 * 1024 * 1024);
        switch (permission)
        {
        case MemoryPermission_None:
            {
                // Win32: メモリ属性を None に変更
                BOOL result = VirtualProtect( reinterpret_cast<LPVOID>( ptr ),
                                              static_cast<DWORD>( tmpSize ),
                                              PAGE_NOACCESS,
                                              &flOldProtect );

                NN_ABORT_UNLESS( result != 0, NN_TEXT_OS("nn::os::SetMemoryPermission(): アクセス権を None に設定できません。"));
                NN_UNUSED(result);
                break;
            }

        case MemoryPermission_ReadOnly:
            {
                // Win32: メモリ属性を ReadOnly に変更
                BOOL result = VirtualProtect( reinterpret_cast<LPVOID>( ptr ),
                                              static_cast<DWORD>( tmpSize ),
                                              PAGE_READONLY,
                                              &flOldProtect );

                NN_ABORT_UNLESS( result != 0, NN_TEXT_OS("nn::os::SetMemoryPermission(): アクセス権を ReadOnly に設定できません。"));
                NN_UNUSED(result);
                break;
            }

        case MemoryPermission_ReadWrite:
            {
                // Win32: メモリ属性を ReadWrite に変更
                BOOL result = VirtualProtect( reinterpret_cast<LPVOID>( ptr ),
                                              static_cast<DWORD>( tmpSize ),
                                              PAGE_READWRITE,
                                              &flOldProtect );
                NN_ABORT_UNLESS( result != 0, NN_TEXT_OS("nn::os::SetMemoryPermission(): アクセス権を ReadWrite に設定できません。"));
                NN_UNUSED(result);
                break;
            }

        default:
            {
                NN_ABORT(NN_TEXT_OS("nn::os::SetMemoryPermission(): 不正な permission=0x%x が指定されました。\n"), permission);
                break;
            }
        }
        ptr += tmpSize;
        remainingSize -= tmpSize;
    }
}


}   // namespace detail
}}  // namespace nn::os

//  TORIAEZU: Win32 環境では自動的に呼ぶようにする
//  TORIAEZU: Win32 環境用のスタートアップが出来るまでの暫定対応
extern "C" void nnosInitialize(void);
NN_OS_DETAIL_ATTACH_EARLY_INITIALIZER( nnosInitialize );

