﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <cstring>
#include <nn/os/os_Config.h>

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_Result.h>
#include <nn/util/util_IntrusiveList.h>
#include <nn/os/os_MemoryHeapCommon.h>
#include <nn/os/detail/os_InternalCriticalSection.h>

#if defined(NN_BUILD_CONFIG_OS_WIN32)
    #include "os_MemoryHeapManager-os.win32.h"
#elif defined(NN_BUILD_CONFIG_OS_HORIZON)
    #include "os_MemoryHeapManager-os.horizon.h"
#else
    #error   "未サポートの OS 種別が指定されています。"
#endif


namespace nn { namespace os {
namespace detail {

class   MemoryHeapManager;

//-------------------------------------------------------------
// 空きメモリ領域を管理するためのメモリブロック
//
class   FreeMemoryNode
{
    friend class    MemoryHeapManager;

public:
    // 空きメモリブロックの先頭アドレスを取得
    uintptr_t   GetAddress()        NN_NOEXCEPT
    {
        return reinterpret_cast<uintptr_t>( this );
    }

    // 空きメモリブロックのサイズを取得
    size_t  GetSize()               NN_NOEXCEPT
    {
        return m_size;
    }

    // 空きメモリブロックのサイズを設定
    void    SetSize(size_t size)    NN_NOEXCEPT
    {
        m_size = size;
    }

    void    Clean()    NN_NOEXCEPT
    {
        std::memset( reinterpret_cast<void*>(this), 0, sizeof(FreeMemoryNode) );
    }

private:
    nn::util::IntrusiveListNode m_node; // 空きメモリノード
    size_t                      m_size; // このノードのメモリブロックのサイズ
};


//--------------------------------------------------------------------------
//  MemoryHeapManager クラス
//
class MemoryHeapManager
{
    typedef nn::util::IntrusiveList<FreeMemoryNode, nn::util::IntrusiveListMemberNodeTraits<FreeMemoryNode, &FreeMemoryNode::m_node>>  FreeMemoryList;

    NN_DISALLOW_COPY( MemoryHeapManager );
    NN_DISALLOW_MOVE( MemoryHeapManager );

public:
    // コンストラクタ
    MemoryHeapManager()     NN_NOEXCEPT : m_heapAddress( 0 ),
                                          m_heapSize( 0 ),
                                          m_usedHeapSize( 0 ) {}

    //-------------------------------------------------------------
    // ヒープサイズの変更
    nn::Result  SetHeapSize(size_t size)    NN_NOEXCEPT;

    //-------------------------------------------------------------
    // ヒープアドレスの取得
    uintptr_t   GetHeapAddress() const NN_NOEXCEPT
    {
        return m_heapAddress;
    }

    //-------------------------------------------------------------
    // ヒープサイズの取得
    size_t      GetHeapSize()    const NN_NOEXCEPT
    {
        return m_heapSize;
    }

    //-------------------------------------------------------------
    // メモリヒープの予約済み空間サイズの取得
    size_t      GetReservedSize()    const NN_NOEXCEPT
    {
        return m_impl.GetReservedSize();
    }

    //-------------------------------------------------------------
    //  メモリヒープからメモリを獲得
    nn::Result  AllocateFromHeap(uintptr_t* address, size_t size) NN_NOEXCEPT;

    //-------------------------------------------------------------
    //  メモリヒープへメモリを返却
    void        ReleaseToHeap(uintptr_t address, size_t size) NN_NOEXCEPT;

    //-------------------------------------------------------------
    // 指定されたメモリ領域が、ヒープ内かどうか
    bool    IsRegionInMemoryHeap(uintptr_t address, size_t size) NN_NOEXCEPT
    {
        return ((address >= m_heapAddress) &&
                ((address + size ) <= (m_heapAddress + m_heapSize)));
    }

    //-------------------------------------------------------------
    // 現在のヒープ使用量を返す
    size_t  GetUsedHeapSize()   const NN_NOEXCEPT
    {
        return m_usedHeapSize;
    }

private:
    //-------------------------------------------------------------
    // ヒープの空きメモリから、利用可能な場所を探索
    FreeMemoryList::iterator  FindFreeSpaceUnsafe(size_t size) NN_NOEXCEPT;

    //-------------------------------------------------------------
    // 直前の空きメモリノードと結合を試みる
    FreeMemoryList::iterator    ConcatenatePreviousFreeMemoryNodeUnsafe( FreeMemoryList::iterator node ) NN_NOEXCEPT;

    //-------------------------------------------------------------
    // 空きメモリ領域を２つに分割し、後半を空きメモリノードとして登録
    void SplitFreeMemoryNodeUnsafe(FreeMemoryList::iterator itr, size_t size) NN_NOEXCEPT;

    //-------------------------------------------------------------
    // ヒープに空きメモリ領域を追加し、空きメモリリストを更新
    void AddToFreeSpaceUnsafe(uintptr_t address, size_t size) NN_NOEXCEPT;

    //-------------------------------------------------------------
    // 指定されたメモリ領域が、ヒープ内の獲得済み領域かどうかを検査
    bool IsRegionAllocatedMemoryUnsafe(uintptr_t address, size_t size) NN_NOEXCEPT;

private:
    uintptr_t               m_heapAddress;      // ヒープ先頭アドレス
    size_t                  m_heapSize;         // ヒープサイズ

    size_t                  m_usedHeapSize;     // ヒープの使用量
    FreeMemoryList          m_freeMemoryList;   // FreeMemoryNode のリスト
    InternalCriticalSection m_criticalSection;
    MemoryHeapManagerImpl   m_impl;
};

//---------------------------------------------------------------------------

}   // namespace detail
}}  // namespace nn::os

