﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "os_Common.h"
#include <new>
#include <mutex>

#include <nn/nn_Macro.h>
#include <nn/diag/text/diag_SdkTextOs.h>
#include <nn/nn_SdkAssert.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/os/os_Result.h>
#include <nn/os/os_MemoryPermission.h>
#include <nn/os/os_MemoryAttribute.h>

#include "os_Diag.h"
#include "os_MemoryHeapManager.h"


//-----------------------------------------------------------------------------

namespace nn { namespace os {

namespace detail {

//-----------------------------------------------------------------------------
// 指定されたノードと直前の FreeMemoryNode ノードが結合できるなら結合する。
// 結合した場合、結合した新しい FreeMemoryNode を指すイテレータを返す。
// 結合できなかった場合、引数に指定されたイテレータをそのまま返す。
// 予め、MemoryHeapManager のロックを取得しておく必要がある。
MemoryHeapManager::FreeMemoryList::iterator    MemoryHeapManager::ConcatenatePreviousFreeMemoryNodeUnsafe( FreeMemoryList::iterator node ) NN_NOEXCEPT
{
    FreeMemoryList::iterator prev = node;
    --prev;

    // ノードがリストヘッダなら node をそのまま返す。
    if ( (prev == m_freeMemoryList.end()) ||
         (node == m_freeMemoryList.end()) )
    {
        return node;
    }

    // 直前ノードと自ノードが隣接していなければ連結しない
    if ( (prev->GetAddress() + prev->GetSize()) != node->GetAddress() )
    {
        return node;
    }

    // 直前ノードと自ノードのサイズを統合し、自ノードを消滅
    prev->SetSize( prev->GetSize() + node->GetSize() );
    m_freeMemoryList.erase( node );
    node->Clean();

    // 結合前の直前ノード（＝新しいノード）のイテレータを返す
    return prev;
}

//-----------------------------------------------------------------------------
//  ヒープの空き領域から利用可能な場所を探す（リストに繋いだまま返す）。
//  予め、MemoryHeapManager のロックを取得しておく必要がある。
MemoryHeapManager::FreeMemoryList::iterator  MemoryHeapManager::FindFreeSpaceUnsafe(size_t size) NN_NOEXCEPT
{
    FreeMemoryList::iterator    candidate = m_freeMemoryList.end();

    // 空きメモリ領域の全ノードを探索
    for (FreeMemoryList::iterator itr =  m_freeMemoryList.begin();
                                  itr != m_freeMemoryList.end();
                                  itr++ )
    {
        // 空きメモリノードが、要求された size 以上なら候補にする
        size_t nodeSize = itr->GetSize();
        if (nodeSize >= size)
        {
            // 初候補なら candidate に設定、もしくは、
            // 以前の候補より更に小さいメモリ領域であれば新しい候補とする
            if ((candidate == m_freeMemoryList.end()) ||
                (nodeSize < candidate->GetSize()))
            {
                candidate = itr;
            }
        }
    }

    // 最終候補のノードを返す
    return candidate;
}


//-----------------------------------------------------------------------------
//  空きメモリノードを２つに分割する。
//  前半部分は指定された size に縮められ、空きメモリリストに接続したまま残る。
//  後半部分のサイズは自動計算され、新たな空きメモリノードとして新規に登録する。
//  予め、MemoryHeapManager のロックを取得しておく必要がある。
void MemoryHeapManager::SplitFreeMemoryNodeUnsafe(FreeMemoryList::iterator itr, size_t size) NN_NOEXCEPT
{
    // 分割前の空きメモリノードは size より大きくなければならない
    NN_SDK_ASSERT( itr->GetSize() > size, NN_TEXT_OS("nn::os::AllocateMemoryBlock(): MemoryHeapManager 内部関数の事前条件違反です。"));

    // size は MemoryBlockUnitSize 単位でなければならない
    NN_SDK_REQUIRES( IsAligned( size, MemoryBlockUnitSize ), NN_TEXT_OS("nn::os::AllocateMemoryBlock(): size=0x%zx が MemoryBlockUnitSize の整数倍ではありません。"), size);

    // 後半部分の新しい空きメモリノードを作成
    FreeMemoryNode* newFreeNode = new( reinterpret_cast<void*>(itr->GetAddress() + size) ) FreeMemoryNode;
    newFreeNode->SetSize( itr->GetSize() - size );

    // 前半部分のメモリサイズを変更
    itr->SetSize( size );

    // 新しいノードを現在のノードの直後に連結
    m_freeMemoryList.insert( ++itr, *newFreeNode );
}


//-----------------------------------------------------------------------------
//  ヒープに空きメモリを追加し、新たな freeMemoryNode ノードとして登録する。
//  必要に応じて、前後の freeMemoryNode ノードとも結合する。
//  ヒープへのメモリ返却だけでなく、ヒープの拡大時にも本関数が使用される。
//  予め、MemoryHeapManager のロックを取得しておく必要がある。
void MemoryHeapManager::AddToFreeSpaceUnsafe(uintptr_t address, size_t size) NN_NOEXCEPT
{
    // 指定された場所に、新しい空きメモリノードを作成
    FreeMemoryNode* newFreeNode = new( reinterpret_cast<void*>(address) ) FreeMemoryNode;
    newFreeNode->SetSize( size );

    // 作成したノードを空きメモリリストへ繋ぐ。
    // リストはアドレス昇順に並んでいるので、まずは挿入ポイントを探す。
    FreeMemoryList::iterator    itr = m_freeMemoryList.begin();
    for (; itr != m_freeMemoryList.end(); itr++ )
    {
        // リスト中の空きメモリノードのアドレスより若ければ、
        // そこが挿入ポイントなのでループから抜ける。
        // 最後まで break しなかった場合は、リストの末尾に挿入する。
        if (address < itr->GetAddress())
        {
            break;
        }
    }

    // 見つけた node の直前に新しいノードを挿入する。itr は挿入ノードを指す。
    itr = m_freeMemoryList.insert( itr, *newFreeNode );

    // 挿入ノードの１つ前のノードと結合できるなら結合する。
    // 結合に成功した場合、返値の itr には結合したノードが返る。
    itr = ConcatenatePreviousFreeMemoryNodeUnsafe( itr );

    // 挿入ノードの１つ後のノードと結合できるなら結合する
    ConcatenatePreviousFreeMemoryNodeUnsafe( ++itr );
}


//-----------------------------------------------------------------------------
//  指定したメモリ領域が確保済み領域の一部かどうかを検査する。
//  言い換えると、指定したメモリ領域が空きメモリ領域と重複していないかどうか。
//  確保済みメモリであれば true が、空きメモリと重複していれば false が返る。
//  予め、MemoryHeapManager のロックを取得しておく必要がある。
bool MemoryHeapManager::IsRegionAllocatedMemoryUnsafe(uintptr_t address, size_t size) NN_NOEXCEPT
{
    // 空きメモリ領域の個々のノードと、指定メモリ領域が重複していたら false。
    for (FreeMemoryList::iterator itr =  m_freeMemoryList.begin();
                                  itr != m_freeMemoryList.end();
                                  itr++ )
    {
        uintptr_t   nodeAddress = itr->GetAddress();
        size_t      nodeSize    = itr->GetSize();

        if ((nodeAddress < (address + size)) &&
            (address < (nodeAddress + nodeSize)))
        {
            return false;
        }
    }

    return true;
}


//-----------------------------------------------------------------------------
//  ヒープサイズ領域の変更
nn::Result  MemoryHeapManager::SetHeapSize(size_t size) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( IsAligned( size, MemoryHeapUnitSize ), NN_TEXT_OS("nn::os::SetMemoryHeapSize(): size=0x%zx が MemoryHeapUnitSize の整数倍ではありません。"), size);

    {   //  クリティカルセクション
        //  m_heapSize はクリティカルセクション内で参照する
        std::lock_guard<InternalCriticalSection> lockForSuspend( Get(GetCurrentThread()->_csThread) );
        std::lock_guard<InternalCriticalSection> lock( m_criticalSection );

        if (size > m_heapSize)
        {   // 現在のヒープサイズより大きくする場合

            // ヒープサイズを変更する
            uintptr_t address = 0;
            auto result = m_impl.SetHeapSize( &address, size );
            NN_RESULT_DO(result);
            NN_RESULT_THROW_UNLESS(address, os::ResultOutOfMemory());

            if (m_heapSize == 0)
            {
                // 初ヒープ獲得なら、アドレスのアライメントをチェック
                NN_SDK_REQUIRES( IsAligned( address, MemoryHeapUnitSize ), NN_TEXT_OS("nn::os::SetMemoryHeapSize(): 取得したメモリが MemoryHeapUnitSize でアライメントされていません。"));
            }
            else
            {
                // ヒープ拡張なら、アドレスの正当性をチェック
                NN_SDK_ASSERT( address == m_heapAddress, NN_TEXT_OS("nn::os::SetMemoryHeapSize(): メモリヒープのアドレスが異常です。"));
            }

            // 既存のヒープの末尾に更にヒープを拡張する
            uintptr_t beginAddress  = address + m_heapSize;
            size_t    commitSize    = size    - m_heapSize;

            // 獲得したヒープ領域を空きメモリリストに追加
            AddToFreeSpaceUnsafe( beginAddress, commitSize );

            // 拡張後のヒープアドレスとヒープサイズを設定
            m_heapAddress   = address;
            m_heapSize      = size;
        }
        else if (size < m_heapSize)
        {   // 現在のヒープサイズより小さくする場合

            // 開放対象になる部分のアドレスとサイズを計算
            uintptr_t   beginAddress = m_heapAddress + size;
            size_t      decommitSize = m_heapSize    - size;

            // 開放するノードは、空きメモリノードの末尾の１つ前
            FreeMemoryList::iterator itr = m_freeMemoryList.end();
            --itr;
            // 空きメモリノードがない場合は Busy を返す
            if (itr == m_freeMemoryList.end())
            {
                NN_RESULT_THROW( os::ResultBusy() );
            }

            // 開放対象ノードが解放対象領域より大きくない場合も Busy を返す
            if ( !((itr->GetAddress() <= beginAddress) &&
                   (itr->GetSize()    >= decommitSize)) )
            {
                // メモリヒープの末尾が使用中
                NN_RESULT_THROW( os::ResultBusy() );
            }

            // 変更後のヒープサイズ
            size_t  nodeSize = itr->GetSize() - decommitSize;
            if (nodeSize == 0)
            {
                // ノードサイズを 0 にする場合は その空きメモリノードを削除
                m_freeMemoryList.erase( itr );
                itr->Clean();
            }
            else
            {
                // サイズ縮小の場合は、そのノードのサイズを再設定
                itr->SetSize( nodeSize );
            }

            // ヒープサイズを変更
            uintptr_t address = 0;
            auto result = m_impl.SetHeapSize( &address, size );
            NN_ABORT_UNLESS(result.IsSuccess(), NN_TEXT_OS("nn::os::SetMemoryHeapSize(): メモリヒープの縮小に失敗しました。"));

            // 縮小後のヒープサイズに更新
            m_heapSize = size;
            if (size == 0)
            {
                m_heapAddress = 0;
            }
        }
    }

    NN_RESULT_SUCCESS;
}


//-----------------------------------------------------------------------------
//  メモリヒープからメモリを獲得（size は MemoryBlockUnitSize 単位）
nn::Result  MemoryHeapManager::AllocateFromHeap(uintptr_t* address, size_t size) NN_NOEXCEPT
{
    {   // クリティカルセクション
        std::lock_guard<InternalCriticalSection> lockForSuspend( Get(GetCurrentThread()->_csThread) );
        std::lock_guard<InternalCriticalSection> lock( m_criticalSection );

        // 指定した size 以上の空きメモリを探す
        FreeMemoryList::iterator itr = FindFreeSpaceUnsafe( size );
        if (itr == m_freeMemoryList.end())
        {
            // 空きメモリ領域がない
            NN_RESULT_THROW( os::ResultOutOfMemory() );
        }

        // 探し出した空きメモリのサイズが、要求 size より大きければ２つに分割
        if (itr->GetSize() > size)
        {
            // 空きメモリ領域を２つに分割し、後半を空きメモリとして登録
            SplitFreeMemoryNodeUnsafe( itr, size );
        }

        // itr が指すノードを、空きメモリリストから削除
        m_freeMemoryList.erase( itr );
        itr->Clean();

        // ヒープ使用量を加算
        m_usedHeapSize += size;

        // 返値として、獲得したメモリ領域の先頭アドレスを代入
        *address   = itr->GetAddress();
    }

    NN_RESULT_SUCCESS;
}


//-----------------------------------------------------------------------------
//  メモリヒープへメモリを返却
void MemoryHeapManager::ReleaseToHeap(uintptr_t address, size_t size) NN_NOEXCEPT
{
    {   // クリティカルセクション
        std::lock_guard<InternalCriticalSection> lockForSuspend( Get(GetCurrentThread()->_csThread) );
        std::lock_guard<InternalCriticalSection> lock( m_criticalSection );

        // freeMemoryList を辿り、対象メモリ領域が確保済み領域であることを確認
        NN_ABORT_UNLESS( IsRegionAllocatedMemoryUnsafe(address, size), NN_TEXT_OS("nn::os::FreeMemoryBlock(): 指定されたメモリ領域に返却済みの領域が含まれています。(address=0x%p, size=0x%zx)"), address, size);

        // 返却対象のメモリ領域を ReadWrite, Cacheable 属性に変更
        SetMemoryPermission( address, size, MemoryPermission_ReadWrite );
        SetMemoryAttribute( address, size, MemoryAttribute_Normal );

        // 返却されたメモリ領域を freeMemoryList に追加。
        // 必要に応じて、前後の freeNode と結合。
        AddToFreeSpaceUnsafe( address, size );

        // ヒープ使用量から減算
        m_usedHeapSize -= size;
    }
}


}   // namespace detail
}}  // namespace nn::os


